﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/util/util_StringView.h>

namespace nn { namespace utilTool {

class XmlNode;

//! @brief XML パーサです。
//!
//! 内部でオープンソースのライブラリを使用します。使用する際はライセンスの確認が必要です。
//! デフォルトヒープからアロケーションを行います。
//!
class XmlParser
{
    NN_DISALLOW_COPY( XmlParser );
public:

    //! @brief コンストラクタです。
    XmlParser();

    //! @brief デストラクタです。
    ~XmlParser();

    //! @brief      XML ドキュメントを解析します。
    //! @param[in]  pDocument   解析する XML ドキュメントです。
    //! @pre        XML は NUL 終端文字列である必要があります。
    //! @details    渡された文字列は解析中に書き換えられます。
    bool Parse( char* pDocument );

    //! @brief 解析結果をクリアします。
    void Clear();

    //! @brief  解析結果のルート要素を取得します。
    //! @return 結果を返します。
    const XmlNode* GetRoot() const;

    //! @brief  エラー文字列を取得します。
    //! @return 結果を返します。
    nn::util::string_view GetErrorString() const;

    //! @brief  エラー発生箇所の文字列を取得します。
    //! @return 結果を返します。
    nn::util::string_view GetErrorPosition() const;

private:
    class Impl;
    Impl* m_pImpl;
};

//! @brief XML ノード
class XmlNode
{
    NN_DISALLOW_COPY( XmlNode );
public:

    //! @brief  要素名を取得します。
    //! @return 結果を返します。
    nn::util::string_view GetName() const;

    //! @brief  親要素を取得します。
    //! @return 結果を返します。
    const XmlNode* GetParent() const;

    //! @brief  最初の子要素を取得します。
    //! @return 結果を返します。
    const XmlNode* GetChild() const;

    //! @brief  次の要素を取得します。
    //! @return 結果を返します。
    const XmlNode* GetNext() const;

    //! @brief  前の要素を取得します。
    //! @return 結果を返します。
    const XmlNode* GetPrev() const;

    //! @brief      指定した名前の子要素を探索します。
    //! @param[in]  name    指定する名前です。
    //! @return     結果を返します。
    const XmlNode* FindChild( const nn::util::string_view& name ) const;

    //! @brief  自身と同じ名前の次の要素を探索します。
    //! @return 結果を返します。
    const XmlNode* FindNext() const;

    //! @brief  自身と同じ名前の前の要素を探索します。
    //! @return 結果を返します。
    const XmlNode* FindPrev() const;

    //! @brief      指定した名前の属性を探索します。
    //! @param[in]  name    指定する名前です。
    //! @return     結果を返します。
    nn::util::string_view FindAttribute( const nn::util::string_view& name ) const;

    //! @brief テキストを取得します。
    //! @return 結果を返します。
    nn::util::string_view GetText() const;
};

}} // namespace nn::utilTool
