﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <utility>
#include <functional> // for std::reference_wrapper
#include <type_traits>

namespace nn { namespace util {

/**
    @brief 関数呼び出しを抽象化するインターフェイスです。

    @tparam T 関数型を指定します。

    @details
     特殊化 IFunction<R(Args...)> を参照してください。
*/
template <typename T>
class IFunction; /* undefined */

/**
    @brief 関数型をとる IFunction の特殊化です。

    @tparam R 関数の返り値型を指定します。
    @tparam Args 関数の引数型を指定します。

    @details
     Args を引数にとり R を返すような関数を抽象化するためのクラステンプレートです。
     本クラステンプレートのインスタンスクラスは抽象クラスでありインスタンス化できません。
*/
template <typename R, typename... Args>
class IFunction<R(Args...)>
{
public:

    /**
        @brief 関数呼び出し演算子です。

        @param args 引数を指定します。
        @return 演算結果が返ります。
    */
    virtual R operator()(Args... args) const = 0;

    /**
        @brief 指定された関数様のオブジェクトを持つ IFunction<R(Args...)> を基底とする型のオブジェクトを取得します。

        @tparam F 引数となる関数様の型を指定します。

        @param f 関数様のオブジェクトを指定します。
        @return 指定された関数様のオブジェクトを持つ IFunction<R(Args...)> を基底とする型のオブジェクトを返します。

        @details
         多くの場合、本関数の代わりに、R と Args を自動推論する MakeIFunction() 関数テンプレートを使用します。

         F には以下のものを指定可能です。

         - 関数型
         - 関数オブジェクト型
         - std::reference_wrapper<関数オブジェクト型>

         本関数は、以下のようなオブジェクトを返します。

         - IFunction<R(Args...)> を基底クラスとする final クラスのインスタンス
         - 内部に f をコピーした F 型のメンバフィールドを持つ
         - コピーした f への operator() 呼び出しを IFunction<R(Args...)>::operator()(Args...) の実装とする

         このため、const IFunction<R(Args...)>& などを引数としてとる箇所に、本関数にラムダ式を渡した結果をそのまま渡すことなどができます。

         以下のような場合には、必要に応じて std::ref() を使って std::reference_wrapper に包んで渡すなどしてください。

         - f のコピーコストを気にする必要がある
         - f を参照で渡したい
         - F::operator()(Args...) が const メンバ関数でありコンパイルが通らない
    */
    template <typename F>
    static auto Make(F&& f);

protected:

    /**
        @brief default 実装の protected デストラクタです。
    */
    virtual ~IFunction() = default;

};

/**
    @brief 指定された関数様のオブジェクトに対応する IFunction<> を基底とする型のオブジェクトを取得します。

    @tparam F 引数となる関数様の型を指定します。

    @param f 関数様のオブジェクトを指定します。
    @return 指定された関数様のオブジェクトに対応する IFunction<> を基底とする型のオブジェクトを返します。

    @details
     F の型から呼び出し可能な関数型 R(Args...) を推論し、IFunction<R(Args...)>::Make(f) を返します。
*/
template <typename F>
auto MakeIFunction(F&& f);


/////////////////
// implementation

namespace detail {

template <typename>
struct GetIFunctionTypeForObject;

template <typename F, typename R, typename... Args>
struct GetIFunctionTypeForObject<R (F::*)(Args...)>
{
    using type = R(Args...);
};

template <typename F, typename R, typename... Args>
struct GetIFunctionTypeForObject<R (F::*)(Args...) const>
{
    using type = R(Args...);
};

template <typename>
struct GetIFunctionType;

template <typename R, typename... Args>
struct GetIFunctionType<R(Args...)>
{
    using type = R(Args...);
};

template <typename R, typename T, typename... Args>
struct GetIFunctionType<R (T::*)(Args...)>
{
    using type = R(Args...);
};

template <typename F>
struct GetIFunctionType<std::reference_wrapper<F>> : GetIFunctionType<F> {};

template <typename F>
struct GetIFunctionType : GetIFunctionTypeForObject<decltype(&F::operator())> {};

template <typename T, typename F>
class Function;

template <typename R, typename... Args, typename F>
class Function<R(Args...), F> final
    : public IFunction<R(Args...)>
{
public:
    explicit Function(F f) : m_F(std::forward<F>(f)) {}
    virtual R operator()(Args... args) const override final
    {
        return m_F(std::forward<Args>(args)...);
    }
private:
    F m_F;
};

} // detail

template <typename R, typename... Args>
template <typename F>
inline auto IFunction<R(Args...)>::Make(F&& f)
{
    return detail::Function<R(Args...), typename std::decay<F>::type>{std::forward<F>(f)};
}

template <typename F>
inline auto MakeIFunction(F&& f)
{
    using FunctionType = typename detail::GetIFunctionType<typename std::decay<F>::type>::type;
    return IFunction<FunctionType>::Make(std::forward<F>(f));
}

}}
