﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief API for querying audio information.
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/edid/edid_SpeakerAllocation.h>

namespace nn { namespace edid {

    struct AudioFormatInfo;
    struct Edid;

    /**
     * @brief Callback function that will execute when an audio format is found.
     *
     * @param[in] pFormat   Description of a supported audio format.
     * @param[in] pUserData User defined data passed from VisitAudioFormats().
     *
     * @return Whether to continue searching for additional audio formats.
     *
     * @details @ref AudioFormat_Pcm may appear multiple times, each with a different bit depth.
     */
    typedef bool (*AudioFormatVisitor)(const AudioFormatInfo* pFormat, void* pUserData);

    /**
     * @brief Visitor function that will execute a callback for every audio format found.
     *
     * @param[in] pEdid     EDID created from OpenEdid().
     * @param[in] visitor   Callback function to execute.
     * @param[in] pUserData User defined data to pass to the callback function.
     *
     * @pre
     *       - @a pEdid != nullptr
     *       - @a visitor != nullptr
     * @post The function @a visitor is executed for each audio format found.
     */
    void VisitAudioFormats(const Edid* pEdid, AudioFormatVisitor visitor, void* pUserData) NN_NOEXCEPT;

    /**
     * @brief Retrieves the max bit rate supported (in kHz).
     *
     * @param[out] pOutBitRate Max bit rate in kHz.
     * @param[in]  pFormat     Audio format.
     *
     * @return Whether @a pOutBitRate contains the max bit rate.
     *
     * @pre
     *       - @a pOutBitRate != nullptr
     *       - @a pFormat != nullptr
     * @post The bit rate may be written to @a pOutBitRate.
     *
     * @details Only these formats have information about the maximum bit rate:
     *          - @ref AudioFormat_Ac3
     *          - @ref AudioFormat_Mpeg1
     *          - @ref AudioFormat_Mp3
     *          - @ref AudioFormat_Mpeg2
     *          - @ref AudioFormat_AacLc
     *          - @ref AudioFormat_Dts
     *          - @ref AudioFormat_Atrac
     */
    bool GetAudioFormatBitRateMax(int* pOutBitRate, const AudioFormatInfo* pFormat) NN_NOEXCEPT;

    /**
     * @brief Retrieves the bit depth (in bits).
     *
     * @param[out] pOutBitDepth Bit depth of the format in bits.
     * @param[in]  pFormat      Audio format.
     *
     * @return Whether @a pOutBitDepth contains the bit depth.
     *
     * @pre
     *       - @a pOutBitDepth != nullptr
     *       - @a pFormat != nullptr
     * @post The bit depth may be written to @a pOutBitDepth.
     *
     * @details Only these formats have information about the bit depth:
     *          - @ref AudioFormat_Pcm
     */
    bool GetAudioFormatBitDepth(int* pOutBitDepth, const AudioFormatInfo* pFormat) NN_NOEXCEPT;

    /**
     * @brief Retrieves information about the speaker setup.
     *
     * @param[in] pEdid EDID created from OpenEdid().
     *
     * @return Speaker setup information.
     *
     * @pre
     *       - @a pEdid != nullptr
     * @post The speaker setup information is returned.
     *
     * @details This function should not be used to check if the sink supports
     *          audio.  This information is optional in the EDID, even if the sink
     *          supports basic audio.  Please use IsAudioSupported() to query basic
     *          audio support instead.
     */
    SpeakerAllocationSet GetSpeakerAllocation(const Edid* pEdid) NN_NOEXCEPT;

    /**
     * @brief Retrieves whether audio is supported on the sink device.
     *
     * @param[in] pEdid EDID created from OpenEdid().
     *
     * @return Whether the EDID indicates that audio is supported.
     *
     * @pre
     *       - @a pEdid != nullptr
     * @post Returns if audio support is indicated in the EDID.
     */
    bool IsAudioSupported(const Edid* pEdid) NN_NOEXCEPT;

}}
