﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>

namespace nn { namespace crypto { namespace detail {

template <typename EllipticCurve, typename HashFunction>
class EcdsaImpl
{
public:
    static const size_t SignatureSize = EllipticCurve::PublicKeySize;
    static const size_t PrivateKeySize = EllipticCurve::PrivateKeySize;
    static const size_t RandomSize = EllipticCurve::PrivateKeySize;

    void Sign(void* outSignature, size_t signatureSize,
              const void* privateKeyOctets, size_t privateKeySize,
              const void* rawMessage, size_t rawMessageSize,
              const void* random, size_t randomSize) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(signatureSize == SignatureSize);
        NN_SDK_ASSERT(randomSize >= RandomSize);

        EllipticCurve key;
        key.Initialize( nullptr, 0, privateKeyOctets, privateKeySize );

        HashFunction hashGen;
        char digest[hashGen.HashSize];
        hashGen.Initialize();
        hashGen.Update(rawMessage, rawMessageSize);
        hashGen.GetHash( digest, sizeof(digest) );

        key.SignHash( outSignature, signatureSize,
                      digest, sizeof(digest),
                      random, randomSize );
    }

    bool Verify(const void* pSignature, size_t signatureSize,
                const void* publicKeyOctets, size_t publicKeySize,
                const void* rawMessage, size_t rawMessageSize ) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(signatureSize == SignatureSize);

        EllipticCurve key;
        key.Initialize( publicKeyOctets, publicKeySize, nullptr, 0 );

        HashFunction hashGen;
        char digest[hashGen.HashSize];
        hashGen.Initialize();
        hashGen.Update(rawMessage, rawMessageSize);
        hashGen.GetHash( digest, sizeof(digest) );

        return key.VerifyHash( pSignature, signatureSize, digest, sizeof(digest) );
    }
};

}}} // namespace nn::crypto::detail
