﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
    @file
    @brief RSA-PKCS1-SHA-256 のメッセージをエンコードする API の宣言
 */

#include <cstring>
#include <nn/nn_SdkAssert.h>
#include <nn/crypto/crypto_Sha256Generator.h>
#include <nn/crypto/detail/crypto_RsaPkcs1Padding.h>

namespace nn { namespace crypto {

/**
    @brief      平文を鍵サイズが 2048 bit の RSA-PKCS1-SHA-256 メッセージにエンコードする関数です。

    @param[out] pOutMessage            エンコードされたデータが格納されるバッファへのポインタ。
    @param[in]  outMessageSize         pOutMessage が指すバッファのバイトサイズ。
    @param[in]  pRawMessage            エンコード対象のデータへのポインタ。
    @param[in]  rawMessageSize         エンコード対象のデータのバイトサイズ。

    @pre
    - outMessageSize == 256

    @post
    - pOutMessage にエンコードされたメッセージが格納される。

    @details
    与えられたデータ列にパディングを付与し、RSA-PKCS1-SHA-256 メッセージにエンコードします。

    この関数ではメッセージに署名は行いません。署名が必要な場合は SignRsa2048Pkcs1Sha256 を利用してください。
 */
inline void EncodeRsa2048Pkcs1Sha256(void* pOutMessage, size_t outMessageSize,
                                     const void* pRawMessage, size_t rawMessageSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(outMessageSize == 256);

    detail::RsaPkcs1Padding<Sha256Generator> rsaPkcs1;
    Sha256Generator                          sha256;

    // メッセージをハッシュ計算
    sha256.Initialize();
    sha256.Update(pRawMessage, rawMessageSize);

    // エンコード
    rsaPkcs1.BuildPad(pOutMessage, outMessageSize, &sha256);
}

}} // namespace nn::crypto
