﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;
using System.Threading.Tasks;

namespace ZarfCreator.Parser
{
    public class WildcardPattern
    {
        /// <summary>
        /// ワイルドカードを正規表現文字列に置換するためのRegexオブジェクト。
        /// </summary>
        private static readonly Regex regexWildcard = new Regex(@"((?<=(^|/))\\\*\\\*(/|$)?)|\\\*|\\\?");

        private Regex regex;

        public WildcardPattern(IEnumerable<string> strs)
        {
            this.regex =
                new Regex(
                    @"^("
                    + string.Join(
                        "|",
                        strs.Select(
                            file => ConvertWildcardToRegex(file)))
                    + @")$");
        }

        public bool IsMatch(string path)
        {
            return this.regex.IsMatch(path.ToLowerInvariant().Replace('\\', '/'));
        }

        /// <summary>
        /// ワイルドカードを正規表現に変換します。
        ///
        /// ? は任意の1文字に一致します。
        /// * は任意の長さの文字列に一致します。
        /// ** は、ディレクトリをまたいで任意の文字列に一致します。
        /// </summary>
        /// <param name="str">ワイルドカードを含んだ文字列。</param>
        /// <returns>正規表現に変換された文字列。</returns>
        private static string ConvertWildcardToRegex(string str)
        {
            var wk1 =
                str
                .Replace('\\', '/')
                .ToLowerInvariant();

            // '/' で終わっている場合は、'**'を追加する。
            var wk2 = wk1.EndsWith("/") ? wk1 + "**" : wk1;

            // 念のため、行頭が'/'で始まっている場合は取り除く。
            var wk3 = wk2.StartsWith("/") ? wk2.Substring(1) : wk2;

            var wk4 = Regex.Escape(wk3);    // エスケープしておく。

            return
                regexWildcard.Replace(
                    wk4,
                    match =>
                    {
                        switch (match.ToString())
                        {
                        case @"\*\*/": return @"([^/]+/)*";
                        case @"\*\*": return @"([^/]+/)*[^/]*";
                        case @"\*": return @"[^/]*";
                        default:
                        case @"\?": return @"[^/]";
                        }
                    }
                );
        }
    }
}
