﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace YamlConverter
{
    using System;
    using System.IO;
    using System.Reflection;
    using System.Text;
    using System.Xml;
    using YamlDotNet.Core;
    using YamlDotNet.RepresentationModel;

    /// <summary>
    /// エントリーポイントを定義します。
    /// </summary>
    internal class Program
    {
        private const string OptionXmlToYaml = "--xml-to-yaml";

        private const string OptionYamlToXml = "--yaml-to-xml";

        private const string YamlName = "yaml";

        private const string YamlNamespaceUri = "http://yaml.org/xml";

        private const string YamlTypeName = "type";

        private const string YamlSeqTypeValue = "seq";

        private const string YamlMapTypeValue = "map";

        private const string YamlSeqNodeName = "_";

        private enum Mode : int
        {
            Help,
            XmlToYaml,
            YamlToXml
        }

        /// <summary>
        /// エントリーポイントです。
        /// </summary>
        /// <param name="args">コマンドラインオプションです。</param>
        internal static void Main(string[] args)
        {
            try
            {
                Mode mode = GetMode(args);

                if (mode == Mode.Help)
                {
                    Console.Error.WriteLine(GetHelpMessage());

                    Environment.Exit(1);
                }
                else
                {
                    switch (mode)
                    {
                        case Mode.XmlToYaml:
                            ConvertXmlToYaml();
                            break;

                        case Mode.YamlToXml:
                            ConvertYamlToXml();
                            break;
                    }
                }
            }
            catch (Exception ex)
            {
                Console.Error.WriteLine("Error: {0}", ex.Message);

                Environment.Exit(1);
            }
        }

        private static Mode GetMode(string[] args)
        {
            if (args.Length == 0)
            {
                return Mode.Help;
            }

            switch (args[0])
            {
                case OptionXmlToYaml:
                    return Mode.XmlToYaml;

                case OptionYamlToXml:
                    return Mode.YamlToXml;

                default:
                    return Mode.Help;
            }
        }

        private static string GetHelpMessage()
        {
            var name = Path.GetFileName(Assembly.GetExecutingAssembly().Location);

            return new StringBuilder()
                .AppendLine("Usage:")
                .AppendLine(string.Format("  {0} {1,-20} Convert XML data from STDIN to YAML", name, OptionXmlToYaml))
                .AppendLine(string.Format("  {0} {1,-20} Convert YAML data from STDIN to XML", name, OptionYamlToXml))
                .ToString()
                .TrimEnd();
        }

        private static void ConvertXmlToYaml()
        {
            var xd = new XmlDocument();

            xd.Load(Console.In);

            new YamlStream(new YamlDocument(TransformXmlIntoYaml(xd.DocumentElement))).Save(Console.Out);
        }

        private static bool IsYamlNodeType(XmlNode xmlNode, string value)
        {
            if (xmlNode.Attributes != null)
            {
                foreach (XmlAttribute attr in xmlNode.Attributes)
                {
                    if (attr.Prefix == YamlName && attr.LocalName == YamlTypeName && attr.Value == value)
                    {
                        return true;
                    }
                }
            }

            return false;
        }

        private static bool HasElementNodes(XmlNode xmlNode)
        {
            if (!xmlNode.HasChildNodes)
            {
                return false;
            }

            foreach (XmlNode node in xmlNode.ChildNodes)
            {
                if (node.NodeType != XmlNodeType.Element)
                {
                    return false;
                }
            }

            return true;
        }

        private static YamlNode TransformXmlIntoYaml(XmlNode xmlNode)
        {
            if (IsYamlNodeType(xmlNode, YamlSeqTypeValue))
            {
                var yamlNode = new YamlSequenceNode();

                foreach (XmlNode node in xmlNode.ChildNodes)
                {
                    yamlNode.Add(TransformXmlIntoYaml(node));
                }

                return yamlNode;
            }
            else if (IsYamlNodeType(xmlNode, YamlMapTypeValue) || HasElementNodes(xmlNode))
            {
                var yamlNode = new YamlMappingNode();

                foreach (XmlNode node in xmlNode.ChildNodes)
                {
                    yamlNode.Add(node.Name, TransformXmlIntoYaml(node));
                }

                return yamlNode;
            }
            else
            {
                return new YamlScalarNode(xmlNode.InnerText);
            }
        }

        private static void ConvertYamlToXml()
        {
            var ys = new YamlStream();

            ys.Load(Console.In);

            foreach (var doc in ys.Documents)
            {
                var xml = new XmlDocument();

                xml.AppendChild(xml.CreateElement(YamlName));

                AppendYamlToXml(xml, xml.DocumentElement, doc.RootNode);

                xml.Save(Console.Out);
            }
        }

        private static void AppendYamlToXml(XmlDocument xml, XmlNode xmlNode, YamlNode yamlNode)
        {
            if (yamlNode is YamlMappingNode)
            {
                var attr = xml.CreateAttribute(YamlName, YamlTypeName, YamlNamespaceUri);

                attr.Value = YamlMapTypeValue;

                xmlNode.Attributes.Append(attr);

                foreach (var entry in ((YamlMappingNode)yamlNode).Children)
                {
                    var newXmlNode = xml.CreateElement(((YamlScalarNode)entry.Key).Value);

                    xmlNode.AppendChild(newXmlNode);

                    AppendYamlToXml(xml, newXmlNode, entry.Value);
                }
            }
            else if (yamlNode is YamlSequenceNode)
            {
                var attr = xml.CreateAttribute(YamlName, YamlTypeName, YamlNamespaceUri);

                attr.Value = YamlSeqTypeValue;

                xmlNode.Attributes.Append(attr);

                foreach (var nextYamlNode in (YamlSequenceNode)yamlNode)
                {
                    var newXmlNode = xml.CreateElement(YamlSeqNodeName);

                    xmlNode.AppendChild(newXmlNode);

                    AppendYamlToXml(xml, newXmlNode, nextYamlNode);
                }
            }
            else
            {
                xmlNode.InnerText = ((YamlScalarNode)yamlNode).Value;
            }
        }
    }
}
