﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using Microsoft.Build.Construction;
using Microsoft.Build.Exceptions;

namespace VsSolutionBuilderNinjaExecutor
{
    public class SolutionFile
    {
        /// <summary>
        /// ソリューションファイルのパス
        /// </summary>
        public string FilePath { get; set; }
        /// <summary>
        /// ソリューションファイルに設定済みのビルドの種類とプラットフォームの種類の組み合わせ一覧
        /// </summary>
        public IReadOnlyList<BuildSettingInfo> Configuration { get; set; }
        /// <summary>
        /// Generic/NX　ソリューションファイル名から判断する。
        /// </summary>
        /// <note>
        /// 具体例）
        /// Generic ：Demo1-spec.Generic.autogen.vs2017.sln
        /// NX　　　：Demo1-spec.NX.autogen.vs2017.sln
        /// </note>
        public string Spec { get; set; }
        /// <summary>
        /// ソリューションファイルに設定済みのプロジェクトファイル一覧
        /// </summary>
        private IReadOnlyList<ProjectInSolution> Projects { get; set; }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="file">ソリューションファイルのパス</param>
        /// <param name="excludeDependencies">依存関係から除外するプロジェクトファイル名の一覧
        /// </param>
        public SolutionFile(string file, IReadOnlyList<string> excludeDependencies)
        {
            if (!File.Exists(file))
            {
                throw new FileNotFoundException($"'{file}' not found");
            }

            this.FilePath = file;
            if (this.FilePath.Contains(SpecDefinition.GENERIC) == true)
            {
                this.Spec = SpecDefinition.GENERIC;
            }
            else if (this.FilePath.Contains(SpecDefinition.NX) == true)
            {
                this.Spec = SpecDefinition.NX;
            }
            else
            {
                this.Spec = string.Empty;
            }

            try
            {
                var solution = Microsoft.Build.Construction.SolutionFile.Parse(file);
                this.Projects = solution.ProjectsInOrder;

                var setting = from config in solution.SolutionConfigurations
                              select new BuildSettingInfo()
                              {
                                  Configuration = config.ConfigurationName,
                                  Platform = config.PlatformName,
                                  ProjectFiles = this.GetProjectFiles(config.ConfigurationName,
                                                    config.PlatformName, solution.ProjectsByGuid, excludeDependencies),
                                  IsErrorOccurred = false
                              };

                this.Configuration = setting.ToList();
                foreach (var configurationItems in this.Configuration)
                {
                    foreach (var projectFile in configurationItems.ProjectFiles)
                    {
                        if (projectFile.IsErrorOccurred == true)
                        {
                            configurationItems.IsErrorOccurred = true;
                        }
                    }
                }
            }
            catch (InvalidProjectFileException)
            {
                throw;
            }
        }
        /// <summary>
        /// プロジェクトファイルの一覧を取得する。
        /// </summary>
        /// <note>
        /// ＜機能１＞
        /// slnファイルには以下に所属するプロジェクトファイル(vcxproj)があらかじめ設定されている。
        /// そのプロジェクトファイルのプルパスを取得する。
        /// Project("{(pid)}") = "Demo1-spec.Generic.autogen", "Demo1-spec.Generic.autogen.vcxproj", "{(pid)}"
        /// Project("{(pid)}") = "libnns_audio-spec.Generic.autogen",
        ///  "..\..\Libraries\audio\libnns_audio\libnns_audio-spec.Generic.autogen.vcxproj", "{(pid)}"
        /// Project("{(pid)}") = "libnns_gfx-spec.Generic.autogen",
        ///  "..\..\Libraries\gfx\libnns_gfx\libnns_gfx-spec.Generic.autogen.vcxproj", "{(pid)}"
        /// Project("{(pid)}") = "libnns_hid-spec.Generic.autogen",
        ///  "..\..\Libraries\hid\libnns_hid\libnns_hid-spec.Generic.autogen.vcxproj", "{(pid)}"
        /// ＜機能２＞
        /// slnファイルには以下に所属するプロジェクトファイル(vcxproj)
        /// 　に対するビルド時のconfigurationとplatformの組み合わせがあらかじめ設定されている。
        /// そのプロジェクトファイルに指定するときのconfigurationとplatformの組み合わせを取得する。
        /// {(pid)}.Debug|Win32.Build.0 = VS2015_Debug|Win32　・・・①
        /// {(pid)}.Debug|x64.Build.0 = VS2015_Debug|x64
        /// {(pid)}.Develop|Win32.Build.0 = VS2015_Develop|Win32
        /// {(pid)}.Develop|x64.Build.0 = VS2015_Develop|x64
        /// ①・・・slnファイルをDebug／Win32でビルドする場合は、
        /// 　プロジェクトファイルに対して、VS2015_Debug／Win32を指定する必要があり、
        /// 　この対となる情報をProjectFileインスタンスに保管しておく。
        /// </note>
        /// <param name="configuration">ビルドの種類</param>
        /// <param name="platform">プラットフォームの種類</param>
        /// <param name="projectsByGuid">slnファイルに定義されているGUIDとプロジェクト情報の一覧
        /// </param>
        /// <param name="excludeDependencies">依存関係から除外するプロジェクトファイル名の一覧
        /// </param>
        /// <returns>プロジェクトファイルの一覧</returns>
        private List<ProjectFile> GetProjectFiles(string configuration, string platform,
            IReadOnlyDictionary<string, ProjectInSolution> ProjectsByGuid,
            IReadOnlyList<string> excludeDependencies)
        {
            var projectFiles = new List<ProjectFile>();
            foreach (var projectItem in this.Projects)
            {
                // slnファイル内に記載されたプロジェクトファイルのフルパスを取得する。
                var baseUri = new Uri(this.FilePath);
                var projectFilePath = new Uri(baseUri, projectItem.RelativePath).LocalPath;

                var ext = Path.GetExtension(projectFilePath);
                if (ext != ProjectFile.VcProjectFileExtention)
                {
                    continue;
                }

                // slnファイル内に記載された
                // プロジェクト指定用のConfigurationとPlatformの組み合わせを取得する。
                var projectConfigurationKey = configuration + "|" + platform;
                var projectConfigurationInSolution =
                    projectItem.ProjectConfigurations[projectConfigurationKey];

                var projectFile = new ProjectFile(this.FilePath,
                    projectFilePath,
                    projectConfigurationInSolution.ConfigurationName,
                    projectConfigurationInSolution.PlatformName,
                    projectItem.Dependencies,
                    excludeDependencies,
                    ProjectsByGuid,
                    ProjectFile.DependenciesType.NEED_DEPENDENCIES);

                projectFiles.Add(projectFile);
            }

            return projectFiles;
        }
    }
}
