﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Xml;
using Microsoft.Build.Construction;
using Microsoft.Build.Evaluation;
using Microsoft.Build.Exceptions;
using Microsoft.Build.Execution;

namespace VsSolutionBuilderNinjaExecutor
{
    public class ProjectFile
    {
        /// <summary>
        /// vcxprojファイルの拡張子
        /// </summary>
        public const string VcProjectFileExtention = ".vcxproj";
        /// <summary>
        ///　プロジェクトファイルに依存設定されたプロジェクトファイルを解析するかどうか
        /// </summary>
        public enum DependenciesType
        { NEED_DEPENDENCIES, UNNEED_DEPENDENCIES }
        /// <summary>
        /// プロジェクトファイルのパス
        /// </summary>
        public string ProjectFileName { get; set; }
        /// <summary>
        /// ビルドの種別
        /// ・VS2015_Debug
        /// ・VS2015_Develop
        /// ・VS2015_Release
        /// ・VS2017_Debug
        /// ・VS2017_Develop
        /// ・VS2017_Release
        /// ・NX_Debug
        /// ・NX_Develop
        /// ・NX_Release
        /// ・NX_VS2015_Debug
        /// ・NX_VS2015_Develop
        /// ・NX_VS2015_Release
        /// ・NX_VS2017_Debug
        /// ・NX_VS2017_Develop
        /// ・NX_VS2017_Release
        /// </summary>
        public string Configuration { get; set; }
        /// <summary>
        /// プラットフォームの種別
        /// ・Win32
        /// ・x64
        /// ・NX32
        /// ・NX64
        /// </summary>
        public string Platform { get; set; }
        /// <summary>
        /// 依存するcsprojファイル
        /// </summary>
        public List<string> DependenciesCsproj { get; set; }
        /// <summary>
        /// vcxprojファイルロード時のエラー発生フラグ
        /// </summary>
        public bool IsErrorOccurred { get; set; }
        /// <summary>
        /// vcxprojファイルロード時のエラーメッセージ
        /// </summary>
        public string ErrorOccurredMessage { get; set; }

        private const string PropertiesPlatform = "Platform";
        private const string PropertiesConfiguration = "Configuration";
        /// <summary>
        /// MSBuild実行後のファイル名を合成する際に使用する定数群
        /// </summary>
        private const string IntermediatesExtentionObj = ".obj";
        private const string IntermediatesExtentionO = ".o";
        /// <summary>
        /// vcxprojファイルを解析する際に使用する定数群
        /// </summary>
        private const string ProjectFileAttributeCondition = "Condition";
        private const string ProjectFileAttributeInclude = "Include";
        private const string ProjectFileElementExcludedFromBuild = "ExcludedFromBuild";
        private const string ProjectFileElementItemDefinitionGroup = "ItemDefinitionGroup";
        private const string ProjectFileElementItemGroup = "ItemGroup";
        private const string ProjectFileValueTrue = "true";
        private const string ProjectFileItemDefineLink = "Link";
        private const string ProjectFileItemDefineOutputFile = "OutputFile";
        private const string ProjectFileItemDefineClCompile = "ClCompile";
        private const string ProjectFileItemDefineObjectFileName = "ObjectFileName";
        private const string ProjectFileItemDefineObjectFileNamePattern = "%(FileName).o";
        private const string ProjectFileItemDefinePreBuildEvent = "PreBuildEvent";

        /// <summary>
        /// MSBuildのツールバージョン
        /// </summary>
        private const string MsbuildDictContains140 = "VS140COMNTOOLS";

        /// <summary>
        /// vcxprojファイルの拡張子
        /// </summary>
        private const string CsProjectFileExtention = ".csproj";

        /// <summary>
        /// MSBuildが作成するOutputのタイプ
        /// ・windows系
        /// ・NX系
        /// </summary>
        private enum MsbuildOutputType
        { TYPE_WINDOWS, TYPE_NX }

        /// <summary>
        /// プロジェクトに設定済みのソースファイルリスト
        /// </summary>
        private List<string> SourceFileList { get; set; }
        /// <summary>
        /// 該当プロジェクトファイルが依存するプロジェクトファイル情報一覧(vcxproj)
        /// </summary>
        private List<ProjectFile> DependenciesProjectFiles { get; set; }

        /// <summary>
        /// プロジェクトを解析したインスタンス
        /// </summary>
        public ProjectInstance ProjectInstanceImpl { get; set; }

        /// <summary>
        /// spec、platformに応じたMSBuild成果物の種類を定義したテーブル
        /// </summary>
        /// <note>
        /// windows系の場合：
        /// 　成果物フォルダ　　　：Win32-v140
        /// 　中間ファイル　　　　：.obj
        /// 　ライブラリファイル　：.lib
        /// 　実行ファイル　　　　：.exe
        /// NX系の場合：
        /// 　成果物フォルダ　　　：NX32
        /// 　中間ファイル　　　　：.o
        /// 　ライブラリファイル　：.a
        /// 　実行ファイル　　　　：.nca、.nsp
        /// </note>
        private Dictionary<string, Dictionary<string, MsbuildOutputType>> OutputTypeDictionary
            = new Dictionary<string, Dictionary<string, MsbuildOutputType>>
            {
                { SpecDefinition.GENERIC, new Dictionary<string, MsbuildOutputType> {
                    { PlatformDefinition.WIN32, MsbuildOutputType.TYPE_WINDOWS },
                    { PlatformDefinition.X64, MsbuildOutputType.TYPE_WINDOWS } } },
                { SpecDefinition.NX, new Dictionary<string, MsbuildOutputType> {
                    { PlatformDefinition.WIN32, MsbuildOutputType.TYPE_WINDOWS },
                    { PlatformDefinition.X64, MsbuildOutputType.TYPE_WINDOWS },
                    { PlatformDefinition.NX32, MsbuildOutputType.TYPE_NX },
                    { PlatformDefinition.NX64, MsbuildOutputType.TYPE_NX } } }
            };
        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="slnPath">slnファイルのパス</param>
        /// <param name="filePath">プロジェクトファイルのパス</param>
        /// <param name="configuration">ビルドの種別</param>
        /// <param name="platform">プラットフォームの種別</param>
        /// <param name="dependencies">該当プロジェクトファイルが依存しているプロジェクトのGUID一覧
        /// </param>
        /// <param name="excludeDependencies">依存関係から除外するプロジェクトファイル名の一覧
        /// </param>
        /// <param name="projectsByGuid">slnファイルに定義されているGUIDとプロジェクト情報の一覧
        /// </param>
        /// <param name="dependenciesType">
        /// 　プロジェクトファイルが依存設定されたプロジェクトファイルを解析するかどうか</param>
        public ProjectFile(string slnPath, string filePath, string configuration, string platform,
            IReadOnlyList<string> dependencies,
            IReadOnlyList<string> excludeDependencies,
            IReadOnlyDictionary<string, ProjectInSolution> projectsByGuid,
            DependenciesType dependenciesType)
        {
            try
            {
                var xmlDocument = new XmlDocument();

                this.ProjectFileName = filePath;
                this.Configuration = configuration;
                this.Platform = platform;
                this.DependenciesCsproj = new List<string>();
                this.IsErrorOccurred = false;
                this.ErrorOccurredMessage = string.Empty;
                xmlDocument.Load(this.ProjectFileName);
                LoadSourceFileList(xmlDocument);

                //  依存関係に設定されたプロジェクトファイルの解析
                DependenciesProjectFiles = new List<ProjectFile>();
                if (dependenciesType == DependenciesType.NEED_DEPENDENCIES)
                {
                    foreach (var onedep in dependencies)
                    {
                        var projectItem = projectsByGuid[onedep];
                        // slnファイル内に記載されたプロジェクトファイルのフルパスを取得する。
                        var baseUri = new Uri(slnPath);
                        var projectFilePath = new Uri(baseUri, projectItem.RelativePath).LocalPath;

                        var ext = Path.GetExtension(projectFilePath);
                        if (ext != VcProjectFileExtention)
                        {
                            if (ext == CsProjectFileExtention)
                            {
                                DependenciesCsproj.Add(projectFilePath);
                            }
                            continue;
                        }
                        var isExclude = false;
                        foreach (var excludeDependency in excludeDependencies)
                        {
                            if (projectFilePath.Contains(excludeDependency) == true)
                            {
                                isExclude = true;
                                break;
                            }
                        }
                        if (isExclude == true)
                        {
                            continue;
                        }

                        // 依存関係のプロジェクトファイル解析時は更なる依存関係の解析はしない。
                        // DependenciesType.UNNEED_DEPENDENCIES
                        var projectFile = new ProjectFile(slnPath,
                            projectFilePath,
                            configuration,
                            platform,
                            projectItem.Dependencies,
                            excludeDependencies,
                            projectsByGuid,
                            DependenciesType.UNNEED_DEPENDENCIES);

                        DependenciesProjectFiles.Add(projectFile);
                    }
                }

                var properties = new Dictionary<string, string>()
                {
                    { PropertiesPlatform, platform },
                    { PropertiesConfiguration, configuration }
                };

                var ToolsVersion = new Lazy<Tuple<string, string>>(GetToolsVersion);

                this.ProjectInstanceImpl = new ProjectInstance(
                    this.ProjectFileName,
                    properties,
                    ToolsVersion.Value.Item1,
                    ToolsVersion.Value.Item2,
                    ProjectCollection.GlobalProjectCollection);
            }
            catch (Exception e)
            {
                this.IsErrorOccurred = true;
                this.ErrorOccurredMessage = e.Message;
            }
        }
        /// <summary>
        /// 中間ファイルの一覧を作成する。
        /// </summary>
        /// <param name="spec">Generic/NX</param>
        /// <param name="slnConfiguration">
        /// slnファイル内に定義されたビルドの種別[Debug/Develop/Release]</param>
        /// <returns>中間ファイルの一覧</returns>
        public List<string> GetObjectFileList(string spec, string slnConfiguration)
        {
            var objectFileList = new List<string>();

            if (spec == string.Empty)
            {
                return objectFileList;
            }

            var metas = this.ProjectInstanceImpl
                .ItemDefinitions[ProjectFileItemDefineClCompile].Metadata;

            var objectFilePath = metas
                .FirstOrDefault(x => x.Name == ProjectFileItemDefineObjectFileName)
                .EvaluatedValue;

            if (this.OutputTypeDictionary[spec][this.Platform] == MsbuildOutputType.TYPE_NX)
            {
                objectFilePath = objectFilePath.Replace(ProjectFileItemDefineObjectFileNamePattern,
                    string.Empty);
            }
            foreach (string sourceFileName in this.SourceFileList)
            {
                string sourceFileNameWithExtension;
                if (this.OutputTypeDictionary[spec][this.Platform]
                    == MsbuildOutputType.TYPE_WINDOWS)
                {
                    sourceFileNameWithExtension = Path.ChangeExtension(
                        sourceFileName, IntermediatesExtentionObj);
                }
                else
                {
                    sourceFileNameWithExtension = Path.ChangeExtension(
                        sourceFileName, IntermediatesExtentionO);
                }
                sourceFileNameWithExtension = Path.Combine(
                    objectFilePath, sourceFileNameWithExtension);
                objectFileList.Add(sourceFileNameWithExtension);
            }
            return objectFileList;
        }
        /// <summary>
        /// 成果物ファイルの一覧を作成する。
        /// </summary>
        /// <param name="slnConfiguration">
        /// slnファイル内に定義されたビルドの種別[Debug/Develop/Release]</param>
        /// <returns>成果物ファイルの一覧</returns>
        /// <note>
        /// ひとつのslnファイルに複数のvcxprojファイルを含む場合、
        /// 成果物ファイルの一覧は複数となる。
        /// </note>
        public List<string> GetOutputFiles(string slnConfiguration)
        {
            var outputFiles = new List<string>();
            var metas = this.ProjectInstanceImpl
                .ItemDefinitions[ProjectFileItemDefineLink].Metadata;
            var outputFile = metas
                .FirstOrDefault(x => x.Name == ProjectFileItemDefineOutputFile)
                .EvaluatedValue;
            outputFiles.Add(outputFile);
            return outputFiles;
        }
        /// <summary>
        /// プロジェクトファイルに依存設定されたプロジェクトファイルの成果物ファイルパスを取得する
        /// </summary>
        /// <param name="slnConfiguration">
        /// slnファイル内に定義されたビルドの種別[Debug/Develop/Release]</param>
        /// <returns>プロジェクトファイルに依存設定された
        /// 　プロジェクトファイルの成果物ファイルパス一覧
        /// </returns>
        public List<string> GetDependenciesOutputFiles(string slnConfiguration)
        {
            var res = new List<string>();
            foreach (var one in this.DependenciesProjectFiles)
            {
                res.AddRange(one.GetOutputFiles(slnConfiguration));
            }
            return res;
        }

        /// <summary>
        /// プロジェクトファイルに含まれるソースファイルの一覧を抽出する。
        /// </summary>
        /// <note>
        /// NXでビルド時はconfiguration／platformの組み合わせによりビルド対象外
        /// となっているソースコードがある。（ExcludedFromBuild）
        /// そのビルド対象外のソースファイルは一覧抽出の対象外とする。
        /// </note>
        /// <param name="xmlDocument">XML形式のプロジェクトファイル</param>
        private void LoadSourceFileList(XmlDocument xmlDocument)
        {
            this.SourceFileList = new List<string>();

            var nodelist
                = xmlDocument.DocumentElement.GetElementsByTagName(TargetTypeDefinition.CLCOMPILE);

            foreach (var xmlNode in nodelist)
            {
                var xmlElement = (XmlElement)xmlNode;
                if (xmlElement.ParentNode.Name == ProjectFileElementItemGroup)
                {
                    //  ビルド対象外の判断
                    bool isExcludedFromBuild = false;
                    var nodelistTagName = xmlElement.GetElementsByTagName(
                                                    ProjectFileElementExcludedFromBuild);
                    foreach (var nodelistTagNameItem in nodelistTagName)
                    {
                        var nodelistTagNameItemElement = (XmlElement)nodelistTagNameItem;
                        if (nodelistTagNameItemElement.GetAttribute(
                            ProjectFileAttributeCondition)
                            .Contains(string.Format(
                                "'{0}|{1}'", this.Configuration, this.Platform)) &&
                                nodelistTagNameItemElement.InnerText == ProjectFileValueTrue)
                        {
                            var sourceFilename
                                = xmlElement.GetAttribute(ProjectFileAttributeInclude);
                            //　ビルド除外対象のソースファイル
                            isExcludedFromBuild = true;
                            break;
                        }
                    }
                    //  ビルド対象であれば一覧に追加
                    if (isExcludedFromBuild == false)
                    {
                        var sourceFilename
                            = xmlElement.GetAttribute(ProjectFileAttributeInclude);
                        // パス部分は除去する。
                        sourceFilename = Path.GetFileName(sourceFilename);
                        this.SourceFileList.Add(sourceFilename);
                    }
                }
            }
        }

        /// <summary>
        /// MSBuildのツールバージョンを取得する。
        /// </summary>
        /// <returns>MSBuildのツールバージョン</returns>
        private static Tuple<string, string> GetToolsVersion()
        {
            var dict = Environment.GetEnvironmentVariables();

            if (dict.Contains(MsbuildDictContains140))
            {
                return new Tuple<string, string>(BuildRuleMaker.MsbuildToolsetVersion140,
                    BuildRuleMaker.MsbuildToolsetVersion140);
            }

            throw new VsSolutionBuilderNinjatExecuteorException(
                "Supported Visual Studio version not found");
        }
    }
}
