﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace TestRunner
{
    using System;
    using System.Collections.Generic;
    using System.Linq;
    using System.Text;
    using System.Text.RegularExpressions;
    using Executer;
    using static TargetManager;

    /// <summary>
    /// テストリスト中の変数を管理します。
    /// </summary>
    internal sealed class VariableManager
    {
        static private readonly string[] reservedKeys = {
            "_", "LastExitCode",
        };

        /// <summary>
        /// VariableManager クラスの新しいインスタンスを初期化します。
        /// </summary>
        /// <param name="pairs">管理対象となる変数を記録した辞書です。</param>
        internal VariableManager(Dictionary<string, string> pairs)
        {
            this.Pairs = new Dictionary<string, string>(pairs);

            this.RemoveResultDirectory();

            this.RemovePlatform();

            this.RemoveBuildType();

            this.RemoveLogPath();

            this.RemoveReportPath();
        }

        /// <summary>
        /// VariableManager クラスの新しいインスタンスを初期化します。
        /// </summary>
        internal VariableManager()
            : this(new Dictionary<string, string>())
        {
        }

        private Dictionary<string, string> Pairs { get; set; }

        /// <summary>
        /// 自身の開発機を言及する予約テストリスト変数が含まれているか否かを表す値を返します。
        /// </summary>
        /// <param name="context">テストコンテキストです。</param>
        /// <returns>自身の開発機を言及する予約テストリスト変数が含まれているか否かを表す値です。</returns>
        internal static bool RefersToOwnTarget(TestContext context)
        {
            string suffix = MakeTargetSuffix(context.RoleType, context.UnitId);

            var variables = new List<string>
            {
                Key.TargetName,
                Key.TargetInterface,
                Key.TargetAddress,
            };

            variables = variables.Select(x => x + suffix).ToList();

            var pattern = string.Join("|", variables);

            pattern = string.Format(@"\${{({0})}}", pattern);

            var regex = new Regex(pattern, RegexOptions.Compiled);

            return regex.IsMatch(context.Option) ||
                regex.IsMatch(context.Parameter) ||
                regex.IsMatch(context.RunSettings);
        }

        /// <summary>
        /// 予約テストリスト変数 RESULT_DIR を環境に追加します。
        /// </summary>
        /// <param name="value">予約テストリスト変数 RESULT_DIR に設定する値です。</param>
        internal void AddResultDirectory(string value)
        {
            this.Pairs.Add(Key.ResultDirectory, value);
        }

        /// <summary>
        /// 予約テストリスト変数 PLATFORM を環境に追加します。
        /// </summary>
        /// <param name="value">予約テストリスト変数 PLATFORM に設定する値です。</param>
        internal void AddPlatfrom(string value)
        {
            this.Pairs.Add(Key.Platform, value);
        }

        /// <summary>
        /// 予約テストリスト変数 BUILD_TYPE を環境に追加します。
        /// </summary>
        /// <param name="value">予約テストリスト変数 BUILD_TYPE に設定する値です。</param>
        internal void AddBuildType(string value)
        {
            this.Pairs.Add(Key.BuildType, value);
        }

        /// <summary>
        /// 予約テストリスト変数 LOG_PATH を環境に追加します。
        /// </summary>
        /// <param name="value">予約テストリスト変数 LOG_PATH に設定する値です。</param>
        internal void AddLogPath(string value)
        {
            this.Pairs.Add(Key.LogPath, value);
        }

        /// <summary>
        /// 予約テストリスト変数 REPORT_PATH を環境に追加します。
        /// </summary>
        /// <param name="value">予約テストリスト変数 REPORT_PATH に設定する値です。</param>
        internal void AddReportPath(string value)
        {
            this.Pairs.Add(Key.ReportPath, value);
        }

        /// <summary>
        /// 開発機関連の予約テストリスト変数を環境に追加します。
        /// </summary>
        /// <param name="roleType">ロールタイプです。</param>
        /// <param name="unitId">ユニット ID です。</param>
        /// <param name="entry">開発機関連の予約テストリスト変数に設定する値です。</param>
        internal void AddTarget(
            RoleType roleType, uint unitId, TargetEntry entry)
        {
            string suffix = MakeTargetSuffix(roleType, unitId);

            this.Pairs.Add(Key.TargetName + suffix, entry.Name);

            this.Pairs.Add(Key.TargetInterface + suffix, entry.Interface);

            this.Pairs.Add(Key.TargetAddress + suffix, entry.Address);
        }

        /// <summary>
        /// 予約テストリスト変数 RESULT_DIR を環境から削除します。
        /// </summary>
        internal void RemoveResultDirectory()
        {
            this.Pairs.Remove(Key.ResultDirectory);
        }

        /// <summary>
        /// 予約テストリスト変数 PLATFORM を環境から削除します。
        /// </summary>
        internal void RemovePlatform()
        {
            this.Pairs.Remove(Key.Platform);
        }

        /// <summary>
        /// 予約テストリスト変数 BUILD_TYPE を環境から削除します。
        /// </summary>
        internal void RemoveBuildType()
        {
            this.Pairs.Remove(Key.BuildType);
        }

        /// <summary>
        /// 予約テストリスト変数 LOG_PATH を環境から削除します。
        /// </summary>
        internal void RemoveLogPath()
        {
            this.Pairs.Remove(Key.LogPath);
        }

        /// <summary>
        /// 予約テストリスト変数 REPORT_PATH を環境から削除します。
        /// </summary>
        internal void RemoveReportPath()
        {
            this.Pairs.Remove(Key.ReportPath);
        }

        /// <summary>
        /// 開発機関連の予約テストリスト変数を環境から削除します。
        /// </summary>
        /// <param name="roleType">ロールタイプです。</param>
        /// <param name="unitId">ユニット ID です。</param>
        internal void RemoveTarget(RoleType roleType, uint unitId)
        {
            string suffix = MakeTargetSuffix(roleType, unitId);

            this.Pairs.Remove(Key.TargetName + suffix);

            this.Pairs.Remove(Key.TargetInterface + suffix);

            this.Pairs.Remove(Key.TargetAddress + suffix);
        }

        /// <summary>
        /// 指定された文字列中の変数を展開します。
        /// </summary>
        /// <param name="str">変数の展開の対象となる文字列です。</param>
        /// <returns>変数が展開された文字列です。</returns>
        internal string ExpandVariables(string str)
        {
            return ExpandVariables(str, this.Pairs, new HashSet<string>());
        }

        private static string MakeTargetSuffix(RoleType roleType, uint unitId)
        {
            return string.Format(
                "{0}{1}",
                roleType == RoleType.Test
                    ? string.Empty
                    : string.Format("_{0}", roleType.ToString().ToUpper()),
                string.Format("_{0}", unitId));
        }

        private static string ExpandVariables(
            string str,
            Dictionary<string, string> pairs, ISet<string> exceptKeys)
        {
            var sb = new StringBuilder();

            var position = 0;

            var matches = Regex.Matches(str, @"`?\${([_A-Za-z][_A-Za-z\d]*)}");

            foreach (Match match in matches)
            {
                sb.Append(str.Substring(position, match.Index - position));

                position = match.Index + match.Value.Length;

                if (match.Value[0] == '`')
                {
                    sb.Append(match.Value.Substring(1));
                }
                else
                {
                    var key = match.Groups[1].Value;

                    if (pairs.ContainsKey(key) && !exceptKeys.Contains(key))
                    {
                        sb.Append(
                            ExpandVariables(
                                pairs[key],
                                pairs,
                                new HashSet<string>(
                                    exceptKeys.Concat(new[] { key }))));
                    }
                    else
                    {
                        string value = reservedKeys.Contains(key) ?
                            null : Environment.GetEnvironmentVariable(key);

                        if (value != null)
                        {
                            sb.Append(value);
                        }
                        else
                        {
                            sb.Append(match.Value);
                        }
                    }
                }
            }

            sb.Append(str.Substring(position));

            return sb.ToString();
        }

        private static class Key
        {
            internal const string ResultDirectory = "RESULT_DIR";

            internal const string Platform = "PLATFORM";

            internal const string BuildType = "BUILD_TYPE";

            internal const string LogPath = "LOG_PATH";

            internal const string ReportPath = "REPORT_PATH";

            internal const string TargetName = "TARGET_NAME";

            internal const string TargetInterface = "TARGET_INTERFACE";

            internal const string TargetAddress = "TARGET_ADDRESS";
        }
    }
}
