﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace TestRunner
{
    using System.Collections.Generic;
    using System.Linq;
    using System.Text.RegularExpressions;
    using Executer;

    /// <summary>
    /// テストユニットのローカル環境の検証と保持を行います。
    /// </summary>
    internal sealed class TestUnitInfo
    {
        private static readonly Regex ProgramIdPattern=
            new Regex(@"(gamecard|0x[\dA-Fa-f]{16})");

        private static readonly IReadOnlyList<string> Platforms = new[] {
            PlatformDefinition.Win32,
            PlatformDefinition.Win32Vs2013,
            PlatformDefinition.Win32Vs2015,
            PlatformDefinition.Win32Vs2017,
            PlatformDefinition.X86,
            PlatformDefinition.Win64,
            PlatformDefinition.Win64Vs2013,
            PlatformDefinition.Win64Vs2015,
            PlatformDefinition.Win64Vs2017,
            PlatformDefinition.X64,
            PlatformDefinition.BdslImx6,
            PlatformDefinition.JetsonTk1,
            PlatformDefinition.JetsonTk2,
            PlatformDefinition.JetsonTk2A64,
            PlatformDefinition.NxOnJetsonTk2,
            PlatformDefinition.NxOnWin32,
            PlatformDefinition.NxWin32,
            PlatformDefinition.NxWin32Vs2013,
            PlatformDefinition.NxWin32Vs2015,
            PlatformDefinition.NxWin32Vs2017,
            PlatformDefinition.NxOnWin64,
            PlatformDefinition.NxWin64,
            PlatformDefinition.NxWin64Vs2013,
            PlatformDefinition.NxWin64Vs2015,
            PlatformDefinition.NxWin64Vs2017,
            PlatformDefinition.Nx32,
            PlatformDefinition.Nx64,
            PlatformDefinition.NxFp2,
            PlatformDefinition.NxFp2A32,
            PlatformDefinition.NxFp2A64,
        };

        private static readonly IReadOnlyList<string> BuildTypes = new[] {
            BuildTypeDefinition.Debug,
            BuildTypeDefinition.Develop,
            BuildTypeDefinition.Release
        };

        private static readonly IReadOnlyList<ProcessType>
            ProcessTypes = new[] {
            ProcessType.Standard,
            ProcessType.Background
        };

        private static readonly IReadOnlyList<string>
            ApplicationProgramFormats = new[] {
            ApplicationProgramFormatDefinition.Raw,
            ApplicationProgramFormatDefinition.Nca,
            ApplicationProgramFormatDefinition.Nsp,
        };

        /// <summary>
        /// TestUnitInfo クラスの新しいインスタンスを初期化します。
        /// </summary>
        /// <param name="node">テストノードです。</param>
        internal TestUnitInfo(object node)
        {
            this.MappingNode = node as Dictionary<string, object>;

            this.Resources = new List<string>();

            this.VariableManager = new VariableManager();
        }

        /// <summary>
        /// ワーキングディレクトリの絶対パスを取得または設定します。
        /// </summary>
        internal string WorkingDirectory { get; set; }

        /// <summary>
        /// テスト結果ファイルの出力先となるディレクトリの絶対パスを取得または設定します。
        /// </summary>
        internal string ResultDirectory { get; set; }

        /// <summary>
        /// 並列実行に対応しているか否かを示す値を取得または設定します。
        /// </summary>
        internal bool Parallelizable { get; set; }

        /// <summary>
        /// ブレークレベルを取得または設定します。
        /// </summary>
        internal BreakLevel BreakLevel { get; set; }

        /// <summary>
        /// ロールタイプを取得または設定します。
        /// </summary>
        internal RoleType RoleType { get; set; }

        /// <summary>
        /// テスト ID を取得または設定します。
        /// </summary>
        internal uint TestId { get; set; }

        /// <summary>
        /// グローバルリピート ID を取得または設定します。
        /// </summary>
        internal uint GlobalRepeatId { get; set; }

        /// <summary>
        /// ローカルリピート ID を取得または設定します。
        /// </summary>
        internal uint LocalRepeatId { get; set; }

        /// <summary>
        /// テストノードに指定されたユニットの数を取得または設定します。
        /// </summary>
        internal uint UnitCount { get; set; }

        /// <summary>
        /// ユニット ID を取得または設定します。
        /// </summary>
        internal uint UnitId { get; set; }

        /// <summary>
        /// 実行するプログラムが参照するファイルの絶対パスを取得または設定します。
        internal IReadOnlyList<string> Resources { get; set; }

        /// <summary>
        /// 実行するプログラムの失敗条件とするログ出力のパターンを取得または設定します。
        /// </summary>
        internal IReadOnlyList<string> FailurePatterns { get; set; }

        /// <summary>
        /// タイムアウトまでの待ち時間（秒）を取得または設定します。
        /// </summary>
        internal int Timeout { get; set; }

        /// <summary>
        /// テスト対象のプラットフォームを取得または設定します。
        /// </summary>
        internal string Platform { get; set; }

        /// <summary>
        /// テスト対象のビルドタイプを取得または設定します。
        /// </summary>
        internal string BuildType { get; set; }

        /// <summary>
        /// テスト対象のアプリケーションプログラムフォーマットを取得または設定します。
        /// </summary>
        internal string ApplicationProgramFormat { get; set; }

        /// <summary>
        /// 開発機の名前の正規表現を取得または設定します。
        /// </summary>
        internal string TargetNamePattern { get; set; }

        /// <summary>
        /// 開発機のインターフェイス名の正規表現を取得または設定します。
        /// </summary>
        internal string TargetInterfacePattern { get; set; }

        /// <summary>
        /// 変数マネージャを取得または設定します。
        /// </summary>
        internal VariableManager VariableManager { get; set; }

        private IReadOnlyDictionary<string, object> MappingNode { get; set; }

        /// <summary>
        /// テストユニットを識別するためのハッシュ値を返します。
        /// </summary>
        /// <param name="roleType">ロールタイプです。</param>
        /// <param name="unitId">ユニット ID です。</param>
        /// <returns>テストユニットを識別するためのハッシュ値です。</returns>
        internal static ulong EncodeUnitHash(RoleType roleType, uint unitId)
        {
            return ((ulong)roleType << 32) + unitId;
        }

        /// <summary>
        /// テストユニットを識別するためのハッシュ値からロールタイプを取得します。
        /// </summary>
        /// <param name="unitHash">テストユニットを識別するためのハッシュ値です。</param>
        /// <returns>ロールタイプです。</returns>
        internal static RoleType DecodeRoleType(ulong unitHash)
        {
            return (RoleType)(unitHash >> 32);
        }

        /// <summary>
        /// テストユニットを識別するためのハッシュ値からユニット ID を取得します。
        /// </summary>
        /// <param name="unitHash">テストユニットを識別するためのハッシュ値です。</param>
        /// <returns>ユニット ID です。</returns>
        internal static uint DecodeUnitId(ulong unitHash)
        {
            return (uint)(unitHash & uint.MaxValue);
        }

        /// <summary>
        /// マッピングノードからテストユニットノードと共通の要素を抽出します。
        /// </summary>
        /// <param name="mappingNode">マッピングノードです。</param>
        /// <returns>抽出された要素です。</returns>
        internal static Dictionary<string, object> ExtractCommonElements(
            IReadOnlyDictionary<string, object> mappingNode)
        {
            var node = new Dictionary<string, object>();

            string[] keys = new[]
            {
                Tag.Path,
                Tag.ProgramId,
                Tag.ProgramName,
                Tag.TestName,
                Tag.Option,
                Tag.Parameter,
                Tag.Resources,
                Tag.FailurePatterns,
                Tag.Platform,
                Tag.BuildType,
                Tag.ApplicationProgramFormat,
                Tag.ReportPath,
                Tag.TargetName,
                Tag.TargetInterface,
                Tag.PostProcessorPath,
                Tag.PostProcessorOption,
            };

            foreach (string key in keys)
            {
                if (mappingNode.ContainsKey(key))
                {
                    node[key] = mappingNode[key];
                }
            }

            return node;
        }

        /// <summary>
        /// マッピングノードが有効な Resources ノードを含むかどうかを示す値を取得します。
        /// </summary>
        /// <param name="mappingNode">マッピングノードです。</param>
        /// <returns>マッピングノードが Resources ノードを含むかどうかを示す値です。</returns>
        internal static bool HasValidResourcesNode(
            IReadOnlyDictionary<string, object> mappingNode)
        {
            return TestListValidation.HasValidSequencePathStringNode(
                mappingNode, Tag.Resources);
        }

        /// <summary>
        /// Resources ノードの値をマッピングノードから取得します。
        /// </summary>
        /// <param name="mappingNode">マッピングノードです。</param>
        /// <returns>Resources ノードの値です。</returns>
        internal static List<string> GetResourcesNodeValue(
            IReadOnlyDictionary<string, object> mappingNode)
        {
            var sequenceNode = (List<object>)mappingNode[Tag.Resources];

            return sequenceNode.Cast<string>().ToList();
        }

        /// <summary>
        /// マッピングノードが有効な FailurePatterns ノードを含むかどうかを示す値を取得します。
        /// </summary>
        /// <param name="mappingNode">マッピングノードです。</param>
        /// <returns>マッピングノードが FailurePatterns ノードを含むかどうかを示す値です。</returns>
        internal static bool HasValidFailurePatternsNode(
            IReadOnlyDictionary<string, object> mappingNode)
        {
            return TestListValidation.HasValidSequenceNodeOf<string>(
                mappingNode, Tag.FailurePatterns);
        }

        /// <summary>
        /// FailurePatterns ノードの値をマッピングノードから取得します。
        /// </summary>
        /// <param name="mappingNode">マッピングノードです。</param>
        /// <returns>FailurePatterns ノードの値です。</returns>
        internal static List<string> GetFailurePatternsNodeValue(
            IReadOnlyDictionary<string, object> mappingNode)
        {
            var sequenceNode = (List<object>)mappingNode[Tag.FailurePatterns];

            return sequenceNode.Cast<string>().ToList();
        }

        /// <summary>
        /// マッピングノードが有効な Platform ノードを含むかどうかを示す値を取得します。
        /// </summary>
        /// <param name="mappingNode">マッピングノードです。</param>
        /// <returns>マッピングノードが Platform ノードを含むかどうかを示す値です。</returns>
        internal static bool HasValidPlatformNode(
            IReadOnlyDictionary<string, object> mappingNode)
        {
            return TestListValidation.HasValidChoiceNode(
                mappingNode, Tag.Platform, Platforms);
        }

        /// <summary>
        /// Platform ノードの値をマッピングノードから取得します。
        /// </summary>
        /// <param name="mappingNode">マッピングノードです。</param>
        /// <returns>Platform ノードの値です。</returns>
        internal static string GetPlatformNodeValue(
            IReadOnlyDictionary<string, object> mappingNode)
        {
            return (string)mappingNode[Tag.Platform];
        }

        /// <summary>
        /// マッピングノードが有効な BuildType ノードを含むかどうかを示す値を取得します。
        /// </summary>
        /// <param name="mappingNode">マッピングノードです。</param>
        /// <returns>マッピングノードが BuildType ノードを含むかどうかを示す値です。</returns>
        internal static bool HasValidBuildTypeNode(
            IReadOnlyDictionary<string, object> mappingNode)
        {
            return TestListValidation.HasValidChoiceNode(
                mappingNode, Tag.BuildType, BuildTypes);
        }

        /// <summary>
        /// BuildType ノードの値をマッピングノードから取得します。
        /// </summary>
        /// <param name="mappingNode">マッピングノードです。</param>
        /// <returns>BuildType ノードの値です。</returns>
        internal static string GetBuildTypeNodeValue(
            IReadOnlyDictionary<string, object> mappingNode)
        {
            return (string)mappingNode[Tag.BuildType];
        }

        /// <summary>
        /// マッピングノードが有効な ApplicationProgramFormat ノードを含むかどうかを示す値を取得します。
        /// </summary>
        /// <param name="mappingNode">マッピングノードです。</param>
        /// <returns>マッピングノードが ApplicationProgramFormat ノードを含むかどうかを示す値です。</returns>
        internal static bool HasValidApplicationProgramFormatNode(
            IReadOnlyDictionary<string, object> mappingNode)
        {
            return TestListValidation.HasValidChoiceNode(
                mappingNode,
                Tag.ApplicationProgramFormat,
                ApplicationProgramFormats);
        }

        /// <summary>
        /// ApplicationProgramFormat ノードの値をマッピングノードから取得します。
        /// </summary>
        /// <param name="mappingNode">マッピングノードです。</param>
        /// <returns>ApplicationProgramFormat ノードの値です。</returns>
        internal static string GetApplicationProgramFormatNodeValue(
            IReadOnlyDictionary<string, object> mappingNode)
        {
            return (string)mappingNode[Tag.ApplicationProgramFormat];
        }

        /// <summary>
        /// マッピングノードが有効な TargetName ノードを含むかどうかを示す値を取得します。
        /// </summary>
        /// <param name="mappingNode">マッピングノードです。</param>
        /// <returns>マッピングノードが TargetName ノードを含むかどうかを示す値です。</returns>
        internal static bool HasValidTargetNamePatternNode(
            IReadOnlyDictionary<string, object> mappingNode)
        {
            return TestListValidation.HasValidRegexStringNode(
                mappingNode, Tag.TargetName);
        }

        /// <summary>
        /// TargetName ノードの値をマッピングノードから取得します。
        /// </summary>
        /// <param name="mappingNode">マッピングノードです。</param>
        /// <returns>TargetName ノードの値です。</returns>
        internal static string GetTargetNamePatternNodeValue(
            IReadOnlyDictionary<string, object> mappingNode)
        {
            return (string)mappingNode[Tag.TargetName];
        }

        /// <summary>
        /// マッピングノードが有効な TargetInterface ノードを含むかどうかを示す値を取得します。
        /// </summary>
        /// <param name="mappingNode">マッピングノードです。</param>
        /// <returns>マッピングノードが TargetInterface ノードを含むかどうかを示す値です。</returns>
        internal static bool HasValidTargetInterfacePatternNode(
            IReadOnlyDictionary<string, object> mappingNode)
        {
            return TestListValidation.HasValidRegexStringNode(
                mappingNode, Tag.TargetInterface);
        }

        /// <summary>
        /// TargetInterface ノードの値をマッピングノードから取得します。
        /// </summary>
        /// <param name="mappingNode">マッピングノードです。</param>
        /// <returns>TargetInterface ノードの値です。</returns>
        internal static string GetTargetInterfacePatternNodeValue(
            IReadOnlyDictionary<string, object> mappingNode)
        {
            return (string)mappingNode[Tag.TargetInterface];
        }

        /// <summary>
        /// マッピングノードが有効な Units ノードを含むかどうかを示す値を取得します。
        /// </summary>
        /// <param name="mappingNode">マッピングノードです。</param>
        /// <returns>マッピングノードが有効な Units ノードを含むかどうかを示す値です。</returns>
        internal static bool HasValidUnitsNode(
            IReadOnlyDictionary<string, object> mappingNode)
        {
            return !HasKeyConflict(mappingNode, Tag.Units, Tag.Path) &&
                   !HasKeyConflict(mappingNode, Tag.Units, Tag.ProgramId) &&
                   !HasKeyConflict(mappingNode, Tag.Units, Tag.ProgramName) &&
                   TestListValidation.HasValidSequenceNode(
                       mappingNode, Tag.Units);
        }

        /// <summary>
        /// マッピングノードが不正な Units ノードを含むかどうかを示す値を取得します。
        /// </summary>
        /// <param name="message">不正なノードだと判断された理由を取得します。</param>
        /// <param name="mappingNode">マッピングノードです。</param>
        /// <returns>マッピングノードが不正な Units ノードを含むかどうかを示す値です。</returns>
        internal static bool HasWrongUnitsNode(
            out string message,
            IReadOnlyDictionary<string, object> mappingNode)
        {
            var tags = new[] { Tag.Path, Tag.ProgramId, Tag.ProgramName };

            foreach (var tag in tags)
            {
                if (HasKeyConflict(mappingNode, Tag.Units, tag))
                {
                    message = $"'{Tag.Units}' conflicting with '{tag}'";

                    return true;
                }
            }

            return TestListValidation.HasWrongSequenceNode(
                out message, mappingNode, Tag.Units);
        }

        /// <summary>
        /// Units ノードの値をマッピングノードから取得します。
        /// </summary>
        /// <param name="mappingNode">マッピングノードです。</param>
        /// <returns>Units ノードの値です。</returns>
        internal static IReadOnlyList<object> GetUnitsNodeValue(
            IReadOnlyDictionary<string, object> mappingNode)
        {
            return (List<object>)mappingNode[Tag.Units];
        }

        /// <summary>
        /// マッピングノードが WRONG_YAML に該当する共通ノードを含むかどうかを示す値を取得します。
        /// </summary>
        /// <param name="str">エラーメッセージを取得します。</param>
        /// <param name="mappingNode">マッピングノードです。</param>
        /// <returns>マッピングノードが WRONG_YAML に該当する共通ノードを含むかどうかを示す値です。</returns>
        internal static bool HasWrongCommonNode(
            out string str, IReadOnlyDictionary<string, object> mappingNode)
        {
            if (TestListValidation.HasWrongSequencePathStringNode(
                    out str, mappingNode, Tag.Resources))
            {
                return true;
            }

            if (TestListValidation.HasWrongSequenceNodeOf<string>(
                    out str, mappingNode, Tag.FailurePatterns))
            {
                return true;
            }

            if (TestListValidation.HasWrongChoiceNode(
                    out str, mappingNode, Tag.Platform, Platforms))
            {
                return true;
            }

            if (TestListValidation.HasWrongChoiceNode(
                    out str, mappingNode, Tag.BuildType, BuildTypes))
            {
                return true;
            }

            if (TestListValidation.HasWrongChoiceNode(
                    out str, mappingNode, Tag.ApplicationProgramFormat,
                    ApplicationProgramFormats))
            {
                return true;
            }

            if (TestListValidation.HasWrongRegexStringNode(
                    out str, mappingNode, Tag.TargetName))
            {
                return true;
            }

            if (TestListValidation.HasWrongRegexStringNode(
                    out str, mappingNode, Tag.TargetInterface))
            {
                return true;
            }

            str = string.Empty;

            return false;
        }

        /// <summary>
        /// WRONG_YAML に該当するテストノードかどうかを示す値を取得します。
        /// </summary>
        /// <param name="str">エラーメッセージを取得します。</param>
        /// <returns>WRONG_YAML に該当するテストノードかどうかを示す値です。</returns>
        internal bool IsWrongYaml(out string str)
        {
            return IsWrongYaml(
                out str,
                GetExpandedMappingNode(
                    this.MappingNode, this.VariableManager));
        }

        /// <summary>
        /// NO_PATH に該当するテストノードかどうかを示す値を取得します。
        /// </summary>
        /// <returns>NO_PATH に該当するテストノードかどうかを示す値です。</returns>
        internal bool IsNoPath()
        {
            var expandedNode =
                GetExpandedMappingNode(this.MappingNode, this.VariableManager);

            var errorMessage = string.Empty;

            if (IsWrongYaml(out errorMessage, expandedNode))
            {
                return true;
            }

            return !HasValidPathNode(expandedNode) &&
                   !HasValidProgramIdNode(expandedNode) &&
                   !HasValidProgramNameNode(expandedNode);
        }

        /// <summary>
        /// テストコンテキストを取得します。
        /// </summary>
        /// <returns>テストコンテキストです。</returns>
        internal TestContext GetTestContext()
        {
            return new TestContext()
            {
                Path = string.Empty,
                TestName =
                    GetStringValueOrEmpty(this.MappingNode, Tag.TestName),
                RoleType = this.RoleType,
                TestId = this.TestId,
                GlobalRepeatId = this.GlobalRepeatId,
                LocalRepeatId = this.LocalRepeatId,
                UnitCount = this.UnitCount,
                UnitId = this.UnitId,
                TargetName = string.Empty,
                TargetInterface = string.Empty,
                TargetAddress = string.Empty,
                TargetNamePattern =
                    HasValidTargetNamePatternNode(this.MappingNode)
                        ? GetTargetNamePatternNodeValue(this.MappingNode)
                        : this.TargetNamePattern,
                TargetInterfacePattern =
                    HasValidTargetInterfacePatternNode(this.MappingNode)
                        ? GetTargetInterfacePatternNodeValue(this.MappingNode)
                        : this.TargetInterfacePattern,
                ResultCode = ResultCode.PASS,
                Command = string.Empty,
                Option =
                    GetStringValueOrEmpty(this.MappingNode, Tag.Option),
                Parameter =
                    GetStringValueOrEmpty(this.MappingNode, Tag.Parameter),
                Resources = (
                    HasValidResourcesNode(this.MappingNode)
                        ? GetResourcesNodeValue(this.MappingNode)
                        : this.Resources).ToArray(),
                FailurePatterns = (
                    HasValidFailurePatternsNode(this.MappingNode)
                        ? GetFailurePatternsNodeValue(this.MappingNode)
                        : this.FailurePatterns)?.ToArray(),
                RunSettings = string.Empty,
                Timeout = this.Timeout,
                Platform =
                    HasValidPlatformNode(this.MappingNode)
                        ? GetPlatformNodeValue(this.MappingNode)
                        : this.Platform,
                BuildType =
                    HasValidBuildTypeNode(this.MappingNode)
                        ? GetBuildTypeNodeValue(this.MappingNode)
                        : this.BuildType,
                ApplicationProgramFormat =
                    HasValidApplicationProgramFormatNode(this.MappingNode)
                        ? GetApplicationProgramFormatNodeValue(
                              this.MappingNode)
                        : this.ApplicationProgramFormat,
                ProcessType =
                    HasValidProcessTypeNode(this.MappingNode)
                        ? GetProcessNodeValue(this.MappingNode)
                        : ProcessType.Standard,
                Duration = 0,
                OpeningTime = string.Empty,
                ClosingTime = string.Empty,
                ExitCode = 0,
                ErrorMessage = string.Empty,
                WorkingDirectory = this.WorkingDirectory,
                ResultDirectory = this.ResultDirectory,
                TargetPath =
                    GetStringValueOrEmpty(this.MappingNode, Tag.Path),
                TargetProgramId =
                    GetStringValueOrEmpty(this.MappingNode, Tag.ProgramId),
                TargetProgramName =
                    GetStringValueOrEmpty(this.MappingNode, Tag.ProgramName),
                TargetProjectPath = string.Empty,
                TargetEpiPath = string.Empty,
                LogPath = string.Empty,
                ReportPath =
                    GetStringValueOrEmpty(this.MappingNode, Tag.ReportPath),
                CsxPath = string.Empty,
                RunSettingsPath = string.Empty,
                Parallelizable = this.Parallelizable,
                BreakLevel = this.BreakLevel,
                PostProcessorCommand =
                    GetStringValueOrEmpty(
                        this.MappingNode, Tag.PostProcessorPath),
                PostProcessorOption =
                    GetStringValueOrEmpty(
                        this.MappingNode, Tag.PostProcessorOption),
                Dependencies = new string[0],
                DumpFileType = DumpFileTypeDefinition.None,
                DumpFilePath = string.Empty,
            };
        }

        private static string GetStringValueOrEmpty(
            IReadOnlyDictionary<string, object> node, string key)
        {
            return node.ContainsKey(key) && node[key] is string ?
                (string)node[key] : string.Empty;
        }

        private static IReadOnlyDictionary<string, object> GetExpandedMappingNode(
            IReadOnlyDictionary<string, object> mappingNode,
            VariableManager variableManager)
        {
            if (mappingNode == null)
            {
                return mappingNode;
            }

            var expandedNode = mappingNode
                .ToDictionary(x => x.Key, x => x.Value);

            var keys = new[] {
                Tag.Path, Tag.ProgramId, Tag.ProgramName, Tag.ReportPath
            };

            foreach (var key in keys)
            {
                if (expandedNode.ContainsKey(key) &&
                    expandedNode[key] is string)
                {
                    expandedNode[key] = variableManager
                        .ExpandVariables((string)expandedNode[key]);
                }
            }

            return expandedNode;
        }

        private static bool HasValidPathNode(
            IReadOnlyDictionary<string, object> mappingNode)
        {
            return TestListValidation.HasValidPathStringNode(
                mappingNode, Tag.Path);
        }

        private static bool HasValidProgramIdNode(
            IReadOnlyDictionary<string, object> mappingNode)
        {
            return TestListValidation.HasValidPatternStringNode(
                mappingNode, Tag.ProgramId, ProgramIdPattern);
        }

        private static bool HasValidProgramNameNode(
            IReadOnlyDictionary<string, object> mappingNode)
        {
            return TestListValidation.HasValidTypedNode<string>(
                mappingNode, Tag.ProgramName);
        }

        private static ProcessType GetProcessNodeValue(
            IReadOnlyDictionary<string, object> mappingNode)
        {
            return ProcessTypes.First(
                x => x.ToString() == (string)mappingNode[Tag.ProcessType]);
        }

        private static bool HasValidProcessTypeNode(
            IReadOnlyDictionary<string, object> mappingNode)
        {
            return TestListValidation.HasValidChoiceNode(
                mappingNode,
                Tag.ProcessType,
                ProcessTypes.Select(x => x.ToString()).ToList());
        }

        private static bool HasWrongProcessTypeNode(
            out string message,
            IReadOnlyDictionary<string, object> mappingNode)
        {
            return TestListValidation.HasWrongChoiceNode(
                out message,
                mappingNode,
                Tag.ProcessType,
                ProcessTypes.Select(x => x.ToString()).ToList());
        }

        private static bool HasKeyConflict(
            IReadOnlyDictionary<string, object> mappingNode,
            string lhs, string rhs)
        {
            return mappingNode.ContainsKey(lhs) &&
                   mappingNode.ContainsKey(rhs);
        }

        private static bool MissesCorrespondingNode(
            out string message,
            IReadOnlyDictionary<string, object> mappingNode,
            string key, string corrKey)
        {
            if ( mappingNode.ContainsKey(key) &&
                !mappingNode.ContainsKey(corrKey))
            {
                message = $"'{key}' missing the corresponding '{corrKey}'";

                return true;
            }

            message = string.Empty;

            return false;
        }

        private static bool MissesCorrespondingTargetNode(
            out string message,
            IReadOnlyDictionary<string, object> mappingNode,
            string key)
        {
            return MissesCorrespondingNode(
                       out message, mappingNode, key, Tag.ProgramId) &&
                   MissesCorrespondingNode(
                       out message, mappingNode, key, Tag.ProgramName) &&
                   MissesCorrespondingNode(
                       out message, mappingNode, key, Tag.Path);
        }

        private static bool IsWrongYaml(
            out string str, IReadOnlyDictionary<string, object> mappingNode)
        {
            if (mappingNode == null)
            {
                str = "Invalid node type for a test unit";

                return true;
            }

            if (TestListValidation.HasWrongPathStringNode(
                    out str, mappingNode, Tag.Path))
            {
                return true;
            }

            if (TestListValidation.HasWrongPatternStringNode(
                    out str, mappingNode, Tag.ProgramId, ProgramIdPattern))
            {
                return true;
            }

            if (TestListValidation.HasWrongTypedNode<string>(
                    out str, mappingNode, Tag.ProgramName))
            {
                return true;
            }

            if (MissesCorrespondingTargetNode(
                    out str, mappingNode, Tag.TestName) ||
                TestListValidation.HasWrongTestNameStringNode(
                    out str, mappingNode, Tag.TestName))
            {
                return true;
            }

            if (MissesCorrespondingTargetNode(
                    out str, mappingNode, Tag.Option) ||
                TestListValidation.HasWrongTypedNode<string>(
                    out str, mappingNode, Tag.Option))
            {
                return true;
            }

            if (MissesCorrespondingTargetNode(
                    out str, mappingNode, Tag.Parameter) ||
                TestListValidation.HasWrongTypedNode<string>(
                    out str, mappingNode, Tag.Parameter))
            {
                return true;
            }

            if (MissesCorrespondingTargetNode(
                    out str, mappingNode, Tag.ReportPath) ||
                TestListValidation.HasWrongPathStringNode(
                    out str, mappingNode, Tag.ReportPath))
            {
                return true;
            }

            if (MissesCorrespondingTargetNode(
                    out str, mappingNode, Tag.ProcessType) ||
                HasWrongProcessTypeNode(out str, mappingNode))
            {
                return true;
            }

            if (MissesCorrespondingTargetNode(
                    out str, mappingNode, Tag.PostProcessorPath) ||
                TestListValidation.HasWrongPathStringNode(
                    out str, mappingNode, Tag.PostProcessorPath))
            {
                return true;
            }

            if (MissesCorrespondingTargetNode(
                    out str, mappingNode, Tag.PostProcessorOption) ||
                TestListValidation.HasWrongTypedNode<string>(
                    out str, mappingNode, Tag.PostProcessorOption))
            {
                return true;
            }

            if (HasWrongCommonNode(out str, mappingNode))
            {
                return true;
            }

            str = string.Empty;

            return false;
        }

        private static class Tag
        {
            internal const string Path = "Path";

            internal const string ProgramId = "ProgramId";

            internal const string ProgramName = "ProgramName";

            internal const string TestName = "TestName";

            internal const string Option = "Option";

            internal const string Parameter = "Parameter";

            internal const string Resources = "Resources";

            internal const string FailurePatterns = "FailurePatterns";

            internal const string Platform = "Platform";

            internal const string BuildType = "BuildType";

            internal const string ApplicationProgramFormat =
                "ApplicationProgramFormat";

            internal const string ProcessType = "ProcessType";

            internal const string ReportPath = "ReportPath";

            internal const string TargetName = "TargetName";

            internal const string TargetInterface = "TargetInterface";

            internal const string PostProcessorPath = "PostProcessorPath";

            internal const string PostProcessorOption = "PostProcessorOption";

            internal const string Units = "Units";
        }
    }
}
