﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace TestRunner
{
    using System;
    using System.Collections.Generic;
    using System.Collections.ObjectModel;
    using System.IO;
    using System.Linq;
    using System.Text;
    using System.Text.RegularExpressions;
    using System.Xml;
    using Executer;
    using GenerationRule;
    using static TargetManager;

    /// <summary>
    /// テストの実行とその結果の集計を行います。
    /// </summary>
    internal class TestRunner
    {
        private static readonly Regex specExtension = new Regex(
            @"-spec\.[a-zA-Z_0-9]+(\.autogen)?\.vcxproj$",
            RegexOptions.Compiled);

        /// <summary>
        /// TestRunner クラスの新しいインスタンスを初期化します。
        /// </summary>
        internal TestRunner()
        {
            this.Option = new CommandLineOption();

            this.Summarizer = new ResultSummarizer();

            this.Date = DateTime.Now;

            this.ResultRootPath = string.Empty;

            this.TestRootPath = string.Empty;

            this.TargetManager = new TargetManager();

            this.EpiManager = new EpiManager();

            this.Generator = new InstructionGenerator();

            this.BuildRuleMaker = new BuildRuleMaker();
        }

        /// <summary>
        /// コマンドラインオプションを取得または設定します。
        /// </summary>
        internal CommandLineOption Option { get; set; }

        /// <summary>
        /// サマライザーを取得または設定します。
        /// </summary>
        internal ResultSummarizer Summarizer { get; set; }

        private DateTime Date { get; set; }

        private string ResultRootPath { get; set; }

        private string TestRootPath { get; set; }

        private TargetManager TargetManager { get; set; }

        private EpiManager EpiManager { get; set; }

        private InstructionGenerator Generator { get; set; }

        private BuildRuleMaker BuildRuleMaker { get; set; }

        /// <summary>
        /// テスト結果ファイルの出力先となるディレクトリの絶対パスを取得します。
        /// </summary>
        /// <param name="dirPath">親ディレクトリの絶対パスです。</param>
        /// <param name="listId">テストリスト ID です。</param>
        /// <param name="statemantId">テストステートメント ID です。</param>
        /// <param name="platformId">プラットフォーム ID です。</param>
        /// <param name="buildTypeId">ビルドタイプ ID です。</param>
        /// <param name="listFileName">テストリストのファイル名です。</param>
        /// <returns>テスト結果ファイルの出力先となるディレクトリの絶対パスです。</returns>
        internal static string GetResultDirectoryPath(
            string dirPath,
            uint listId, uint statementId, uint platformId, uint buildTypeId,
            string listFileName)
        {
            var listSubName =
                Utility.RemoveExtension(
                    listFileName, ExtensionDefinition.TestList);

            return Path.Combine(
                dirPath,
                string.Format(
                    "{0:D5}_{1:D3}_{2:D2}_{3:D2}_{4}",
                    listId, statementId, platformId, buildTypeId,
                    listSubName));
        }

        /// <summary>
        /// テストリストの評価結果の記録先となるテスト結果ファイルの絶対パスを取得します。
        /// </summary>
        /// <param name="dirPath">テスト結果ファイルの出力先となるディレクトリの絶対パスです。</param>
        /// <param name="listFileName">テストリストのファイル名です。</param>
        /// <returns>テストリストの評価結果の記録先となるテスト結果ファイルの絶対パスです。</returns>
        internal static string GetListResultFilePath(
            string dirPath, string name)
        {
            string path = GetResultSubName(dirPath, RoleType.Test, 0, 0, name);

            return path + ExtensionDefinition.ListResult;
        }

        /// <summary>
        /// テストの実行とその結果の集計を開始します。
        /// </summary>
        /// <returns>終了コードです。</returns>
        internal int Run()
        {
            this.TestRootPath = GetTestRootPath(this.Option);

            this.ResultRootPath = Path.Combine(
                this.Option.OutputPath, this.Date.ToString("yyyyMMddHHmmss"));

            this.Summarizer.BranchName = this.Option.BranchName;

            this.Summarizer.CommitHash = this.Option.CommitHash;

            if (IsParallelModeEnabled(this.Option))
            {
                this.Summarizer.EnablesInstantSummary = false;

                this.BuildRuleMaker = new BuildRuleMaker(this.Option.JobCount)
                {
                    ResultRootPath = this.ResultRootPath,
                    TargetManager = this.TargetManager,
                    EnablesVerboseMode = this.Option.EnablesVerboseMode
                };
            }

            if (this.Option.EnablesVerboseMode)
            {
                this.Summarizer.EnablesInstantSummary = false;
            }

            if (this.Option.EnablesFileListingMode)
            {
                this.Summarizer.EnablesInstantSummary = false;

                this.Summarizer.EnablesFileSummary = true;
            }

            this.Summarizer.Date = this.Date.ToString();

            this.Summarizer.ResultRootPath = this.ResultRootPath;

            this.Summarizer.TestRootPath = this.TestRootPath;

            UpdateTestSearchConditions(this.Summarizer, this.Option);

            this.TargetManager.Platforms = this.Option.Platforms;

            UpdateTargetSearchConditions(this.TargetManager, this.Option);

            Directory.CreateDirectory(this.ResultRootPath);

            if (!this.Option.EnablesFileListingMode)
            {
                this.TargetManager.UpdateTargetEntries(
                    IsParallelModeEnabled(
                        this.Option) ? this.Option.JobCount : 0);
            }

            if (!string.IsNullOrEmpty(this.Option.EpiDirectory))
            {
                this.EpiManager.LoadEpiPaths(this.Option.EpiDirectory);
            }

            this.Summarizer.WriteHeader();

            var fileCount = 0u;

            foreach (var filePath in EnumerateFiles(this.Option))
            {
                ++fileCount;

                try
                {
                    this.EvaluateTestList(fileCount, filePath);
                }
                catch (ListContextException ex)
                {
                    Directory.CreateDirectory(
                        Path.GetDirectoryName(ex.ListContext.Path));

                    this.WriteListResult(ex.ListContext);
                }
            }

            if (IsParallelModeEnabled(this.Option))
            {
                this.BuildRuleMaker.Run();
            }

            int exitCode = this.Summarizer.WriteFooter();

            return this.Option.EnablesQuietExit ? 0 : exitCode;
        }

        private static string GetResultSubName(
            string dirPath, RoleType roleType, uint testId, uint unitId,
            string name)
        {
            return Path.Combine(
                dirPath,
                string.Format(
                    "{0:D2}_{1:D3}_{2:D2}_{3}",
                    (uint)roleType, testId, unitId, name));
        }

        private static string GetTestResultFilePath(
            string dirPath, RoleType roleType, uint testId, uint unitId,
            string name)
        {
            string path =
                GetResultSubName(dirPath, roleType, testId, unitId, name);

            return path + ExtensionDefinition.TestResult;
        }

        private static string GetBaseResultDirectoryPath(
            string dirPath, uint listId, string fileName)
        {
            return GetResultDirectoryPath(
                dirPath,
                listId,
                TestStatementInfo.DefaultStatementId,
                TestStatementInfo.DefaultPlatformId,
                TestStatementInfo.DefaultBuildTypeId,
                fileName);
        }

        private static string GetListNameOfFilePath(
            string filePath, string testRootPath)
        {
            if (filePath.StartsWith(testRootPath))
            {
                filePath = filePath.Substring(testRootPath.Length);

                while (filePath.IndexOfAny(new[] { '\\', '/' }) == 0)
                {
                    filePath = filePath.Substring(1);
                }
            }

            return filePath;
        }

        private static bool IsParallelModeEnabled(CommandLineOption option)
        {
            if (option.EnablesFileListingMode)
            {
                return false;
            }

            return option.EnablesParallelMode;
        }

        private static string GetTestRootPath(CommandLineOption option)
        {
            return (option.FilePaths.Count == 0 &&
                    option.DirectoryPaths.Count == 1)
                ? option.DirectoryPaths.First()
                : Directory.GetCurrentDirectory();
        }

        private static void UpdateTestSearchConditions(
            ResultSummarizer summarizer, CommandLineOption option)
        {
            summarizer.Platforms = option.Platforms;

            summarizer.BuildTypes = option.BuildTypes;

            summarizer.Modules = option.Modules;

            summarizer.Categories = option.Categories;
        }

        private static void UpdateTargetSearchConditions(
            TargetManager targetManager, CommandLineOption option)
        {
            targetManager.TargetNamePattern =
                option.TargetNamePattern;

            targetManager.TargetInterfacePattern =
                option.TargetInterfacePattern;

            targetManager.TargetAddressPattern =
                option.TargetAddressPattern;
        }

        private static IEnumerable<string> EnumerateFiles(
            CommandLineOption option)
        {
            var directoryPath = (option.DirectoryPaths.Count > 0)
                ? option.DirectoryPaths
                : (option.FilePaths.Count == 0)
                    ? new List<string>() { Directory.GetCurrentDirectory() }
                    : new List<string>();

            var filePaths = directoryPath
                .SelectMany(path => Directory.EnumerateFiles(
                    path,
                    "*" + ExtensionDefinition.TestList,
                    SearchOption.AllDirectories))
                .Concat(option.FilePaths);

            return filePaths;
        }

        private static string GetFullPath(string path, TestContext testContext)
        {
            try
            {
                return Path.Combine(testContext.WorkingDirectory, path);
            }
            catch (Exception ex)
            {
                testContext.ResultCode = ResultCode.WRONG_YAML;

                var message = $"Invalid file path: {path}";

                testContext.ErrorMessage = message;

                throw new TestContextException(message, testContext, ex);
            }
        }

        private static void ResolveTargetWithPath(
            TestContext testContext,
            VariableManager variableManager,
             EpiManager epiManager)
        {
            var targetPath = testContext.TargetPath;

            targetPath = variableManager.ExpandVariables(targetPath);

            targetPath =
                Path.Combine(testContext.WorkingDirectory, targetPath);

            testContext.TargetPath = Path.GetFullPath(targetPath);

            var fileName = Path.GetFileName(targetPath);

            Match match = specExtension.Match(fileName);

            testContext.TargetProgramName = Utility.RemoveExtension(
                fileName,
                match.Success ? match.Value : Path.GetExtension(fileName));

            var epiPaths = epiManager.GetEpiPaths(
                testContext.TargetProgramName,
                testContext.Platform, testContext.BuildType);

            if (epiPaths.Length == 1)
            {
                testContext.TargetEpiPath = epiPaths[0];
            }
        }

        private static void ResolveTargetWithId(
            TestContext testContext,
            VariableManager variableManager)
        {
            var programId = testContext.TargetProgramId;

            programId = variableManager.ExpandVariables(programId);

            testContext.TargetProgramId = programId;
        }

        private static void ResolveTargetWithName(
            TestContext testContext,
            VariableManager variableManager,
            EpiManager epiManager)
        {
            var programName = testContext.TargetProgramName;

            programName = variableManager.ExpandVariables(programName);

            testContext.TargetProgramName = programName;

            var epiPaths = epiManager.GetEpiPaths(
                programName, testContext.Platform, testContext.BuildType);

            if (epiPaths.Length == 0)
            {
                testContext.ResultCode = ResultCode.NO_FILE;

                testContext.ErrorMessage =
                    $"File which is corresponding to " +
                    $"the program name '{programName}' is not found";

                throw new TestContextException(
                    testContext.ErrorMessage, testContext);
            }

            if (epiPaths.Length > 1)
            {
                testContext.ResultCode = ResultCode.NOT_UNIQUE;

                testContext.ErrorMessage =
                    $"Multiple files correspond to " +
                    $"the program name '{programName}'";

                throw new TestContextException(
                    testContext.ErrorMessage, testContext);
            }

            testContext.TargetEpiPath = epiPaths[0];

            try
            {
                var epi = epiManager.GetEpi(epiPaths[0]);

                testContext.TargetPath = epi.ExecutableFiles[0].Path;

                testContext.Platform = epi.Platform;

                testContext.BuildType = epi.BuildType;
            }
            catch (Exception ex)
            {
                testContext.ResultCode = ResultCode.WRONG_FILE;

                testContext.ErrorMessage = ex.Message;

                throw new TestContextException(
                    testContext.ErrorMessage, testContext);
            }
        }

        private static ITargetAssigner CreateTargetAssigner(TestRunner runner)
        {
            if (runner.Option.EnablesFileListingMode)
            {
                return new IdleTargetAssigner(runner);
            }

            if (runner.Option.EnablesParallelMode)
            {
                return new ParallelTargetAssigner(runner);
            }

            return new SerialTargetAssigner(runner);
        }

        private static void UpdatePrimaryPropertyOf(
            TestContext testContext,
            VariableManager variableManager,
            EpiManager epiManager)
        {
            try
            {
                variableManager
                    .AddResultDirectory(testContext.ResultDirectory);

                if (testContext.TargetProgramId != string.Empty)
                {
                    ResolveTargetWithId(testContext, variableManager);
                }
                else
                {
                    if (testContext.TargetProgramName != string.Empty)
                    {
                        ResolveTargetWithName(
                            testContext, variableManager, epiManager);
                    }
                    else
                    {
                        ResolveTargetWithPath(
                            testContext, variableManager, epiManager);
                    }
                }

                if (string.IsNullOrEmpty(testContext.TestName))
                {
                    if (testContext.TargetProgramId != string.Empty)
                    {
                        testContext.TestName = testContext.TargetProgramId;
                    }
                    else
                    {
                        var targetPath = testContext.TargetPath;

                        testContext.TestName = Path.GetFileName(targetPath);
                    }
                }

                testContext.Path = GetTestResultFilePath(
                    testContext.ResultDirectory,
                    testContext.RoleType,
                    testContext.TestId,
                    testContext.UnitId,
                    testContext.TestName.Replace(' ', '_'));

                testContext.LogPath = Utility.RemoveExtension(
                    testContext.Path, ExtensionDefinition.TestResult) +
                    ExtensionDefinition.Log;

                if (testContext.ReportPath != string.Empty)
                {
                    var reportPath = testContext.ReportPath;

                    reportPath = variableManager.ExpandVariables(reportPath);

                    reportPath = Path.Combine(
                        testContext.WorkingDirectory, reportPath);

                    testContext.ReportPath = Path.GetFullPath(reportPath);
                }
            }
            finally
            {
                variableManager.RemoveResultDirectory();
            }
        }

        private static void UpdateSecondaryPropertyOf(
            TestContext testContext, VariableManager variableManager)
        {
            try
            {
                string resultDirectory = testContext.ResultDirectory;

                variableManager.AddResultDirectory(resultDirectory);

                variableManager.AddPlatfrom(testContext.Platform);

                variableManager.AddBuildType(testContext.BuildType);

                variableManager.AddLogPath(testContext.LogPath);

                variableManager.AddReportPath(testContext.ReportPath);

                testContext.Option = variableManager
                    .ExpandVariables(testContext.Option);

                testContext.Parameter = variableManager
                    .ExpandVariables(testContext.Parameter);

                testContext.Resources = testContext.Resources
                    .Select(x => variableManager.ExpandVariables(x))
                    .Select(x => GetFullPath(x, testContext)).ToArray();

                if (testContext.FailurePatterns != null)
                {
                    testContext.FailurePatterns = testContext.FailurePatterns
                        .Select(x => variableManager.ExpandVariables(x))
                        .ToArray();
                }

                string runSettings = testContext.RunSettings;

                if (!string.IsNullOrEmpty(runSettings))
                {
                    var xml = new XmlDocument();

                    xml.LoadXml(runSettings);

                    XmlNode root = xml.DocumentElement;

                    foreach (XmlNode node in root.SelectNodes("//Parameter"))
                    {
                        XmlAttribute attribute = node.Attributes["value"];

                        string value = attribute.Value;

                        value = variableManager.ExpandVariables(value);

                        attribute.Value = value;
                    }

                    testContext.RunSettings = xml.OuterXml;
                }

                testContext.PostProcessorCommand = variableManager
                    .ExpandVariables(testContext.PostProcessorCommand);

                testContext.PostProcessorOption = variableManager
                    .ExpandVariables(testContext.PostProcessorOption);
            }
            finally
            {
                variableManager.RemoveResultDirectory();

                variableManager.RemovePlatform();

                variableManager.RemoveBuildType();

                variableManager.RemoveLogPath();

                variableManager.RemoveReportPath();
            }
        }

        private static void ExpandTestContextsTargets(
            IReadOnlyList<TestContext[]> tests, TestContext[] observers,
            VariableManager variableManager, ITargetAssigner assigner)
        {
            foreach (var pair in assigner.AllocationTable)
            {
                variableManager.AddTarget(
                    TestUnitInfo.DecodeRoleType(pair.Key),
                    TestUnitInfo.DecodeUnitId(pair.Key),
                    pair.Value);
            }

            foreach (var testContexts in tests.Concat(new[] { observers }))
            {
                foreach (var testContext in testContexts)
                {
                    UpdateSecondaryPropertyOf(testContext, variableManager);
                }
            }

            foreach (var pair in assigner.AllocationTable)
            {
                variableManager.RemoveTarget(
                    TestUnitInfo.DecodeRoleType(pair.Key),
                    TestUnitInfo.DecodeUnitId(pair.Key));
            }
        }

        private void WriteListResult(ListContext context)
        {
            var filePath =
                GetListResultFilePath(
                    Path.GetDirectoryName(context.Path),
                    Path.GetFileName(context.ListName));

            File.AppendAllText(
                filePath, ObjectSerializer.Serialize(context), Encoding.UTF8);

            this.Summarizer.WriteListResult(filePath);
        }

        private void WriteTestResult(TestContext context)
        {
            var dirPath = Path.GetDirectoryName(context.Path);

            var filePath = GetTestResultFilePath(
                dirPath,
                context.RoleType,
                context.TestId,
                context.UnitId,
                context.TestName.Replace(' ', '_'));

            File.AppendAllText(
                filePath, ObjectSerializer.Serialize(context), Encoding.UTF8);

            this.Summarizer.WriteTestResult(filePath);
        }

        private void EvaluateTestList(uint fileCount, string filePath)
        {
            var fileName = Path.GetFileName(filePath);

            var baseListContext = new ListContext()
            {
                Path = GetListResultFilePath(
                    GetBaseResultDirectoryPath(
                        this.ResultRootPath, fileCount, fileName), fileName),
                ListName = GetListNameOfFilePath(filePath, this.TestRootPath),
                ListId = fileCount,
                StatementCount = TestStatementInfo.DefaultStatementId,
                StatementId = TestStatementInfo.DefaultStatementId,
                PlatformCount = (uint)this.Option.Platforms.Count,
                PlatformId = TestStatementInfo.DefaultPlatformId,
                PlatformName = string.Empty,
                BuildTypeCount = (uint)this.Option.BuildTypes.Count,
                BuildTypeId = TestStatementInfo.DefaultBuildTypeId,
                BuildTypeName = string.Empty,
                ResultCode = ResultCode.PASS,
                ErrorMessage = string.Empty
            };

            IReadOnlyList<object> documents = new List<object>();

            try
            {
                documents = YamlParser.Read(filePath);
            }
            catch (Exception ex)
            {
                baseListContext.ResultCode = ResultCode.WRONG_YAML;

                baseListContext.ErrorMessage = ex.Message;

                throw new ListContextException(
                    ex.Message, baseListContext, ex);
            }

            if (documents.Count == 0)
            {
                baseListContext.ResultCode = ResultCode.NO_ENTRIES;

                throw new ListContextException(
                    "No test entries found", baseListContext);
            }

            baseListContext.StatementCount = (uint)documents.Count;

            var baseStatementInfo = new TestStatementInfo(documents.First())
            {
                WorkingDirectory = Path.GetDirectoryName(filePath),
            };

            var statementInfos = this
                .ProduceTestStatementInfos(baseStatementInfo, documents)
                .Select(x => x.UpdateResultDirectory(
                                 this.ResultRootPath, fileCount, fileName))
                .ToList<TestStatementInfo>();

            var statementContexts = statementInfos
                .Select(x => x.GetListContext(baseListContext))
                .ToList<ListContext>();

            var statementTuples = statementInfos
                .Zip(statementContexts, Tuple.Create)
                .ToList<Tuple<TestStatementInfo, ListContext>>();

            foreach (var statementTuple in statementTuples)
            {
                this.CheckTestStatement(
                    statementTuple.Item1, statementTuple.Item2);
            }

            foreach (var statementTuple in statementTuples)
            {
                try
                {
                    Directory.CreateDirectory(
                        Path.GetDirectoryName(statementTuple.Item2.Path));

                    this.EvaluateTestStatement(
                        statementTuple.Item1, statementTuple.Item2);
                }
                catch (ListContextException ex)
                {
                    this.WriteListResult(ex.ListContext);
                }
            }
        }

        private IReadOnlyList<TestStatementInfo> ProduceTestStatementInfos(
            TestStatementInfo baseStatementInfo,
            IReadOnlyList<object> documents)
        {
            var statementInfos = new List<TestStatementInfo>();

            IReadOnlyList<TestStatementInfo>
                genericStatementInfos = baseStatementInfo
                    .ProduceTestStatementInfosByDocuments(documents);

            foreach (var genericStatementInfo in genericStatementInfos)
            {
                if (genericStatementInfo.IsWrongYaml())
                {
                    statementInfos.Add(genericStatementInfo);

                    break;
                }
                else
                {
                    if ( this.Option.Modules.Count > 0 &&
                        !this.Option.Modules.Any(
                             x => genericStatementInfo.Modules.Contains(x)))
                    {
                            continue;
                    }

                    if ( this.Option.Categories.Count > 0 &&
                        !this.Option.Categories.Any(
                            x => genericStatementInfo.Categories.Contains(x)))
                    {
                        continue;
                    }

                    statementInfos.AddRange(genericStatementInfo
                        .ProduceTestStatementInfosByPlatforms(
                            this.Option.Platforms)
                        .SelectMany(x => x
                            .ProduceTestStatementInfosByBuildTypes(
                                this.Option.BuildTypes)));
                }
            }

            return statementInfos;
        }

        private void CheckTestStatement(
            TestStatementInfo statementInfo, ListContext statementContext)
        {
            var errorMessage = string.Empty;

            if (statementInfo.IsWrongYaml(out errorMessage))
            {
                statementContext.ResultCode = ResultCode.WRONG_YAML;

                statementContext.ErrorMessage = errorMessage;

                throw new ListContextException(errorMessage, statementContext);
            }

            this.CheckObserverUnitInfos(statementInfo, statementContext);

            this.CheckTestNodeInfos(statementInfo, statementContext);
        }

        private void CheckObserverUnitInfos(
            TestStatementInfo statementInfo, ListContext statementContext)
        {
            var errorMessage = string.Empty;

            var unitInfos = statementInfo.GetObserverUnitInfos();

            foreach (var unitInfo in unitInfos)
            {
                if (unitInfo.IsWrongYaml(out errorMessage))
                {
                    statementContext.ResultCode = ResultCode.WRONG_YAML;

                    statementContext.ErrorMessage = string.Format(
                        "{0} (Observer{1})",
                        errorMessage,
                        unitInfos.Count == 1
                            ? string.Empty
                            : string.Format(", UnitId: {0}", unitInfo.UnitId));

                    throw new ListContextException(
                        statementContext.ErrorMessage, statementContext);
                }
            }

            foreach (var unitInfo in unitInfos)
            {
                if (unitInfo.IsNoPath())
                {
                    statementContext.ResultCode = ResultCode.NO_PATH;

                    statementContext.ErrorMessage = string.Format(
                        "{0} (Observer{1})",
                        "Target path is not specified",
                        unitInfos.Count == 1
                            ? string.Empty
                            : string.Format(", UnitId: {0}", unitInfo.UnitId));

                    throw new ListContextException(
                        statementContext.ErrorMessage, statementContext);
                }
            }

            var testContexts =
                unitInfos.Select(x => x.GetTestContext()).ToArray();

            var tuples =
                unitInfos.Zip(testContexts, (x, y) => Tuple.Create(x, y));

            try
            {
                foreach (var tuple in tuples)
                {
                    UpdatePrimaryPropertyOf(
                        tuple.Item2, tuple.Item1.VariableManager,
                        this.EpiManager);

                    this.CheckTestContext(tuple.Item2);
                }
            }
            catch (TestContextException ex)
            {
                var testContext = ex.TestContext;

                statementContext.ResultCode = testContext.ResultCode;

                statementContext.ErrorMessage = string.Format(
                    "{0} (Observer{1})",
                    ex.Message,
                    testContext.UnitCount == 1
                        ? string.Empty
                        : string.Format(", UnitId: {0}", testContext.UnitId));

                throw new ListContextException(
                    statementContext.ErrorMessage, statementContext, ex);
            }
        }

        private void CheckTestNodeInfos(
            TestStatementInfo statementInfo, ListContext statementContext)
        {
            var errorMessage = string.Empty;

            var observerCount = (uint)statementInfo.GetObserverUnitInfos().Count;

            foreach (var nodeInfo in statementInfo.GetTestNodeInfos())
            {
                if (nodeInfo.IsWrongYaml(out errorMessage))
                {
                    statementContext.ResultCode = ResultCode.WRONG_YAML;

                    statementContext.ErrorMessage = string.Format(
                        "{0} (TestId: {1})", errorMessage, nodeInfo.TestId);

                    throw new ListContextException(
                        statementContext.ErrorMessage, statementContext);
                }

                if (nodeInfo.IsNoPath())
                {
                    statementContext.ResultCode = ResultCode.NO_PATH;

                    statementContext.ErrorMessage = string.Format(
                        "{0} (TestId: {1})",
                        "Target path is not specified", nodeInfo.TestId);

                    throw new ListContextException(
                        statementContext.ErrorMessage, statementContext);
                }

                try
                {
                    var testContexts = nodeInfo.GetTestContexts();

                    foreach (var testContext in testContexts)
                    {
                        UpdatePrimaryPropertyOf(
                            testContext, nodeInfo.VariableManager,
                            this.EpiManager);

                        this.CheckTestContext(testContext);
                    }

                    this.CheckJobCount(testContexts[0], (uint)observerCount);
                }
                catch (TestContextException ex)
                {
                    var testContext = ex.TestContext;

                    statementContext.ResultCode = testContext.ResultCode;

                    statementContext.ErrorMessage = string.Format(
                        "{0} (TestId: {1}{2})",
                        ex.Message,
                        testContext.TestId,
                        testContext.UnitCount == 1
                            ? string.Empty
                            : string.Format(
                                ", UnitId: {0}", testContext.UnitId));

                    throw new ListContextException(
                        statementContext.ErrorMessage, statementContext, ex);
                }
            }
        }

        private void CheckJobCount(TestContext testContext, uint obseverCount)
        {
            if (!IsParallelModeEnabled(this.Option))
            {
                return;
            }

            var jobCount = this.Option.JobCount;

            if (testContext.UnitCount + obseverCount > jobCount)
            {
                testContext.ResultCode = ResultCode.OVER_LIMIT;

                throw new TestContextException(
                    string.Format("Over the number of jobs '{0}'", jobCount),
                    testContext);
            }
        }

        private void CheckTestContext(TestContext testContext)
        {
            if (this.Generator.IsWrongType(testContext))
            {
                testContext.ResultCode = ResultCode.WRONG_TYPE;

                throw new TestContextException(
                    string.Format(
                        "File '{0}' is not supported",
                        Path.GetFileName(testContext.TargetPath)),
                    testContext);
            }

            if (this.Generator.IsNoFile(testContext))
            {
                testContext.ResultCode = ResultCode.NO_FILE;

                throw new TestContextException(
                    string.Format(
                        "File '{0}' not found",
                        Path.GetFileName(testContext.TargetPath)),
                    testContext);
            }
        }

        private void EvaluateTestStatement(
            TestStatementInfo statementInfo, ListContext statementContext)
        {
            if (statementInfo.IsNoEntries())
            {
                statementContext.ResultCode = ResultCode.NO_ENTRIES;

                throw new ListContextException(
                    "No test entries found", statementContext);
            }

            IReadOnlyList<TestContext[]> tests = null;

            TestContext[] observers = null;

            try
            {
                tests = this.GetTestContextsOfTests(statementInfo);

                observers = this.GetTestContextsOfObservers(statementInfo);

                ITargetAssigner assigner = CreateTargetAssigner(this);

                foreach (var testContexts in tests)
                {
                    assigner.AddTests(testContexts);
                }

                assigner.SetObservers(observers);

                assigner.Flush();

                ExpandTestContextsTargets(
                    tests, observers, statementInfo.VariableManager, assigner);
            }
            catch (TestContextException ex)
            {
                TestContext testContext = ex.TestContext;

                statementContext.ResultCode = testContext.ResultCode;

                var testId = string.Empty;

                switch (testContext.RoleType)
                {
                    case RoleType.Test:
                        testId = $"TestId: {testContext.TestId}";
                        break;

                    case RoleType.Observer:
                        testId = "Observer";
                        break;
                }

                statementContext.ErrorMessage = string.Format(
                    $"{ex.Message} ({testId}{{0}})",
                    testContext.UnitCount == 1
                        ? string.Empty
                        : $", UnitId: {testContext.UnitId}");

                throw new ListContextException(
                    statementContext.ErrorMessage, statementContext, ex);
            }

            if (!this.Option.EnablesFileListingMode)
            {
                this.WriteOuterFile(tests, observers);
            }

            this.WriteListResult(statementContext);

            if (this.Option.EnablesFileListingMode)
            {
                this.SkipTestContexts(tests, observers);
            }
            else if (!this.Option.EnablesParallelMode)
            {
                this.RunTestContexts(tests, observers);
            }
        }

        private IReadOnlyList<TestContext[]> GetTestContextsOfTests(
            TestStatementInfo statementInfo)
        {
            var tests = new List<TestContext[]>();

            var dependencies = new string[0];

            foreach (var nodeInfo in statementInfo.GetExpandedTestNodeInfos())
            {
                var testContexts = nodeInfo.GetTestContexts();

                foreach (var testContext in testContexts)
                {
                    UpdatePrimaryPropertyOf(
                        testContext, nodeInfo.VariableManager,
                        this.EpiManager);

                    if (testContext.BreakLevel != BreakLevel.None)
                    {
                        testContext.Dependencies = dependencies;
                    }
                }

                dependencies = testContexts.Select(x => x.Path).ToArray();

                foreach (var testContext in testContexts)
                {
                    this.ApplyRuleToTestContext(testContext);
                }

                tests.Add(testContexts);
            }

            return tests;
        }

        private TestContext[] GetTestContextsOfObservers(
            TestStatementInfo statementInfo)
        {
            var testContexts = new List<TestContext>();

            var unitInfos = statementInfo.GetObserverUnitInfos();

            foreach (var unitInfo in unitInfos)
            {
                var testContext = unitInfo.GetTestContext();

                UpdatePrimaryPropertyOf(
                    testContext, unitInfo.VariableManager, this.EpiManager);

                testContexts.Add(testContext);
            }

            foreach (var testContext in testContexts)
            {
                this.ApplyRuleToTestContext(testContext);
            }

            return testContexts.ToArray();
        }

        private void ApplyRuleToTestContext(TestContext testContext)
        {
            var inst = this.Generator.Generate(testContext);

            testContext.Command = inst.Command;

            testContext.Option = inst.Option;

            testContext.FailurePatterns = inst.FailurePatterns;

            testContext.RunSettings = inst.RunSettings;

            testContext.Platform = inst.Platform;

            testContext.BuildType = inst.BuildType;

            testContext.TargetPath = inst.TargetPath;

            testContext.TargetProjectPath = inst.TargetProjectPath;

            testContext.ReportPath = inst.ReportPath;

            testContext.CsxPath = inst.CsxPath;

            testContext.RunSettingsPath = inst.RunSettingsPath;

            testContext.DumpFileType = inst.DumpFileType;

            testContext.DumpFilePath = inst.DumpFilePath;
        }

        private void WriteOuterFile(
            IReadOnlyList<TestContext[]> tests, TestContext[] observers)
        {
            var nspRule = new NspFileRule();

            foreach (var testContexts in tests.Concat(new[] { observers }))
            {
                foreach (var testContext in testContexts)
                {
                    if (!string.IsNullOrEmpty(testContext.CsxPath))
                    {
                        if (nspRule.Supports(testContext))
                        {
                            File.WriteAllText(
                                testContext.CsxPath,
                                nspRule.GetCsxContent(testContext),
                                Encoding.UTF8);
                        }
                    }

                    if (!string.IsNullOrEmpty(testContext.RunSettingsPath))
                    {
                        var xml = new XmlDocument();

                        xml.LoadXml(testContext.RunSettings);

                        xml.Save(testContext.RunSettingsPath);
                    }
                }
            }
        }

        private void SkipTestContexts(
            IReadOnlyList<TestContext[]> tests, TestContext[] observers)
        {
            foreach (var contexts in tests.Concat(new[] { observers }))
            {
                foreach (var context in contexts)
                {
                    context.ResultCode = ResultCode.SKIP;

                    File.AppendAllText(
                        context.Path,
                        ObjectSerializer.Serialize(context),
                        Encoding.UTF8);

                    this.Summarizer.WriteTestResult(context.Path);
                }
            }
        }

        private void RunTestContexts(
            IReadOnlyList<TestContext[]> tests, TestContext[] observers)
        {
            TestExecuter[] executers = null;

            try
            {
                executers = TestExecuter.Start(
                    observers, this.Option.EnablesVerboseMode);
            }
            catch (TestContextException ex)
            {
                File.AppendAllText(
                    ex.TestContext.Path,
                    ObjectSerializer.Serialize(ex.TestContext),
                    Encoding.UTF8);
            }

            foreach (var testContexts in tests)
            {
                try
                {
                    TestExecuter.WaitForExit(TestExecuter.Start(
                        testContexts, this.Option.EnablesVerboseMode));
                }
                catch (TestContextException ex)
                {
                    File.AppendAllText(
                        ex.TestContext.Path,
                        ObjectSerializer.Serialize(ex.TestContext),
                        Encoding.UTF8);
                }

                foreach (var testContext in testContexts)
                {
                    if (File.Exists(testContext.Path))
                    {
                        this.Summarizer.WriteTestResult(testContext.Path);
                    }
                }
            }

            if (executers != null)
            {
                TestExecuter.WaitForExit(executers);
            }

            foreach (var testContext in observers)
            {
                if (File.Exists(testContext.Path))
                {
                    this.Summarizer.WriteTestResult(testContext.Path);
                }
            }
        }

        private interface ITargetAssigner
        {
            IReadOnlyDictionary<ulong, TargetEntry> AllocationTable { get; }

            void AddTests(TestContext[] testContexts);

            void SetObservers(TestContext[] testContexts);

            void Flush();
        }

        private sealed class IdleTargetAssigner : ITargetAssigner
        {
            private readonly TestRunner runner;

            private readonly Dictionary<ulong, TargetEntry> map;

            internal IdleTargetAssigner(TestRunner runner)
            {
                this.runner = runner;

                this.map = new Dictionary<ulong, TargetEntry>();
            }

            public IReadOnlyDictionary<ulong, TargetEntry> AllocationTable
            {
                get
                {
                    return new ReadOnlyDictionary<ulong, TargetEntry>(
                        this.map);
                }
            }

            public void AddTests(TestContext[] testContexts)
            {
                // 何もしない
            }

            public void SetObservers(TestContext[] testContexts)
            {
                // 何もしない
            }

            public void Flush()
            {
                // 何もしない
            }
        }

        private sealed class SerialTargetAssigner : ITargetAssigner
        {
            private readonly TestRunner runner;

            private readonly TargetAssigner targetAssigner;

            internal SerialTargetAssigner(TestRunner runner)
            {
                this.runner = runner;

                this.targetAssigner = new TargetAssigner();
            }

            public IReadOnlyDictionary<ulong, TargetEntry> AllocationTable
            {
                get
                {
                    return this.targetAssigner.AllocationTable;
                }
            }

            public void AddTests(TestContext[] testContexts)
            {
                this.targetAssigner.AddTests(testContexts);
            }

            public void SetObservers(TestContext[] testContexts)
            {
                this.targetAssigner.SetObservers(testContexts);
            }

            public void Flush()
            {
                this.targetAssigner.Flush(
                    this.runner.TargetManager.TargetEntries);
            }
        }

        private sealed class ParallelTargetAssigner : ITargetAssigner
        {
            private readonly TestRunner runner;

            private Dictionary<ulong, TargetEntry> map;

            internal ParallelTargetAssigner(TestRunner runner)
            {
                this.runner = runner;

                this.map = new Dictionary<ulong, TargetEntry>();
            }

            public IReadOnlyDictionary<ulong, TargetEntry> AllocationTable
            {
                get
                {
                    return new ReadOnlyDictionary<ulong, TargetEntry>(
                        this.map);
                }
            }

            public void AddTests(TestContext[] testContexts)
            {
                this.runner.BuildRuleMaker.AddTests(testContexts);
            }

            public void SetObservers(TestContext[] testContexts)
            {
                this.runner.BuildRuleMaker.SetObservers(testContexts);
            }

            public void Flush()
            {
                this.map = this.runner.BuildRuleMaker.Flush(true);
            }
        }
    }
}
