﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Text;

readonly var RunOnTargetPath = @"/* RunOnTargetPath */";
readonly var ControlTargetPath = @"/* ControlTargetPath */";
readonly var ReadSerialPath = @"/* ReadSerialPath */";
readonly var DumpFilePath = @"/* DumpFilePath */";
readonly var ExecutableFilePath = @"/* ExecutableFilePath */";
readonly var ExecutableFileOption = string.Join(" ", Args);
readonly var FailureTimeout = "/* FailureTimeout */";
readonly var FailurePatterns = new string[] { /* FailurePatterns */ };

readonly var TargetName =
    Environment.GetEnvironmentVariable("NNTEST_TARGET_NAME");
readonly var TargetInterface =
    Environment.GetEnvironmentVariable("NNTEST_TARGET_INTERFACE");
readonly var TargetAddress =
    Environment.GetEnvironmentVariable("NNTEST_TARGET_ADDRESS");
readonly var TargetSerialPort =
    Environment.GetEnvironmentVariable($"{TargetName}_SERIAL_PORT");
readonly var TargetPhidgetsSerialNumber =
    Environment.GetEnvironmentVariable($"{TargetName}_PHIDGETS_SN");
readonly var RequiresTargetReset =
    Environment.GetEnvironmentVariable("REQUIRE_TARGET_RESET") == "true";

var isTimeoutDetected = false;

void AppendFailurePatterns(List<string> arguments)
{
    foreach (var pattern in FailurePatterns)
    {
        arguments.Add($"--pattern-failure-exit \"{pattern}\"");
    }
}

ProcessStartInfo MakeDefaultStartInfo()
{
    return new ProcessStartInfo
    {
        CreateNoWindow = true,
        RedirectStandardError = true,
        RedirectStandardOutput = true,
        UseShellExecute = false,
    };
}

readonly var ConsoleSyncObject = new object();

void OnDataReceived(object sender, DataReceivedEventArgs e)
{
    if (e.Data == null)
    {
        return;
    }

    lock (ConsoleSyncObject)
    {
        if (e.Data.StartsWith("[trace] Failure timeout."))
        {
            isTimeoutDetected = true;
        }

        Console.WriteLine(e.Data);
    }
}

Process readSerial = null;

if (!string.IsNullOrEmpty(ReadSerialPath) &&
    !string.IsNullOrEmpty(TargetSerialPort))
{
    var readSerialArguments = new List<string>();
    readSerialArguments.Add("--verbose");
    readSerialArguments.Add($"--port {TargetSerialPort}");
    readSerialArguments.Add("--pattern-success-exit \\bKProcess::Finalize");

    if (!string.IsNullOrEmpty(FailureTimeout))
    {
        readSerialArguments.Add($"--failure-timeout {FailureTimeout}");
    }

    AppendFailurePatterns(readSerialArguments);

    readSerial = new Process();
    readSerial.StartInfo = MakeDefaultStartInfo();
    readSerial.StartInfo.FileName = ReadSerialPath;
    readSerial.StartInfo.Arguments = string.Join(" ", readSerialArguments);
    readSerial.ErrorDataReceived += OnDataReceived;
    readSerial.OutputDataReceived += OnDataReceived;

    readSerial.Start();
    readSerial.BeginOutputReadLine();
    readSerial.BeginErrorReadLine();
}

var runOnTargetArguments = new List<string>();

var isPrivateMode = RunOnTargetPath.EndsWith("RunOnTargetPrivate.exe");

if (isPrivateMode)
{
    runOnTargetArguments.Add("run");
}

runOnTargetArguments.Add($"\"{ExecutableFilePath}\"");
runOnTargetArguments.Add("--verbose");
runOnTargetArguments.Add($"--target \"{TargetName}\"");

if (!string.IsNullOrEmpty(FailureTimeout))
{
    runOnTargetArguments.Add($"--failure-timeout {FailureTimeout}");
}

AppendFailurePatterns(runOnTargetArguments);

if (isPrivateMode && TargetInterface == "Ethernet")
{
    runOnTargetArguments.Add("--monitor-serial");
    runOnTargetArguments.Add($"--hostbridge {TargetAddress}");
}

if (!string.IsNullOrEmpty(ExecutableFileOption))
{
    runOnTargetArguments.Add("\"--\"");
    runOnTargetArguments.Add(ExecutableFileOption);
}

var runOnTarget = new Process();
runOnTarget.StartInfo = MakeDefaultStartInfo();
runOnTarget.StartInfo.FileName = RunOnTargetPath;
runOnTarget.StartInfo.Arguments = string.Join(" ", runOnTargetArguments);
runOnTarget.ErrorDataReceived += OnDataReceived;
runOnTarget.OutputDataReceived += OnDataReceived;

runOnTarget.Start();
runOnTarget.BeginOutputReadLine();
runOnTarget.BeginErrorReadLine();
runOnTarget.WaitForExit();

if (isTimeoutDetected)
{
    var controlTarget = new Process();
    controlTarget.StartInfo = MakeDefaultStartInfo();
    controlTarget.StartInfo.FileName = ControlTargetPath;
    controlTarget.StartInfo.Arguments = string.Join(" ", new[] {
        "create-target-program-dump",
        $"-t \"{TargetName}\"",
        $"\"{DumpFilePath}\"",
        "full",
    });
    controlTarget.ErrorDataReceived += OnDataReceived;
    controlTarget.OutputDataReceived += OnDataReceived;

    controlTarget.Start();
    controlTarget.BeginOutputReadLine();
    controlTarget.BeginErrorReadLine();
    controlTarget.WaitForExit();

    if (controlTarget.ExitCode != 0)
    {
        Console.Error.WriteLine(
            "[TestRunner] Error: failed to create the nxdmp.");
    }
}

int exitCode = runOnTarget.ExitCode;

if (readSerial != null)
{
    try
    {
        if (!readSerial.HasExited)
        {
            readSerial.Kill();
        }
        else
        {
            if (exitCode == 0)
            {
                exitCode = readSerial.ExitCode;
            }
        }
    }
    catch
    {
        // 何もしない
    }

    try
    {
        readSerial.WaitForExit();
    }
    catch
    {
        // 何もしない
    }
}

if (exitCode != 0 || RequiresTargetReset)
{
    if (string.IsNullOrEmpty(TargetPhidgetsSerialNumber))
    {
        var controlTarget = new Process();
        controlTarget.StartInfo = MakeDefaultStartInfo();
        controlTarget.StartInfo.FileName = ControlTargetPath;
        controlTarget.StartInfo.Arguments = string.Join(" ", new[] {
            "connect", "--reset", $"-t \"{TargetName}\""
        });
        controlTarget.ErrorDataReceived += OnDataReceived;
        controlTarget.OutputDataReceived += OnDataReceived;

        controlTarget.Start();
        controlTarget.BeginOutputReadLine();
        controlTarget.BeginErrorReadLine();
        controlTarget.WaitForExit();

        if (controlTarget.ExitCode != 0)
        {
            Console.Error.WriteLine(
                "[TestRunner] Error: failed to reset the target.");

            exitCode = (exitCode != 0) ? exitCode : controlTarget.ExitCode;
        }
    }
    else
    {
        var resetable = true;

        {
            var controlTarget = new Process();
            controlTarget.StartInfo = MakeDefaultStartInfo();
            controlTarget.StartInfo.FileName = ControlTargetPath;
            controlTarget.StartInfo.Arguments = string.Join(" ", new[] {
                "power-off-phidgets",
                $"--phidgets-serial \"{TargetPhidgetsSerialNumber}\""
            });
            controlTarget.ErrorDataReceived += OnDataReceived;
            controlTarget.OutputDataReceived += OnDataReceived;

            controlTarget.Start();
            controlTarget.BeginOutputReadLine();
            controlTarget.BeginErrorReadLine();
            controlTarget.WaitForExit();

            if (controlTarget.ExitCode != 0)
            {
                Console.Error.WriteLine(
                    "[TestRunner] Error: failed to power off the target.");

                exitCode = (exitCode != 0) ? exitCode : controlTarget.ExitCode;

                resetable = false;
            }
        }

        if (resetable)
        {
            var controlTarget = new Process();
            controlTarget.StartInfo = MakeDefaultStartInfo();
            controlTarget.StartInfo.FileName = ControlTargetPath;
            controlTarget.StartInfo.Arguments = string.Join(" ", new[] {
                "power-on-phidgets",
                $"--phidgets-serial \"{TargetPhidgetsSerialNumber}\""
            });
            controlTarget.ErrorDataReceived += OnDataReceived;
            controlTarget.OutputDataReceived += OnDataReceived;

            controlTarget.Start();
            controlTarget.BeginOutputReadLine();
            controlTarget.BeginErrorReadLine();
            controlTarget.WaitForExit();

            if (controlTarget.ExitCode != 0)
            {
                Console.Error.WriteLine(
                    "[TestRunner] Error: failed to power on the target.");

                exitCode = (exitCode != 0) ? exitCode : controlTarget.ExitCode;

                resetable = false;
            }
        }

        if (resetable)
        {
            var controlTarget = new Process();
            controlTarget.StartInfo = MakeDefaultStartInfo();
            controlTarget.StartInfo.FileName = ControlTargetPath;
            controlTarget.StartInfo.Arguments = string.Join(" ", new[] {
                "connect", $"-t \"{TargetName}\""
            });
            controlTarget.ErrorDataReceived += OnDataReceived;
            controlTarget.OutputDataReceived += OnDataReceived;

            controlTarget.Start();
            controlTarget.BeginOutputReadLine();
            controlTarget.BeginErrorReadLine();
            controlTarget.WaitForExit();

            if (controlTarget.ExitCode != 0)
            {
                Console.Error.WriteLine(
                    "[TestRunner] Error: failed to connect the target.");

                exitCode = (exitCode != 0) ? exitCode : controlTarget.ExitCode;
            }
        }
    }
}

Environment.Exit(exitCode);
