﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace TestRunner.GenerationRule
{
    using System;
    using System.Linq;
    using Executer;
    using Properties;

    /// <summary>
    /// nca ファイル用のインストラクション生成ルールです。
    /// </summary>
    internal sealed class NspFileRule : InstructionGenerator.IRule
    {
        private static readonly string[] extensions = new[] {
            ".kip",
            ".nca",
            ".nsp",
            ".nspd_root"
        };

        /// <summary>
        /// サポート対象のテストコンテキストかどうかを示す値を取得します。
        /// </summary>
        /// <param name="context">テストコンテキストです。</param>
        /// <returns>サポート対象のテストコンテキストかどうかを示す値です。</returns>
        public bool Supports(TestContext context)
        {
            if (context.TargetProgramId != string.Empty)
            {
                return true;
            }

            return extensions.Any(x => context.TargetPath.EndsWith(x));
        }

        /// <summary>
        /// ルール適用時にターゲットファイルを必要とするかどうかを示す値を取得します。
        /// </summary>
        /// <returns>ルール適用時にターゲットファイルを必要とするかどうかを示す値です。</returns>
        public bool NeedsTargetFile()
        {
            return false;
        }

        /// <summary>
        /// テストコンテキストにインストラクション生成ルールを適用します。
        /// </summary>
        /// <param name="context">テストコンテキストです。</param>
        /// <returns>生成されたテストインストラクションです。</returns>
        public InstructionGenerator.Instruction Generate(TestContext context)
        {
            var inst = new InstructionGenerator.Instruction();

            inst.Command = UnwrapToolPath(context, () => VisualStudio.CsiPath);

            var csxFilePath = Utility.RemoveExtension(
                context.Path, ExtensionDefinition.TestResult) + ".csx";

            inst.Option = string.Join(" ", new[] {
                $"\"{csxFilePath}\"",
                $"{context.Option.Replace("\"", "\\\"")}"
            });

            inst.FailurePatterns = context.FailurePatterns ?? new[]
            {
                "Assertion Failure:",
                "Precondition not met:",
                "Unexpected Default:",
                "Abort:",
                "UserException handler is called at pid=",
                "Break\\(\\) called\\."
            };

            inst.Platform = !string.IsNullOrEmpty(context.Platform)
                ? context.Platform
                : BuildSystem.DefaultPlatform;

            inst.BuildType = !string.IsNullOrEmpty(context.BuildType)
                ? context.BuildType
                : BuildSystem.DefaultBuildType;

            inst.TargetPath = context.TargetPath;

            inst.TargetProjectPath = context.TargetProjectPath;

            inst.ReportPath = context.ReportPath;

            inst.CsxPath = csxFilePath;

            inst.DumpFileType = DumpFileTypeDefinition.Nxdmp;

            inst.DumpFilePath = Utility.RemoveExtension(
                context.Path, ExtensionDefinition.TestResult
                ) + DumpFileExtensionDefinition.Nxdmp;

            return inst;
        }

        /// <summary>
        /// 実行するプログラムに渡す .csx を取得します。
        /// </summary>
        /// <param name="context">テストコンテキストです。</param>
        /// <returns>実行するプログラムに渡す .csx です。</returns>
        internal string GetCsxContent(TestContext context)
        {
            string targetPath = context.TargetProgramId;

            if (targetPath == string.Empty)
            {
                targetPath = context.TargetPath;

                if (targetPath.EndsWith(".nspd_root"))
                {
                    targetPath = targetPath.Remove(targetPath.Length - 5);
                }
            }

            var failurePattrens = (context.FailurePatterns ?? new string[0])
                .Select(x => x.Replace("\\", "\\\\").Replace("\"", "\\\""))
                .Select(x => $"\"{x}\"")
                .ToArray();

            return Resources.RunOnTarget_csx
                .Replace(
                    "/* RunOnTargetPath */",
                    UnwrapToolPath(
                        context, () => TargetManager.RunOnTargetPath))
                .Replace(
                    "/* ControlTargetPath */",
                    UnwrapToolPath(
                        context, () => TargetManager.ControlTargetPath))
                .Replace(
                    "/* ReadSerialPath */", GetReadSerialPath())
                .Replace(
                    "/* DumpFilePath */", context.DumpFilePath)
                .Replace(
                    "/* ExecutableFilePath */", targetPath)
                .Replace(
                    "/* FailureTimeout */",
                    context.Timeout <= 0 ?
                        string.Empty : context.Timeout.ToString())
                .Replace(
                    "/* FailurePatterns */",
                    string.Join(", ", failurePattrens));
        }

        private static string UnwrapToolPath(TestContext context, Func<string> func)
        {
            try
            {
                return func.Invoke();
            }
            catch (TestRunnerException ex)
            {
                context.ResultCode = ResultCode.NO_TOOL;

                context.ErrorMessage = ex.Message;

                throw new TestContextException(ex.Message, context);
            }
        }

        private static string GetReadSerialPath()
        {
            try
            {
                return TargetManager.ReadSerialPath;
            }
            catch
            {
                return string.Empty;
            }
        }
    }
}
