﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace TestRunner.Executer
{
    using System;
    using System.Diagnostics;
    using System.IO;
    using System.Linq;
    using System.Text;
    using System.Text.RegularExpressions;
    using System.Threading.Tasks;
    using InsertionRule;

    /// <summary>
    /// テストを実行します。
    /// </summary>
    public sealed class TestExecuter
    {
        public const int MinimumTimeout = 0;

        public const int MaximumTimeout = int.MaxValue / 1000;

        private readonly object syncObject = new object();

        /// <summary>
        /// TestExecuter クラスの新しいインスタンスを初期化します。
        /// </summary>
        /// <param name="testContext">テストコンテキストです。</param>
        private TestExecuter(TestContext testContext)
        {
            this.TestContext = testContext;
        }

        private TestContext TestContext { get; set; }

        private bool IsRunning { get; set; } = false;

        private TargetNameInserter TargetNameInserter { get; set; } = null;

        private bool EnablesVerboseMode { get; set; } = false;

        private StreamWriter StreamWriter { get; set; } = null;

        private Process Process { get; set; } = null;

        private ProcessJob ProcessJob { get; set; } = null;

        private DateTime DateTime { get; set; }

        /// <summary>
        /// テストコンテキストに従ってテストを開始します。
        /// </summary>
        /// <param name="testContexts">テストコンテキストの配列です。</param>
        /// <param name="enablesVerboseMode">ログ出力をコンソールに出力するかどうかを示す値を取得します。</param>
        /// <returns>実行中のテストです。</returns>
        public static TestExecuter[] Start(
            TestContext[] testContexts, bool enablesVerboseMode)
        {
            if (testContexts.Length > 0 &&
                testContexts.First().BreakLevel != BreakLevel.None)
            {
                CheckDependencies(testContexts.First());
            }

            foreach (var testContext in testContexts)
            {
                CheckTargetPath(testContext);

                CheckPostProcessorPath(testContext);
            }

            var executers = testContexts
                .Select(testContext => new TestExecuter(testContext))
                .ToArray();

            var inserter = new TargetNameInserter();

            Parallel.ForEach(executers, executer =>
            {
                executer.TargetNameInserter = inserter;

                executer.EnablesVerboseMode = enablesVerboseMode;

                executer.Start();
            });

            return executers;
        }

        /// <summary>
        /// テストの終了を待ちます。
        /// </summary>
        /// <param name="executers">実行中のテストです。</param>
        public static void WaitForExit(TestExecuter[] executers)
        {
            Parallel.ForEach(executers, executer =>
            {
                if (executer.IsRunning &&
                    executer.TestContext.ProcessType != ProcessType.Background)
                {
                    executer.WaitForExit();
                }
            });

            Parallel.ForEach(executers, executer =>
            {
                if (executer.IsRunning &&
                    executer.TestContext.ProcessType == ProcessType.Background)
                {
                    executer.WaitForExit();
                }
            });

            Parallel.ForEach(executers, executer =>
            {
                if (executer.TestContext.ResultCode == ResultCode.PASS &&
                    executer.TestContext.PostProcessorCommand != string.Empty)
                {
                    executer.DoPostProcessing();
                }
            });

            Parallel.ForEach(executers, executer =>
            {
                File.AppendAllText(
                    executer.TestContext.Path,
                    ObjectSerializer.Serialize(executer.TestContext),
                    Encoding.UTF8);
            });
        }

        private static int GetTimeout(TestContext testContext)
        {
            var milliseconds = testContext.Timeout * 1000;

            switch (testContext.DumpFileType)
            {
                case DumpFileTypeDefinition.Nxdmp:
                    milliseconds += 10 * 60 * 1000; // 10 分
                    break;
            }

            return milliseconds;
        }

        private static long GetDuration(DateTime head, DateTime tail)
        {
            TimeSpan duration = tail - head;

            double msecs = duration.TotalMilliseconds;

            return (long)Math.Ceiling(msecs);
        }

        private static string[] GetReportFilePaths(string reportPath)
        {
            var dirName = Path.GetDirectoryName(reportPath);

            var fileName = Path.GetFileName(reportPath);

            return !Directory.Exists(dirName)
                ? new string[0]
                : Directory.GetFiles(dirName, fileName);
        }

        private static void CheckDependencies(TestContext testContext)
        {
            foreach (var dependencyPath in testContext.Dependencies)
            {
                if (!File.Exists(dependencyPath))
                {
                    testContext.ResultCode = ResultCode.SKIP;

                    var message =
                        $"File '{Path.GetFileName(dependencyPath)}' not found";

                    testContext.ErrorMessage = message;

                    throw new TestContextException(message, testContext);
                }

                var dependency = ObjectSerializer.Deserialize<TestContext>(
                    File.ReadAllText(dependencyPath, Encoding.UTF8));

                if ((testContext.BreakLevel == BreakLevel.Error &&
                     dependency.ResultCode.MatchesBreakLevelError()) ||
                    (testContext.BreakLevel == BreakLevel.Timeout &&
                     dependency.ResultCode.MatchesBreakLevelTimeout()) ||
                    (testContext.BreakLevel == BreakLevel.Failure &&
                     dependency.ResultCode.MatchesBreakLevelFailure()))
                {
                    testContext.ResultCode = ResultCode.SKIP;

                    var fileName = Path.GetFileName(dependencyPath);

                    var resultCode = dependency.ResultCode;

                    var message =
                        $"ResultCode of the file '{fileName}' is {resultCode}";

                    testContext.ErrorMessage = message;

                    throw new TestContextException(message, testContext);
                }
            }
        }

        private static void CheckTargetPath(TestContext testContext)
        {
            string filePath = testContext.TargetPath;

            if (filePath != string.Empty && !File.Exists(filePath))
            {
                testContext.ResultCode = ResultCode.NO_FILE;

                var message = $"File '{Path.GetFileName(filePath)}' not found";

                testContext.ErrorMessage = message;

                throw new TestContextException(message, testContext);
            }
        }

        private static void CheckPostProcessorPath(TestContext testContext)
        {
            string filePath = testContext.PostProcessorCommand;

            if (!string.IsNullOrEmpty(filePath) && !File.Exists(filePath))
            {
                testContext.ResultCode = ResultCode.NO_FILE;

                var message = $"File '{Path.GetFileName(filePath)}' not found";

                testContext.ErrorMessage = message;

                throw new TestContextException(message, testContext);
            }
        }

        private static void UpdateDuration(
            TestContext testContext, DateTime head, DateTime tail)
        {
            testContext.Duration = GetDuration(head, tail);

            testContext.OpeningTime = head.ToString("o");

            testContext.ClosingTime = tail.ToString("o");
        }

        private static void UpdateResultCode(
            TestContext testContext, int exitCode, string[] reportFilePaths)
        {
            if (exitCode == 0) { return; }

            if (File.Exists(testContext.DumpFilePath))
            {
                testContext.ResultCode = ResultCode.TIME_OUT;

                return;
            }

            if (reportFilePaths.Length == 0)
            {
                testContext.ResultCode = ResultCode.ERROR;

                return;
            }

            string path = reportFilePaths[0];

            if (path.EndsWith(GoogleTestXmlRule.Extention) && exitCode != 1)
            {
                testContext.ResultCode = ResultCode.ERROR;
            }
            else
            {
                testContext.ResultCode = ResultCode.FAIL;
            }
        }

        private void Start()
        {
            try
            {
                this.StreamWriter = new StreamWriter(
                    this.TestContext.LogPath, false, Encoding.UTF8);

                this.Process = new Process();

                this.Process.StartInfo = new ProcessStartInfo()
                {
                    FileName = this.TestContext.Command,
                    Arguments = this.TestContext.Option,
                    WorkingDirectory = this.TestContext.WorkingDirectory,
                    UseShellExecute = false,
                    CreateNoWindow = true,
                    RedirectStandardOutput = true,
                    RedirectStandardError = true
                };

                this.Process.StartInfo.EnvironmentVariables[
                    "NNTEST_TARGET_NAME"] =
                        this.TestContext.TargetName;

                this.Process.StartInfo.EnvironmentVariables[
                    "NNTEST_TARGET_INTERFACE"] =
                        this.TestContext.TargetInterface;

                this.Process.StartInfo.EnvironmentVariables[
                    "NNTEST_TARGET_ADDRESS"] =
                        this.TestContext.TargetAddress;

                var handler = new DataReceivedEventHandler((obj, args) =>
                {
                    if (args.Data == null)
                    {
                        return;
                    }

                    lock (this.syncObject)
                    {
                        this.StreamWriter.WriteLine(args.Data);

                        if (this.EnablesVerboseMode)
                        {
                            Console.WriteLine(args.Data);
                        }
                    }
                });

                this.Process.OutputDataReceived += handler;

                this.Process.ErrorDataReceived += handler;

                this.DateTime = DateTime.Now;

                this.Process.Start();

                this.IsRunning = true;

                this.ProcessJob = new ProcessJob(this.Process);

                // 互換性を維持するため PowerShell スクリプトからのプロセス起動は見逃す
                if (!Regex.IsMatch(
                        Path.GetFileName(this.TestContext.TargetPath),
                        @"^Start-.+\.ps1$"))
                {
                    this.ProcessJob.SetKillOnJobCloseFlag();
                }

                this.Process.BeginOutputReadLine();

                this.Process.BeginErrorReadLine();
            }
            catch (Exception ex)
            {
                this.TestContext.ResultCode = ResultCode.ERROR;

                this.TestContext.ErrorMessage = ex.Message;

                this.DisposeProcess();
            }
        }

        private void WaitForExit()
        {
            try
            {
                if (this.TestContext.ProcessType != ProcessType.Background)
                {
                    if (this.TestContext.Timeout != 0)
                    {
                        this.Process.WaitForExit(GetTimeout(this.TestContext));
                    }
                    else
                    {
                        this.Process.WaitForExit();
                    }
                }

                if (!this.Process.HasExited)
                {
                    DateTime dateTime = DateTime.Now;

                    UpdateDuration(this.TestContext, this.DateTime, dateTime);

                    if (this.TestContext.ProcessType != ProcessType.Background)
                    {
                        this.TestContext.ResultCode = ResultCode.TIME_OUT;

                        switch (this.TestContext.DumpFileType)
                        {
                            case DumpFileTypeDefinition.MiniDump:
                                MiniDump.Create(
                                    this.TestContext.DumpFilePath,
                                    this.Process);
                                break;
                        }
                    }
                }
                else
                {
                    DateTime dateTime = this.Process.ExitTime;

                    UpdateDuration(this.TestContext, this.DateTime, dateTime);

                    var paths =
                        string.IsNullOrEmpty(this.TestContext.ReportPath)
                            ? new string[0]
                            : GetReportFilePaths(this.TestContext.ReportPath);

                    UpdateResultCode(
                        this.TestContext, this.Process.ExitCode, paths);

                    if (this.TargetNameInserter != null && 0 < paths.Length)
                    {
                        this.TargetNameInserter.Insert(
                            paths[0], this.TestContext.TargetPath);
                    }
                }
            }
            catch (Exception ex)
            {
                this.TestContext.ResultCode = ResultCode.ERROR;

                this.TestContext.ErrorMessage = ex.Message;
            }
            finally
            {
                this.DisposeProcess();
            }
        }

        private void DoPostProcessing()
        {
            try
            {
                var stringBuilder = new StringBuilder();

                this.Process = new Process();

                this.Process.StartInfo = new ProcessStartInfo()
                {
                    FileName = this.TestContext.PostProcessorCommand,
                    Arguments = this.TestContext.PostProcessorOption,
                    WorkingDirectory = this.TestContext.WorkingDirectory,
                    UseShellExecute = false,
                    CreateNoWindow = true,
                    RedirectStandardOutput = true,
                    RedirectStandardError = true
                };

                var handler = new DataReceivedEventHandler((obj, args) =>
                {
                    if (args.Data == null)
                    {
                        return;
                    }

                    lock (this.syncObject)
                    {
                        stringBuilder.AppendLine(args.Data);

                        if (this.EnablesVerboseMode)
                        {
                            Console.WriteLine(args.Data);
                        }
                    }
                });

                this.Process.OutputDataReceived += handler;

                this.Process.ErrorDataReceived += handler;

                this.Process.Start();

                this.IsRunning = true;

                this.Process.BeginOutputReadLine();

                this.Process.BeginErrorReadLine();

                this.Process.WaitForExit();

                var paths = string.IsNullOrEmpty(this.TestContext.ReportPath)
                    ? new string[0]
                    : GetReportFilePaths(this.TestContext.ReportPath);

                UpdateResultCode(
                    this.TestContext, this.Process.ExitCode, paths);

                File.AppendAllText(
                    this.TestContext.LogPath, stringBuilder.ToString(),
                    Encoding.UTF8);
            }
            catch (Exception ex)
            {
                this.TestContext.ResultCode = ResultCode.ERROR;

                this.TestContext.ErrorMessage = ex.Message;
            }
            finally
            {
                this.DisposeProcess();
            }
        }

        private void DisposeProcess()
        {
            if (this.Process != null)
            {
                if (this.IsRunning)
                {
                    try
                    {
                        if (this.ProcessJob != null)
                        {
                            this.ProcessJob.Dispose();

                            this.ProcessJob = null;
                        }

                        this.Process.WaitForExit();

                        if (this.Process.HasExited)
                        {
                            this.TestContext.ExitCode = this.Process.ExitCode;
                        }
                    }
                    catch (Exception ex)
                    {
                        if (this.TestContext.ResultCode == ResultCode.PASS)
                        {
                            this.TestContext.ResultCode = ResultCode.ERROR;

                            this.TestContext.ErrorMessage = ex.Message;
                        }
                    }
                }

                this.Process.Dispose();

                this.Process = null;
            }

            if (this.StreamWriter != null)
            {
                this.StreamWriter.Dispose();

                this.StreamWriter = null;
            }

            this.IsRunning = false;
        }
    }
}
