﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace SystemUpdaterNx.Uninstall
{
    using System.ComponentModel.Composition;
    using System.IO;
    using System.Text;
    using System.Threading;
    using SystemUpdaterNx.Common;
    using TargetShell.Library;
    using TargetShell.CommandPluginBase;
    using TargetShell.DevicePluginBase;

    /// <summary>
    /// アンインストールを行うクラス
    /// </summary>
    [Export(typeof(IDevicePluginInterface<CommandParameter>))]
    public class Nx :
            BaseDevicePlugin<CommandParameter>, IDevicePluginInterface<CommandParameter>
    {
        /// <summary>
        /// コンストラクタ
        /// </summary>
        public Nx()
        {
            this.ParallelModeFlag = true;
        }

        /// <summary>
        /// デバイスプラグイン実行メソッド
        /// </summary>
        /// <param name="parameter">コマンドのパラメータ</param>
        /// <param name="successDeviceCount">成功デバイスの数</param>
        protected override void SpecificProcessDevicePlugin(
                CommandParameter parameter, ref int successDeviceCount)
        {
            try
            {
                // コマンドの構築
                var runOnTargetCommand = CreateRunOnTargetCommand(parameter);
                // アンインストールを行う
                var returnCode = TargetShellLibrary.InvokeExe(
                        Path.Combine(parameter.NintendoSdkRoot,
                                ConstantsNx.RunOnTargetExe),
                        runOnTargetCommand.ToString(), parameter.SerialDirectroy,
                        parameter.TargetEntry.SerialNumber, CommonConstants.OutputLogName);
                if (returnCode)
                {
                    Interlocked.Increment(ref successDeviceCount);
                }
            }
            catch (TargetShellLibraryException exception)
            {
                TargetShellLibrary.PrintException(exception);
            }
        }

        /// <summary>
        /// アンインストールコマンドを構築
        /// </summary>
        /// <param name="parameter">コマンドのパラメータ</param>
        /// <returns>コマンド文字列</returns>
        private string CreateRunOnTargetCommand(CommandParameter parameter)
        {
            var runOnTargetCommandBuilder = new StringBuilder();
            // DevMenuCommand.nsp設定
            runOnTargetCommandBuilder.AppendFormat(" {0}", Path.Combine(
                    parameter.NintendoSdkRoot, ConstantsNx.DevMenuCommandNsp));
            // 詳細ログ
            if (parameter.Verbose)
            {
                runOnTargetCommandBuilder.AppendFormat(" --verbose");
            }
            // 要求する開発機指定
            if (!string.IsNullOrEmpty(parameter.TargetEntry.SerialNumber))
            {
                runOnTargetCommandBuilder.AppendFormat(" --target {0}",
                        parameter.TargetEntry.SerialNumber);
            }
            if (string.IsNullOrEmpty(parameter.Type))
            {
                // type未指定の場合は、applicationを設定する
                runOnTargetCommandBuilder.AppendFormat(" -- application");
                runOnTargetCommandBuilder.AppendFormat(
                        " uninstall {0}", parameter.ApplicationId);
            }
            else
            {
                // typeの設定
                runOnTargetCommandBuilder.AppendFormat(" -- {0}", parameter.Type);
                // アンインストールプログラムパスの設定
                if (parameter.Type.Contains("application") && parameter.AllUninstall)
                {
                    runOnTargetCommandBuilder.AppendFormat(" uninstall --all");
                }
                else
                {
                    runOnTargetCommandBuilder.AppendFormat(
                            " uninstall {0}", parameter.ApplicationId);
                }
                // Aocならインデックスを追加する
                if (parameter.Type.Contains("addoncontent"))
                {
                    // インデックスが設定されていたら設定
                    //（されていない場合は設定せずにAocをすべて消去）
                    if (!string.IsNullOrEmpty(parameter.Index))
                    {
                        runOnTargetCommandBuilder.AppendFormat(" -i {0}", parameter.Index);
                    }
                }
            }
            return runOnTargetCommandBuilder.ToString();
        }
    }
}
