﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace PrivateNx.InitializeBuildImage
{
    using System;
    using System.Collections.Generic;
    using System.ComponentModel.Composition;
    using System.IO;
    using System.Text;
    using System.Threading;
    using PrivateNx.Common;
    using TargetShell.Library;
    using TargetShell.CommandPluginBase;
    using TargetShell.DevicePluginBase;
    using TargetShell.PluginInterface;

    /// <summary>
    /// 初期化イメージ(initimg)を使ってEdevの初期化を行うクラス
    /// </summary>
    [Export(typeof(IDevicePluginInterface<CommandParameter>))]
    [ExportMetadata("SupportedHardwareType", "Edev")]
    public class InitializeBuildImageEdev :
            BaseDevicePlugin<CommandParameter>, IDevicePluginInterface<CommandParameter>
    {
        /// <summary>
        /// コンストラクタ
        /// </summary>
        public InitializeBuildImageEdev()
        {
            this.ParallelModeFlag = false;
        }

        /// <summary>
        /// Edev 初期化した後の待ち時間。連続でこのプログラムで初期化する場合、
        /// ウェイトを入れないとデバイス検索で見つからない
        /// </summary>
        public const int WaitTimeAfterEdevInitialize = 10 * 1000;

        /// <summary>
        /// サーチ条件取得
        /// </summary>
        /// <returns>サーチ条件の文字列</returns>
        public override List<DevicePluginInfo> GetDeviceInfo()
        {
            var searchDeviceInfoList = TargetShellLibrary.GetDeviceInfoEdev();
            return searchDeviceInfoList;
        }

        /// <summary>
        /// 初期化イメージ(initimg)を使ってEdevの初期化を行う
        /// </summary>
        /// <param name="deviceList">デバイスリストのパラメータ</param>
        /// <param name="parameter">デバイスの初期化を行うパラメータ</param>
        /// <returns>成功したデバイスの数</returns>
        public override int RunDevicePlugin(
                List<TargetEntry> deviceList, CommandParameter parameter)
        {
            // 成功したデバイスの数
            var successDeviceCount = 0;
            // コネクションタイプを保持する
            var oldConnectionType = parameter.BuildImageParam.ConnectionType;
            // キータイプを保持する
            var oldKeyType = parameter.BuildImageParam.KeyType;
            // Edevでホストブリッジは使用できないので、強制的にUsbに変更する
            parameter.BuildImageParam.ConnectionType = ConstantsEdev.CommandString.Usb;
            // Edevでホストブリッジは使用できないので、強制的にKeyTypeを指定する
            // M2の場合は、M2を使用する。Autoが指定された場合は警告文を表示する。
            if (parameter.BuildImageParam.KeyType != "M2")
            {
                if (parameter.BuildImageParam.KeyType == "Auto")
                {
                    Console.WriteLine("[WARNING] KeyType Auto is Not Supported on Edev.\n" +
                            "[WARNING] KeyType is forcibly set to K5.");
                }
                parameter.BuildImageParam.KeyType = ConstantsEdev.CommandString.K5;
            }
            // Edevのみを取得する
            var devicesEdev = GetTargetDevice(deviceList, GetDeviceInfo());
            // デバイスの接続数分、シリアルでログディレクトリ作成及び初期化を行う
            foreach (var device in devicesEdev)
            {
                var localParameter = new CommandParameter
                {
                    TargetEntry = device,
                    Verbose = parameter.Verbose,
                    NintendoSdkRoot = parameter.NintendoSdkRoot,
                    BuildImageParam = parameter.BuildImageParam,
                    LogOutputDir = parameter.LogOutputDir
                };
                // ログ出力先を設定している場合のみログディレクトリを作成する
                localParameter.SerialDirectroy = TargetShellLibrary.CreateSerialNumberLogDirectory(
                        parameter.LogOutputDir, localParameter.TargetEntry.SerialNumber);
                SpecificProcessDevicePlugin(localParameter, ref successDeviceCount);
                // Edev 初期化後ウェイトを入れないと次のデバイス検索で Edev が見つからない
                if (devicesEdev.Count > 0)
                {
                    Thread.Sleep(WaitTimeAfterEdevInitialize);
                }
            }
            // 保持しておいたコネクションタイプとキータイプを戻す
            parameter.BuildImageParam.ConnectionType = oldConnectionType;
            parameter.BuildImageParam.KeyType = oldKeyType;
            return successDeviceCount;
        }

        /// <summary>
        /// デバイスプラグイン実行メソッド
        /// </summary>
        /// <param name="parameter">コマンドのパラメータ</param>
        /// <param name="successDeviceCount">成功デバイスの数</param>
        protected override void SpecificProcessDevicePlugin(
                CommandParameter parameter, ref int successDeviceCount)
        {
            // プラットフォームネームをディレクトリ名に変更する
            ConvertPlatformName(ref parameter);

            try
            {
                // Phidgetsが有効なら
                if (!parameter.BuildImageParam.PhidgetsOff)
                {
                    // phidgetsで電源を切る
                    InvokeControlTargetPrivate(parameter, "power-off-phidgets");

                    // フォースリカバリーモードで起動する
                    InvokeControlTargetPrivate(parameter, "force-recovery");
                }

                // 初期化を行う前の準備としてファイル、ディレクトリのチェック、作成、コピーなどを行う
                InitializeDirectory(parameter);

                // これから使用するプロセスを停止する
                StopProcess();

                // ControlTarget.exeを使用して、全ての登録を解除する
                InvokeControlTarget(parameter, "unregister-all");

                // Phidgetsが有効なら
                if (!parameter.BuildImageParam.PhidgetsOff)
                {
                    // RcmWriter.exeを呼び出す
                    InvokeRcmWriter(parameter);

                    // ターゲットが起動するまで待機
                    WaitStartUpDevice(parameter);
                }

                // 初期化イメージの書き込み
                WriteInitializeImage(parameter);
            }
            catch (TargetShellLibraryException exception)
            {
                TargetShellLibrary.PrintException(exception);
            }
            successDeviceCount++;
        }

        /// <summary>
        /// プラットフォームネームをディレクトリ名に変更する
        /// </summary>
        /// <param name="parameter">デバイスの初期化を行うパラメータ</param>
        private void ConvertPlatformName(ref CommandParameter parameter)
        {
            if (parameter.BuildImageParam.PlatformName.Equals(
                    ConstantsEdev.CommandString.Nxfp2a64))
            {
                parameter.BuildImageParam.PlatformName =
                        ConstantsEdev.CommandString.Nxfp2a64Path;
            }
            else if (parameter.BuildImageParam.PlatformName.Equals(
                    ConstantsEdev.CommandString.Nxfp2a32))
            {
                parameter.BuildImageParam.PlatformName =
                    ConstantsEdev.CommandString.Nxfp2a32Path;
            }
        }

        /// <summary>
        /// 初期化を行う前の準備としてファイル、ディレクトリのチェック、作成、コピーなどを行う
        /// </summary>
        /// <param name="parameter">パラメータ</param>
        private void InitializeDirectory(CommandParameter parameter)
        {
            // Output先のディレクトリを作成
            if (!TargetShellLibrary.CreateDirectory(parameter.NintendoSdkRoot,
                ConstantsEdev.OutputDir))
            {
                throw new TargetShellLibraryException("InitializeBuildImageEdev.InitializeDirectory");
            }

            var rcmWriterXml = string.Empty;
            var bootConfigFile = string.Empty;
            // Phidgetsが有効なら
            if (!parameter.BuildImageParam.PhidgetsOff)
            {
                if (parameter.BuildImageParam.KeyType == "M2")
                {
                    rcmWriterXml = Path.Combine(parameter.NintendoSdkRoot,
                        ConstantsEdev.RcmWriterXmlWithBcM2);
                }
                else
                {
                    rcmWriterXml = Path.Combine(parameter.NintendoSdkRoot,
                        ConstantsEdev.RcmWriterXmlWithBc);
                }

                // BootConfigを書き換える
                if (parameter.BuildImageParam.SignedType.Equals(
                        ConstantsEdev.CommandString.SignedString))
                {
                    // 署名ありはダミーファイルを書き込む
                    bootConfigFile = Path.Combine(parameter.NintendoSdkRoot,
                            ConstantsEdev.BootConfigDummy);
                }
                else
                {
                    // 署名なしはBootConfigファイルを取得する
                    bootConfigFile = TargetShellLibrary.ReplaceFirst(ConstantsEdev.BootConfigInRepo,
                            @"\" + ConstantsEdev.Caret, parameter.TargetEntry.SerialNumber);
                    bootConfigFile = Path.Combine(parameter.NintendoSdkRoot, bootConfigFile);
                }

                // BootConfigファイルの存在確認
                if (!TargetShellLibrary.IsExistFile(bootConfigFile))
                {
                    throw new TargetShellLibraryException(
                            $"Write BootConfig file not found\nBootConfigFile = {bootConfigFile}");
                }

                if (!TargetShellLibrary.CopyFile(bootConfigFile,
                        Path.Combine(parameter.NintendoSdkRoot, ConstantsEdev.OutputBootConfig), true))
                {
                    throw new TargetShellLibraryException("InitializeBuildImageEdev." +
                            "InitializeDirectory");
                }
            }
            else
            {
                // Phidgetsが無効ならBootConfigを書き換えない
                if (parameter.BuildImageParam.KeyType == "M2")
                {
                    rcmWriterXml = Path.Combine(parameter.NintendoSdkRoot,
                            ConstantsEdev.RcmWriterXmlWoBcM2);
                }
                else
                {
                    rcmWriterXml = Path.Combine(parameter.NintendoSdkRoot,
                            ConstantsEdev.RcmWriterXmlWoBc);
                }
            }

            var package2 = string.Empty;
            if (parameter.BuildImageParam.SignedType.Equals(ConstantsEdev.CommandString.SignedString))
            {
                package2 = TargetShellLibrary.ReplaceFirst(ConstantsEdev.Package2Signed,
                        @"\" + ConstantsEdev.Caret, parameter.BuildImageParam.BuildType);
                package2 = Path.Combine(parameter.NintendoSdkRoot, package2);
                if (parameter.BuildImageParam.KeyType == "M2")
                {
                    rcmWriterXml = Path.Combine(parameter.NintendoSdkRoot,
                           ConstantsEdev.RcmWriterXmlWithBcM2);
                }
                else
                {
                    rcmWriterXml = Path.Combine(parameter.NintendoSdkRoot,
                           ConstantsEdev.RcmWriterXmlWithBc);
                }
                bootConfigFile = Path.Combine(parameter.NintendoSdkRoot,
                        ConstantsEdev.BootConfigDummy);
                if (!TargetShellLibrary.CopyFile(bootConfigFile,
                        Path.Combine(parameter.NintendoSdkRoot, ConstantsEdev.OutputBootConfig), true))
                {
                    throw new TargetShellLibraryException("InitializeBuildImageEdev." +
                            "InitializeDirectory");
                }
            }
            else
            {
                package2 = TargetShellLibrary.ReplaceFirst(ConstantsEdev.Package2Unsigned,
                        @"\" + ConstantsEdev.Caret, parameter.BuildImageParam.PlatformName);
                package2 = TargetShellLibrary.ReplaceFirst(package2, @"\" + ConstantsEdev.Caret,
                        parameter.BuildImageParam.BuildType);
                package2 = Path.Combine(parameter.NintendoSdkRoot, package2);
            }

            // ファイル存在確認
            if (!TargetShellLibrary.IsExistFile(Path.Combine(parameter.NintendoSdkRoot,
                ConstantsEdev.RcmWriter)))
            {
                throw new TargetShellLibraryException("InitializeBuildImageEdev." +
                        "InitializeDirectory");
            }
            var rcmWriterRcm = TargetShellLibrary.ReplaceFirst(ConstantsEdev.RcmWriterRcm,
                    @"\" + ConstantsEdev.Caret, parameter.BuildImageParam.KeyType);
            rcmWriterRcm = Path.Combine(parameter.NintendoSdkRoot, rcmWriterRcm);
            if (!TargetShellLibrary.IsExistFile(rcmWriterRcm))
            {
                throw new TargetShellLibraryException("InitializeBuildImageEdev." +
                        "InitializeDirectory");
            }
            if (!TargetShellLibrary.IsExistFile(rcmWriterXml))
            {
                throw new TargetShellLibraryException("InitializeBuildImageEdev." +
                        "InitializeDirectory");
            }
            var systemUpdater = TargetShellLibrary.ReplaceFirst(ConstantsEdev.SystemUpdater,
                    @"\" + ConstantsEdev.Caret, parameter.BuildImageParam.PlatformName);
            systemUpdater = TargetShellLibrary.ReplaceFirst(systemUpdater,
                    @"\" + ConstantsEdev.Caret, parameter.BuildImageParam.BuildType);
            systemUpdater = Path.Combine(parameter.NintendoSdkRoot, systemUpdater);
            if (!TargetShellLibrary.IsExistFile(systemUpdater))
            {
                throw new TargetShellLibraryException("InitializeBuildImageEdev." +
                        "InitializeDirectory");
            }
            if (!TargetShellLibrary.IsExistFile(Path.Combine(parameter.NintendoSdkRoot,
                    ConstantsNx.RunOnTargetPrivateExe)))
            {
                throw new TargetShellLibraryException("InitializeBuildImageEdev." +
                        "InitializeDirectory");
            }
            if (!TargetShellLibrary.IsExistFile(Path.Combine(parameter.NintendoSdkRoot,
                    ConstantsNx.ControlTargetExe)))
            {
                throw new TargetShellLibraryException("InitializeBuildImageEdev." +
                        "InitializeDirectory");
            }
            var initialImagePath = GetInitializeImagePath(parameter);
            if (!TargetShellLibrary.IsExistFile(initialImagePath))
            {
                throw new TargetShellLibraryException("InitializeBuildImageEdev." +
                        "InitializeDirectory");
            }
            var bct = TargetShellLibrary.ReplaceFirst(ConstantsEdev.Bct, @"\" + ConstantsEdev.Caret,
                    parameter.BuildImageParam.KeyType);
            bct = Path.Combine(parameter.NintendoSdkRoot, bct);
            if (!TargetShellLibrary.IsExistFile(bct))
            {
                throw new TargetShellLibraryException("InitializeBuildImageEdev." +
                        "InitializeDirectory");
            }
            var bl = TargetShellLibrary.ReplaceFirst(ConstantsEdev.Bl, @"\" + ConstantsEdev.Caret,
                    parameter.BuildImageParam.KeyType);
            bl = Path.Combine(parameter.NintendoSdkRoot, bl);
            if (!TargetShellLibrary.IsExistFile(bl))
            {
                throw new TargetShellLibraryException("InitializeBuildImageEdev." +
                        "InitializeDirectory");
            }
            var gpt = Path.Combine(parameter.NintendoSdkRoot, ConstantsEdev.Gpt);
            if (!TargetShellLibrary.IsExistFile(gpt))
            {
                throw new TargetShellLibraryException("InitializeBuildImageEdev." +
                        "InitializeDirectory");
            }
            if (!TargetShellLibrary.IsExistFile(package2))
            {
                throw new TargetShellLibraryException("InitializeBuildImageEdev." +
                        "InitializeDirectory");
            }
            var systemPartitionPath = string.Empty;
            if (parameter.BuildImageParam.SignedType == "Unsigned")
            {
                systemPartitionPath = ConstantsEdev.SystemPartitionP2Unsigned;
            }
            else if (parameter.BuildImageParam.SignedType == "Signed")
            {
                systemPartitionPath = ConstantsEdev.SystemPartitionP2Signed;
            }
            var systemPartition = TargetShellLibrary.ReplaceFirst(systemPartitionPath,
                    @"\" + ConstantsEdev.Caret, parameter.BuildImageParam.PlatformName);
            systemPartition = TargetShellLibrary.ReplaceFirst(systemPartition,
                    @"\" + ConstantsEdev.Caret, parameter.BuildImageParam.BuildType);
            systemPartition = Path.Combine(parameter.NintendoSdkRoot, systemPartition);
            if (!TargetShellLibrary.IsExistFile(systemPartition))
            {
                throw new TargetShellLibraryException("InitializeBuildImageEdev." +
                        "InitializeDirectory");
            }

            // Outputへファイルコピー
            if (!TargetShellLibrary.CopyFile(rcmWriterXml,
                Path.Combine(parameter.NintendoSdkRoot, ConstantsEdev.OutputConfig), true))
            {
                throw new TargetShellLibraryException("InitializeBuildImageEdev." +
                        "InitializeDirectory");
            }
            if (!TargetShellLibrary.CopyFile(bct,
                Path.Combine(parameter.NintendoSdkRoot, ConstantsEdev.OutputBct), true))
            {
                throw new TargetShellLibraryException("InitializeBuildImageEdev." +
                        "InitializeDirectory");
            }
            if (!TargetShellLibrary.CopyFile(bl,
                Path.Combine(parameter.NintendoSdkRoot, ConstantsEdev.OutputBl), true))
            {
                throw new TargetShellLibraryException("InitializeBuildImageEdev." +
                        "InitializeDirectory");
            }
            if (!TargetShellLibrary.CopyFile(gpt,
                Path.Combine(parameter.NintendoSdkRoot, ConstantsEdev.OutputGpt), true))
            {
                throw new TargetShellLibraryException("InitializeBuildImageEdev." +
                        "InitializeDirectory");
            }
            if (!TargetShellLibrary.CopyFile(package2,
                Path.Combine(parameter.NintendoSdkRoot, ConstantsEdev.OutputPackage2), true))
            {
                throw new TargetShellLibraryException("InitializeBuildImageEdev." +
                        "InitializeDirectory");
            }
            if (!TargetShellLibrary.CopyFile(systemPartition,
                Path.Combine(parameter.NintendoSdkRoot, ConstantsEdev.OutputSystemPartition), true))
            {
                throw new TargetShellLibraryException("InitializeBuildImageEdev." +
                        "InitializeDirectory");
            }
        }

        /// <summary>
        /// 初期化で使用するプロセスを停止する
        /// </summary>
        private void StopProcess()
        {
            if (!TargetShellLibrary.StopProcesses("RunOnTargetPrivate"))
            {
                throw new TargetShellLibraryException("InitializeBuildImageEdev.StopProcess");
            }
            if (!TargetShellLibrary.StopProcesses("RecoveryBoot"))
            {
                throw new TargetShellLibraryException("InitializeBuildImageEdev.StopProcess");
            }
            if (!TargetShellLibrary.StopProcesses("UpdateHostBridge"))
            {
                throw new TargetShellLibraryException("InitializeBuildImageEdev.StopProcess");
            }
            if (!TargetShellLibrary.StopProcesses("RunOnTarget"))
            {
                throw new TargetShellLibraryException("InitializeBuildImageEdev.StopProcess");
            }
            if (!TargetShellLibrary.StopProcesses("ControlTarget"))
            {
                throw new TargetShellLibraryException("InitializeBuildImageEdev.StopProcess");
            }
            if (!TargetShellLibrary.StopProcesses("RunOnTargetFromNand"))
            {
                throw new TargetShellLibraryException("InitializeBuildImageEdev.StopProcess");
            }
            if (!TargetShellLibrary.StopProcesses("RunSystemInitializerManu"))
            {
                throw new TargetShellLibraryException("InitializeBuildImageEdev.StopProcess");
            }
            if (!TargetShellLibrary.StopProcesses("RequestToWsIssuer"))
            {
                throw new TargetShellLibraryException("InitializeBuildImageEdev.StopProcess");
            }
        }

        /// <summary>
        /// ControlTarget.exeを呼び出す
        /// </summary>
        /// <param name="parameter">ControlTarget.exeを呼び出すパラメータ</param>
        /// <param name="command">指定するコマンド</param>
        private void InvokeControlTarget(CommandParameter parameter,
            string command)
        {
            if (!InvokeExe(parameter, ConstantsNx.ControlTargetExe,
                    command, CommonConstants.OutputLogName))
            {
                throw new TargetShellLibraryException("InitializeBuildImageEdev." +
                        "InvokeControlTarget");
            }
        }

        /// <summary>
        /// RcmWriter.exeを呼び出す
        /// </summary>
        /// <param name="parameter">RcmWriterを呼び出すパラメータ</param>
        private void InvokeRcmWriter(CommandParameter parameter)
        {
            // コマンドを設定
            var rcmWriterRcm = TargetShellLibrary.ReplaceFirst(ConstantsEdev.RcmWriterRcm,
                    @"\" + ConstantsEdev.Caret, parameter.BuildImageParam.KeyType);
            rcmWriterRcm = Path.Combine(parameter.NintendoSdkRoot, rcmWriterRcm);
            var outputConfig = Path.Combine(parameter.NintendoSdkRoot, ConstantsEdev.OutputConfig);
            var rcmWriterParam = new StringBuilder();
            rcmWriterParam.Append(" --rcm ");
            rcmWriterParam.Append(rcmWriterRcm);
            rcmWriterParam.Append(" --list ");
            rcmWriterParam.Append(outputConfig);

            if (!InvokeExe(parameter, ConstantsEdev.RcmWriter,
                    rcmWriterParam.ToString(), CommonConstants.OutputLogName))
            {
                throw new TargetShellLibraryException("InitializeBuildImageEdev.InvokeRcmWriter");
            }
        }

        /// <summary>
        /// デバイスが起動するまで待機
        /// </summary>
        /// <param name="parameter">ControlTargetPrivate.exeを呼び出すパラメータ</param>
        private void WaitStartUpDevice(CommandParameter parameter)
        {
            var outputDetectTarget = string.Empty;
            var isSuccess = false;

            // デバイスが起動するまで1秒間隔で待機して起動チェックをする
            // （10秒程で起動するはず）
            for (var i = 0; i < ConstantsEdev.MaxWaitStartUpDevice; i++)
            {
                if (!InvokeExe(parameter, ConstantsEdev.ControlTargetPrivatePath,
                        "detect-target", CommonConstants.OutputLogName,
                        out outputDetectTarget))
                {
                    throw new TargetShellLibraryException("InitializeBuildImageEdev." +
                            "WaitStartUpDevice");
                }
                if (outputDetectTarget.Contains(parameter.TargetEntry.SerialNumber))
                {
                    isSuccess = true;
                    break;
                }
                Thread.Sleep(1000);
            }

            // タイムアウトしている場合はエラーとする
            if (!isSuccess)
            {
                throw new TargetShellLibraryException("InitializeBuildImageEdev." +
                        "WaitStartUpDevice");
            }
        }

        /// <summary>
        /// 初期化イメージの書き込み
        /// </summary>
        /// <param name="parameter">デバイスの初期化を行うパラメータ</param>
        private void WriteInitializeImage(CommandParameter parameter)
        {
            // 実行ファイル
            var systemUpdater = TargetShellLibrary.ReplaceFirst(ConstantsEdev.SystemUpdater,
                    @"\" + ConstantsEdev.Caret, parameter.BuildImageParam.PlatformName);
            systemUpdater = TargetShellLibrary.ReplaceFirst(systemUpdater,
                    @"\" + ConstantsEdev.Caret, parameter.BuildImageParam.BuildType);
            systemUpdater = Path.Combine(parameter.NintendoSdkRoot, systemUpdater);

            // イメージファイル
            var initialImagePath = GetInitializeImagePath(parameter);

            // RunOnTargetPrivateのコマンド生成
            var runOnTargetPrivateCommandBuilder = new StringBuilder();
            runOnTargetPrivateCommandBuilder.AppendFormat(
                    " run --target usb --verbose --failure-timeout 600 --pattern-success-exit");
            runOnTargetPrivateCommandBuilder.AppendFormat(
                    " \"Succeeded initializing the system.\"");
            runOnTargetPrivateCommandBuilder.AppendFormat(
                    " {0}", systemUpdater);
            runOnTargetPrivateCommandBuilder.AppendFormat(
                    " -- --input {0}", initialImagePath);

            // RunOnTargetPrivate.exeの実行
            if (!InvokeExe(parameter, ConstantsNx.RunOnTargetPrivateExe,
                    runOnTargetPrivateCommandBuilder.ToString(), CommonConstants.OutputLogName))
            {
                throw new TargetShellLibraryException("InitializeBuildImageEdev." +
                        "WriteInitializeImage");
            }
        }

        /// <summary>
        /// ControlTargetPrivate.exeを呼び出す
        /// </summary>
        /// <param name="parameter">ControlTargetPrivate.exeを呼び出すパラメータ</param>
        /// <param name="command">指定するコマンド</param>
        private void InvokeControlTargetPrivate(CommandParameter parameter,
            string command)
        {
            if (!InvokeExe(parameter, ConstantsEdev.ControlTargetPrivatePath,
                    command, CommonConstants.OutputLogName))
            {
                throw new TargetShellLibraryException("InitializeBuildImageEdev." +
                        "InvokeControlTargetPrivate");
            }
        }

        /// <summary>
        /// 実行ファイルを呼び出す
        /// </summary>
        /// <param name="parameter">実行ファイルを呼び出すパラメータ</param>
        /// <param name="exeName">SDKルート以降のパスを含む実行ファイル名</param>
        /// <param name="command">指定するコマンド</param>
        /// <param name="logName">ログファイル名</param>
        /// <returns>True:成功 False:失敗</returns>
        private bool InvokeExe(CommandParameter parameter,
            string exeName, string command, string logName)
        {
            var returnCode = TargetShellLibrary.InvokeExe(
                    Path.Combine(parameter.NintendoSdkRoot, exeName), command,
                    parameter.SerialDirectroy, parameter.TargetEntry.SerialNumber, logName);
            return returnCode;
        }

        /// <summary>
        /// 実行ファイルを呼び出す
        /// </summary>
        /// <param name="parameter">実行ファイルを呼び出すパラメータ</param>
        /// <param name="exeName">SDKルート以降のパスを含む実行ファイル名</param>
        /// <param name="command">指定するコマンド</param>
        /// <param name="logName">ログファイル名</param>
        /// <param name="outputString">実行した出力結果の文字列</param>
        /// <returns>True:成功 False:失敗</returns>
        private bool InvokeExe(CommandParameter parameter,
            string exeName, string command, string logName, out string outputString)
        {
            var returnCode = TargetShellLibrary.InvokeExe(
                    Path.Combine(parameter.NintendoSdkRoot, exeName), command,
                    parameter.SerialDirectroy, parameter.TargetEntry.SerialNumber, logName,
                    out outputString);
            return returnCode;
        }

        /// <summary>
        /// 初期化イメージのファイル名を含めたフルパスを返す
        /// </summary>
        /// <param name="parameter">フルパスを求める為のパラメータ</param>
        /// <returns>初期化イメージのファイル名を含めたフルパス</returns>
        private string GetInitializeImagePath(CommandParameter parameter)
        {
            var connectionType = string.Empty;
            if (parameter.BuildImageParam.ConnectionType.Equals(
                    ConstantsEdev.CommandString.HostBridge))
            {
                connectionType = ConstantsEdev.CommandString.Hb;
            }
            else
            {
                connectionType = parameter.BuildImageParam.ConnectionType;
            }
            var initialImageName = TargetShellLibrary.ReplaceFirst(ConstantsEdev.InitialImageName,
                    @"\" + ConstantsEdev.Caret, parameter.BuildImageParam.KeyType);
            initialImageName = TargetShellLibrary.ReplaceFirst(initialImageName,
                    @"\" + ConstantsEdev.Caret, connectionType);
            initialImageName = TargetShellLibrary.ReplaceFirst(initialImageName,
                    @"\" + ConstantsEdev.Caret, parameter.BuildImageParam.SignedType);
            initialImageName = TargetShellLibrary.ReplaceFirst(initialImageName,
                    @"\" + ConstantsEdev.Caret, parameter.BuildImageParam.BootType);
            if (parameter.TargetEntry.SerialNumber.Contains(ConstantsEdev.CommandString.CopperDevice))
            {
                initialImageName = TargetShellLibrary.ReplaceFirst(initialImageName,
                        @"\" + ConstantsEdev.Caret, "32G");
            }
            else
            {
                initialImageName = TargetShellLibrary.ReplaceFirst(initialImageName,
                        @"\" + ConstantsEdev.Caret, "64G");
            }
            initialImageName = TargetShellLibrary.ReplaceFirst(initialImageName,
                    @"\" + ConstantsEdev.Caret, parameter.BuildImageParam.DeveloperType);

            var initialImagePath = TargetShellLibrary.ReplaceFirst(ConstantsEdev.InitialImagePath,
                    @"\" + ConstantsEdev.Caret, parameter.BuildImageParam.PlatformName);
            initialImagePath = TargetShellLibrary.ReplaceFirst(initialImagePath,
                    @"\" + ConstantsEdev.Caret, initialImageName);
            initialImagePath = TargetShellLibrary.ReplaceFirst(initialImagePath,
                    @"\" + ConstantsEdev.Caret, parameter.BuildImageParam.BuildType);
            initialImagePath = TargetShellLibrary.ReplaceFirst(initialImagePath,
                    @"\" + ConstantsEdev.Caret, initialImageName);
            initialImagePath = Path.Combine(parameter.NintendoSdkRoot, initialImagePath);
            return initialImagePath;
        }
    }
}
