﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace TargetShell.DevicePluginBase
{
    using System;
    using System.Collections.Generic;
    using System.Threading;
    using System.Threading.Tasks;
    using TargetShell.Library;
    using TargetShell.PluginInterface;

    /// <summary>
    /// 共通のデバイスプラグイン処理クラス
    /// </summary>
    public class BaseDevicePlugin<CommandParameter>
    {
        /// <summary>
        /// 実行モード
        /// </summary>
        protected bool ParallelModeFlag { get; set; }

        /// <summary>
        /// デバイスの情報取得
        /// （標準ではNXを取得するようになっているので他機種を取得したい場合はオーバーライドしてください）
        /// </summary>
        /// <returns>デバイスの情報リスト</returns>
        public virtual List<DevicePluginInfo> GetDeviceInfo()
        {
            var searchDeviceInfoList = TargetShellLibrary.GetDeviceInfoNx();
            return searchDeviceInfoList;
        }

        /// <summary>
        /// デバイスプラグイン実行
        /// </summary>
        /// <param name="deviceList">デバイスリストのパラメータ</param>
        /// <param name="parameter">パラメータ</param>
        /// <returns>成功したデバイスの数</returns>
        public virtual int RunDevicePlugin(
                List<TargetEntry> deviceList, CommandParameter parameter)
        {
            var successDeviceCount = 0;
            // 対象デバイスを取得する
            var targetDevices = GetTargetDevice(deviceList, GetDeviceInfo());
            if (ParallelModeFlag)
            {
                successDeviceCount = ParallelRun(targetDevices, parameter);
            }
            else
            {
                successDeviceCount = SelialRun(targetDevices, parameter);
            }
            return successDeviceCount;
        }

        /// <summary>
        /// ターゲットデバイス取得
        /// </summary>
        /// <param name="deviceList">デバイスリスト</param>
        /// <param name="filterTargetArray">フィルターするデバイス</param>
        /// <returns>対応するデバイスリスト</returns>
        public List<TargetEntry> GetTargetDevice(
                List<TargetEntry> deviceList, List<DevicePluginInfo> filterTargetArray)
        {
            var filteredDevice = new List<TargetEntry>();
            foreach (var filterTarget in filterTargetArray)
            {
                filteredDevice.AddRange(
                        deviceList.FindAll(device =>
                        device.HardwareType == filterTarget.SupportHardwareType));
            }
            return filteredDevice;
        }

        /// <summary>
        /// デバイスプラグイン固有処理実行メソッド
        /// </summary>
        /// <param name="parameter">コマンドのパラメータ</param>
        protected virtual void SpecificProcessDevicePlugin(
                CommandParameter parameter, ref int successDeviceCount)
        {
            Console.WriteLine("Please override the specific process this method.");
            Interlocked.Increment(ref successDeviceCount);
        }

        /// <summary>
        /// 表示ログ整形
        /// </summary>
        /// <param name="parameter">コマンドのパラメータ</param>
        /// <param name="multiLineLog">複数行ログフラグ</param>
        /// <returns>コマンド文字列</returns>
        protected void PreformatLogString(CommandParameter parameter, bool multiLineLog = false)
        {
            var deviceParameter =
                    TargetShellLibrary.CastParameter<CommonDevicePluginOptions>(parameter);
            var outputStringGenerator = new OutputStringGenerator(
                    deviceParameter.LogHideString[deviceParameter.RunCount])
            {
                Verbose = deviceParameter.Verbose,
                SerialNumber = deviceParameter.TargetEntry.SerialNumber,
                OutputOneLinePrefix = deviceParameter.OutputOneLinePrefix,
                MultiLineLog = multiLineLog,
            };
            outputStringGenerator.GeneratOutputStirng();
            deviceParameter.LogHideString[deviceParameter.RunCount] =
                    outputStringGenerator.LogHideString;
        }

        /// <summary>
        /// 並列実行
        /// </summary>
        /// <param name="targetDevices">開発機情報</param>
        /// <param name="parameter">パラメータ</param>
        /// <returns>成功したデバイスの数</returns>
        private int ParallelRun(
            List<TargetEntry> targetDevices, CommandParameter parameter)
        {
            var successDeviceCount = 0;
            Parallel.For(0, targetDevices.Count, id =>
            {
                ProcessDevicePlugin(targetDevices[id], parameter, id, ref successDeviceCount);
            });
            return successDeviceCount;
        }

        /// <summary>
        /// 直列実行
        /// </summary>
        /// <param name="targetDevices">開発機情報</param>
        /// <param name="parameter">パラメータ</param>
        /// <returns>成功したデバイスの数</returns>
        private int SelialRun(
                List<TargetEntry> targetDevices, CommandParameter parameter)
        {
            var successDeviceCount = 0;
            var runCount = 0;
            foreach (var oneDevice in targetDevices)
            {
                runCount++;
                ProcessDevicePlugin(oneDevice, parameter, runCount,  ref successDeviceCount);
            }
            return successDeviceCount;
        }

        /// <summary>
        /// デバイスプラグイン実行準備
        /// </summary>
        /// <param name="targetEntry">ターゲット情報</param>
        /// <param name="parameter">コマンドパラメータ</param>
        /// <returns>コマンドパラメータ</returns>
        private CommandParameter PrepareDevicePlugin(
                TargetEntry targetEntry, CommandParameter parameter, int runCount)
        {
            var deviceParameter =
                    TargetShellLibrary.CastParameter<CommonDevicePluginOptions>(parameter).Clone();
            deviceParameter.TargetEntry = targetEntry;
            deviceParameter.SerialDirectroy = TargetShellLibrary.CreateSerialNumberLogDirectory(
                    deviceParameter.LogOutputDir, deviceParameter.TargetEntry.SerialNumber);
            deviceParameter.RunCount = runCount;
            return TargetShellLibrary.CastParameter<CommandParameter>(deviceParameter);
        }

        /// <summary>
        /// デバイスプラグイン処理実行
        /// </summary>
        /// <param name="targetEntry">ターゲット情報</param>
        /// <param name="parameter">コマンドパラメータ</param>
        /// <param name="successDeviceCount">成功したデバイスの数</param>
        private void ProcessDevicePlugin(TargetEntry targetEntry,
                CommandParameter parameter, int runCount, ref int successDeviceCount)
        {
            var commadparameter = PrepareDevicePlugin(targetEntry, parameter, runCount);
            SpecificProcessDevicePlugin(commadparameter, ref successDeviceCount);
        }
    }
}
