﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace TargetShell.CommandPluginBase
{
    using System;
    using System.Collections.Generic;
    using System.ComponentModel.Composition;
    using System.ComponentModel.Composition.Hosting;
    using System.IO;
    using System.Reflection;
    using TargetShell.Library;
    using TargetShell.PluginInterface;

    /// <summary>
    /// 共通のデバイスプラグイン呼び出しクラス
    /// </summary>
    public class DevicePluginCaller<ICommonDevicePluginInterface, CommandParameter>
    {
        /// <summary>
        /// CompositionContainer
        /// </summary>
        private CompositionContainer container;

        /// <summary>
        /// MEF のインターフェース
        /// </summary>
        [ImportMany]
        private IEnumerable<Lazy
                <IDevicePluginInterface<CommandParameter>>> operations = null;

        /// <summary>
        /// MEF のロードを行う
        /// </summary>
        public void Load()
        {
            using (var catalog = new AggregateCatalog())
            {
                var myAssembly = Assembly.GetEntryAssembly();
                var path = myAssembly.Location;
                var directory = Path.GetDirectoryName(path);
                catalog.Catalogs.Add(new DirectoryCatalog(
                        Path.Combine(directory, TargetShellPluginConstants.PluginDirectory)));

                // CompositionContainer を作成する(第2引数はスレッドセーフ有効)
                container = new CompositionContainer(catalog, true);

                // Imports を構成する
                try
                {
                    container.ComposeParts(this);
                }
                catch (CompositionException compositionException)
                {
                    TargetShellLibrary.PrintException(compositionException);
                }
            }
        }

        /// <summary>
        /// 各デバイスの検索用キーワードを取得する
        /// <returns>デバイスプラグイン情報</returns>
        /// </summary>
        public List<DevicePluginInfo> GetDevicePluginInfo()
        {
            var devicePluginInfo = new List<DevicePluginInfo>();
            foreach (Lazy<IDevicePluginInterface<CommandParameter>> operation in operations)
            {
                // 各デバイスの検索用キーワードを取得する
                devicePluginInfo.AddRange(operation.Value.GetDeviceInfo());
            }
            return devicePluginInfo;
        }

        /// <summary>
        /// デバイスプラグインを実行する
        /// </summary>
        /// <param name="deviceList">デバイスのリスト</param>
        /// <param name="parameter">プラグインのパラメータ</param>
        /// <returns>成功したデバイスの数</returns>
        public int RunDevicePlugin(List<TargetEntry> deviceList, CommandParameter parameter)
        {
            var successDeviceCount = 0; // 成功したデバイスの数
            foreach (Lazy<IDevicePluginInterface<CommandParameter>> operation in operations)
            {
                successDeviceCount += operation.Value.RunDevicePlugin(
                        deviceList, parameter);
            }
            return successDeviceCount;
        }
    }
}
