﻿#
#    COMMAND [-ConfigFile <files>] [-OutputDir <dirname>] [-OutputOldLogDir <dirname>] [-PreApiUrl <url>]
#            [-English] [-Detail|-Quiet]
#
#       -ConfigFile <name>            Config ファイル。指定がない場合、言語によって
#                                     ".\config\ja\ChangelogConfig.xml" か ".\config\en\ChangelogConfig.xml" か
#       -AliasFile <file>             エイリアス情報が書かれている xml ファイル指定。
#                                     指定がなければ .\config\Aliases.xml を参照する。
#                                     指定したくなければ -AliasFile "" を指定のこと。
#       -OutputDir <dirname>          出力フォルダ。
#                                     指定がなければ、カレントを基準にし、指定があれば、そのフォルダを基準にする。
#                                     例えば html という指定なら html/ChangeLogXXX.html や
#                                     html/ReleaseNotes/ChangeLogXXX_XXX.html が生成される。
#       -PreApiUrl                    APIリファレンスへの位置。トップの html がある位置から。
#                                     "../../../Api/Html" のように指定する。
#       -PreDocUrl                    公開ドキュメントへの位置。トップの html がある位置から。
#                                     "../../../Pages" のように指定する。
#       -English                      指定すると、表示文字列などに英語版を用いるよう、用語ファイルの読み込みを切りかえる
#       -Document                     ドキュメント向け変更履歴の作成
#       -DocumentOldLog               ドキュメントについては、過去ログを一部参照する
#       -Detail                       動作の情報を詳細に表示する。省略化で、省略時は詳細を表示しない。
#                                     Detail と Quiet は片方しか指定できません。
#       -Quiet                        動作の情報の表示を出来るだけ抑制する。エラーや警告だけ表示する。
#                                     Detail と Quiet は片方しか指定できません。
#
#    (例) COMMAND -OutputDir html -Detail
#
#    .\config\ChangelogConfig.xml にコンフィグデータを書き込んでおく必要がある。
#      MainId:      ログのID割り振りに使われる。通常連番だが文字列でも問題ない。
#      PackageName: パッケージ名
#      ReleaseDate: リリース日
#      TopInformation: 先頭に置く重要なお知らせ。改行は "&lt;be>" とする。
#      PackageSupplementation : 追加情報
#
param
(
    [string]$AuthenticationFile = "",
    [String]$OutputDir = ".\",
    [string]$AliasFile = ".\config\Aliases.xml",
    [String]$ConfigFile = ".\config\ja\ChangeLogConfig_NX.xml",
    [String]$PreApiUrl = "./",
    [String]$PreDocUrl = "./",
    [switch]$English = $false,
    [switch]$Document = $false,
    [switch]$Detail = $false,
    [switch]$Quiet = $false
)

$AllCsv = @()   # 複数ファイルから CSVデータをすべてまとめて保持
$ResourceDir = Split-Path (Split-Path (Split-Path (Split-Path $MyInvocation.MyCommand.Path -Parent) -Parent) -Parent) -Parent

$script:VersionForCsv = ""
$script:VersionForCsvNum = ""
$script:ListCsv = @()

. .\outputListChangeLog.ps1

#----------------------------------------------------------------
#---------------- 表示レベル
$LogLevel = 1
if ( $Detail -eq $true -and $Quiet -eq $true )
{
    Write-Host "*** Error: Cannot -Detail and -Quiet options are specified at a time."
}
if ( $Detail -eq $true ){ $LogLevel = 2 }
if ( $Quiet  -eq $true ){ $LogLevel = 0 }

function Write-HostLog( $level, $str )
{
    if ( $level -eq "Detail" )
    {
        if ( $LogLevel -ge 2 ){ Write-Host $str }
    }
    elseif ( $level -eq "Log" )
    {
        if ( $LogLevel -ge 1 ){ Write-Host $str }
    }
    elseif ( $level -eq "Error" )
    {
        Write-Host $str
    }
    elseif ( $level -eq "DetailOnly" )
    {
        if ( $LogLevel -eq 2 ){ Write-Host $str }
    }
    elseif ( $level -eq "LogOnly" )
    {
        if ( $LogLevel -eq 1 ){ Write-Host $str }
    }
    elseif ( $level -eq "QuietOnly" )
    {
        if ( $LogLevel -eq 0 ){ Write-Host $str }
    }
}
function Write-Error( $str )
{
    Write-HostLog "Error" ("*** Error: "+$str)
}
function Write-Warning( $str )
{
    Write-HostLog "Error" ("*** Warning: "+$str)
}
function Write-Detail( $str )
{
    Write-HostLog "Detail" $str
}
function Write-Log( $str )
{
    Write-HostLog "Log" $str
}

#----------------------------------------------------------------
# OutputDir 指定が 'html\' のような形式なら '\' を取る。
if ( $true -eq $OutputDir.EndsWith("\") )
{
    $OutputDir = $OutputDir.Remove( $OutputDir.Length - 1, 1 )
}

# PreApiUrl や PreDocUrl が "/" で終わってないなら "/" を追加する。
if ( $false -eq $PreApiUrl.EndsWith("/") )
{
    $PreApiUrl = $PreApiUrl + "/"
}
if ( $false -eq $PreDocUrl.EndsWith("/") )
{
    $PreDocUrl = $PreDocUrl + "/"
}

#---------------- エイリアス情報の読み込み
$projectFnameAliasRule = @{}
$packageAliasRule = @{}
$versionAliasRule = @{}
$packageFnameAliasRule = @{}
$versionFnameAliasRule = @{}
$idAliasRule = @{}
$IsLoadAliasList = $false
. .\loadAliasList.ps1
if ( $IsLoadAliasList -eq $true )
{
    $projectFnameAliasRule, $projectAliasArray = Create-ProjectAliases $Tags
    $packageAliasRule, $packageFnameAliasRule, $packageAliasArray, $packageFnameAliasArray = Create-PackageAliases $Tags
    $versionAliasRule, $versionFnameAliasRule, $versionAliasArray, $versionFnameAliasArray = Create-VersionAliases $Tags
    $idAliasRule = Create-IdAliases $Tags
}

# Confluence ページからバージョンリストを読む
. .\loadConfigInConfluence.ps1 -AuthenticationFile $AuthenticationFile -AliasFile $AliasFile -ConfigFile $ConfigFile
$versionAliasRule, $versionFnameAliasRule, $versionAliasArray, $versionFnameAliasArray = Create-VersionAliases_Confluence
$idAliasRule = Create-IdAliases_Confluence

#---------------- 英語／日本語
if ( $English -eq $true )
{
    $Lang = "En"
    $Lang2 = "Ja"
}
else
{
    $Lang = "Ja"
    $Lang2 = "En"
}

#----------------------------------------------------------------
# コンフィグファイルの読み込み
if ( "" -eq $ConfigFile )
{
    if ( $Document -eq $false )
    {
        if ( $English -eq $true )
        {
            $ConfigFile = ".\config\en\ChangelogConfig.xml"
        }
        else
        {
            $ConfigFile = ".\config\ja\ChangelogConfig.xml"
        }
    }
    else
    {
        if ( $English -eq $true )
        {
            $ConfigFile = ".\config\en\ChangelogDocumentConfig.xml"
        }
        else
        {
            $ConfigFile = ".\config\ja\ChangelogDocumentConfig.xml"
        }
    }
}

if ( -not ( Test-Path -path $ConfigFile ) )
{
    Write-Error( "Config file {0} not exists." -f $ConfigFile )
    exit 1
}
[Xml]$Config = Get-Content -Encoding UTF8 $ConfigFile

. ".\loadConfigRules.ps1"

# Packages に、履歴リスト
$Packages = [array](Make-PackageList( $Config ))

#----------------------------------------------------------------
#    使用する用語などを xml ファイルから取得する
#
. ".\makeTermRules.ps1"    # 外部スクリプト
$ruleFileName = $Config.Configuration.Template.Term
$categoryTerm, $priorityTerm, $otherTerm = Make-TermRules $ruleFileName

#----------------------------------------------------------------
#   プロジェクト名変換
. ".\convertProjectName.ps1"

#----------------------------------------------------------------
# パッケージ名に対してhtml名を取得
function Get-MainHtmlName( $pk )
{
    return "ChangeLog_" + ( $pk -creplace "\.", "_" -creplace " ","_" ) + ".html"
}
function Get-OldHtmlName( $pk )
{
    return "ChangeLog_" + ( $pk  -creplace "\.", "_" -creplace " ","_" ) + "_old.html"
}

#----------------------------------------------------------------
# カテゴリ (未選択だと対象外)
function Convert-Category( $c )
{
    if ( $categoryTerm.ContainsKey( [string]$c ) -eq $false )
    {
        return $categoryTerm[ "-1" ]
    }
    return $categoryTerm[ [string]$c ]
}

function Convert-CategoryToClass( $c )
{
    switch( $c )
    {
        1 { return "addition_style" }
        2 { return "improvement_style" }
        3 { return "bugfix_style" }
        0 { return "" }
        default { return "" }
    }
}
#----------------------------------------------------------------
# 重要度
function Convert-Priority( $c )
{
    if ( $priorityTerm.ContainsKey( [string]$c ) -eq $false )
    {
        return $priorityTerm[ "-1" ]
    }
    return $priorityTerm[ [string]$c ]
}

#----------------------------------------------------------------
# パッケージ (未選択だと対象外)
function Convert-Package( $c )
{
    return $c
}
#----------------------------------------------------------------
# タイトル  (未入力だと対象外)
function Convert-Title( $c )
{
    $c = $c  -replace '&','&amp;' -replace '<','&lt;' -replace '\\\\','\'

    #---- <code>...</code> 処理
    #     &lt;code> ～ &lt;/code> を見つけたらそれは戻す
    $c0 = $c
    $c = ""
    while($true)
    {
        if ( $c0 -match "(.*?)&lt;code>(.*?)&lt;/code>(.*)" )
        {
            $c = $c + $matches[1] + "<code>" + $matches[2] + "</code>"
            $c0 = $matches[3]
        }
        else
        {
            $c = $c + $c0
            break
        }
    }

    #---- <strong>...</strong> 処理
    #     &lt;strong> ～ &lt;/strong> を見つけたらそれは戻す
    $c0 = $c
    $c = ""
    while($true)
    {
        if ( $c0 -match "(.*?)&lt;strong>(.*?)&lt;/strong>(.*)" )
        {
            $c = $c + $matches[1] + "<strong>" + $matches[2] + "</strong>"
            $c0 = $matches[3]
        }
        else
        {
            $c = $c + $c0
            break
        }
    }

    return $c
}
#----------------------------------------------------------------
# 内容   (未入力だと対象外)
#        (リンクが必要なタイトル文字列にも適用する)
function Convert-Contents( $c, $isLink )
{
    #---- リンク候補
    $c = $c -replace '&','&amp;' -replace '<','&lt;' -replace '\\r\\n',"<br>" -replace '\\\\','\'

    #---- <code>...</code> 処理
    #     &lt;code> ～ &lt;/code> を見つけたらそれは戻す
    $c0 = $c
    $c = ""
    while($true)
    {
        if ( $c0 -match "(.*?)&lt;code>(.*?)&lt;/code>(.*)" )
        {
            $c = $c + $matches[1] + "<code>" + $matches[2] + "</code>"
            $c0 = $matches[3]
        }
        else
        {
            $c = $c + $c0
            break
        }
    }

    #---- <strong>...</strong> 処理
    #     &lt;strong> ～ &lt;/strong> を見つけたらそれは戻す
    $c0 = $c
    $c = ""
    while($true)
    {
        if ( $c0 -match "(.*?)&lt;strong>(.*?)&lt;/strong>(.*)" )
        {
            $c = $c + $matches[1] + "<strong>" + $matches[2] + "</strong>"
            $c0 = $matches[3]
        }
        else
        {
            $c = $c + $c0
            break
        }
    }

    #---- {{nolink}}処理
    $c0 = $c
    $c = ""
    while($true)
    {
        if ( $c0 -match "(.*?){{NoLink}}(.*?){{/NoLink}}(.*)" )
        {
            $dummyStr = $matches[2] -creplace "NN","@@DOUBLEN2@@" -creplace "nn","@@doublen2@@"
            $c = $c + $matches[1] + $dummyStr
            $c0 = $matches[3]
        }
        else
        {
            $c = $c + $c0
            break
        }
    }

    #---- {{ApiLink}}処理
    $c0 = $c
    $c = ""
    while($true)
    {
        if ( $c0 -match "(.*?){{ApiLink=(.*?)}}(.*?){{/ApiLink}}(.*)" )
        {
            if ( $isLink -eq $true )
            {
                $linkStr = "ApiLink_" + ($matches[2].Trim() -replace ":","_" -creplace "NN","@@DOUBLEN2@@" -creplace "nn","@@doublen2@@")
                $dummyStr = $matches[3].Trim() -creplace "NN","@@DOUBLEN2@@" -creplace "nn","@@doublen2@@"
                $c = $c + $matches[1] + "<span class=`"$linkStr`">" + $dummyStr + "</span>"
                $c0 = $matches[4]
            }
            else
            {
                $c = $c + $matches[1] + $matches[3]
                $c0 = $matches[4]
            }
        }
        else
        {
            $c = $c + $c0
            break
        }
    }

    #---- リンク候補を探す
    if ( $isLink -eq $true )
    {
        $c0 = $c
        $c = ""
        while($true)
        {
            if ( $c0 -cmatch "(.*?)(nn::[a-zA-Z0-9_:]*)(.*)" )
            {
                $c = $c + $matches[1] + "<span class=`"ApiLink_" + ($matches[2] -replace ":","_") + "`">" + $matches[2] + "</span>"
                $c0 = $matches[3]
            }
            elseif ( $c0 -cmatch "(.*?)(NN_[A-Z0-9_]*)(.*)" )
            {
                $c = $c + $matches[1] + "<span class=`"ApiLink_" + ($matches[2] -replace ":","_") + "`">" + $matches[2] + "</span>"
                $c0 = $matches[3]
            }
            else
            {
                $c = $c + $c0
                break
            }
        }
    }

    #---- ドキュメントリンクを探す
    $c0 = $c
    $c = ""
    while($true)
    {
        if ( $c0 -cmatch "(.*?){{DocLink=(.*?)}}(.*?){{/DocLink}}(.*)" )
        {
            if ( $isLink -eq $true )
            {
                $c = $c + $matches[1] + "<a href=`"" + $PreDocUrl + "Page_" + $matches[2].Trim() + ".html`" target=`"_top`">" + $matches[3] + "</a>"
                $c0 = $matches[4]
            }
            else
            {
                $c = $c + $matches[1] + $matches[3]
                $c0 = $matches[4]
            }
        }
        else
        {
            $c = $c + $c0
            break
        }
    }

    $c = $c -creplace "@@DOUBLEN2@@","NN" -creplace "@@doublen2@@","nn"
    return $c
}

#----------------------------------------------------------------
function Convert-EachLine( $c, $isLink )
{
    [string[]]$lines = @()
    $ret = ""
    $isPre = 0

    foreach ($s in ($c -split "\\r\\n") )
    {
        if ( $s -imatch "^{{pre" )
        {
            if ( $s -imatch "^{{pre,[ ]*wrap[ ]*}}" )
            {
                $ret += "</p><pre class=`"release_note2`">"
            }
            elseif ( $s -imatch "^{{pre,[ ]*nowrap[ ]*}}" )
            {
                $ret += "</p><pre class=`"release_note`">"
            }
            else
            {
                $ret += "</p><pre class=`"release_note2`">"
            }
            $isPre = 1
        }
        elseif ( $s -imatch "^{{/pre}}" )
        {
            $ret += "</pre><p class=`"release_note`">"
            $isPre = 0
        }
        elseif ( $isPre -eq 1 )
        {
            $s = $s -replace '&','&amp;' -replace '<','&lt;' -replace '\\r\\n',"<br>" -replace '\\\\','\'
            if ( $s -match "^-(.*)" )
            {
              $s = "&#45;" + $matches[1]
            }
            $ret += $s + "<br>"
        }
        else
        {
            $ret += (Convert-Contents $s $isLink) + "<br>"
        }
    }

    # preを閉じてない場合
    if ( $isPre -ne 0 )
    {
        $ret += "</pre><p class=`"release_note`">"
    }
    return $ret
}

#----------------------------------------------------------------
# 箇条書きの考慮
function Convert-Itemization( $c )
{
    [string[]]$lines = @()
    $level = 0

    foreach ($s in ($c -split "<br>") )
    {
        if ( $s -match "^---" )
        {
            if ( $level -eq 0 )
            {
                if ( $lines.Length -ge 1 )
                {
                    $lines[$lines.Length - 1] = ""
                }
                $lines += "</p><ul><ul><ul>"
            }
            elseif ( $level -eq 1 )
            {
                $lines += "<ul><ul>"
            }
            elseif ( $level -eq 2 )
            {
                $lines += "<ul>"
            }
            $level = 3
            $lines += "<li>" + ($s -replace "^---","") + "</li>"
        }
        elseif ( $s -match "^--" )
        {
            if ( $level -eq 0 )
            {
                if ( $lines.Length -ge 1 )
                {
                    $lines[$lines.Length - 1] = ""
                }
                $lines += "</p><ul><ul>"
            }
            elseif ( $level -eq 1 )
            {
                $lines += "<ul>"
            }
            elseif ( $level -eq 3 )
            {
                $lines += "</ul>"
            }
            $level = 2
            $lines += "<li>" + ($s -replace "^--","") + "</li>"
        }
        elseif ( $s -match "^-" )
        {
            if ( $level -eq 0 )
            {
                if ( $lines.Length -ge 1 )
                {
                    $lines[$lines.Length - 1] = ""
                }
                $lines += "</p><ul>"
            }
            elseif ( $level -eq 2 )
            {
                $lines += "</ul>"
            }
            elseif ( $level -eq 3 )
            {
                $lines += "</ul></ul>"
            }
            $level = 1
            $lines += "<li>" + ($s -replace "^-","") + "</li>"
        }
        else
        {
             if ( $level -eq 3 )
             {
                 $lines += "</ul></ul></ul><p class=`"release_note`">"
             }
             elseif ( $level -eq 2 )
             {
                 $lines += "</ul></ul><p class=`"release_note`">"
             }
             elseif ( $level -eq 1 )
             {
                 $lines += "</ul><p class=`"release_note`">"
             }
             $level = 0
             $lines += $s
             $lines += "<br>"
        }
    }

    if ( $level -eq 3 )
    {
        $lines += "</ul></ul></ul><p class=`"release_note`">"
    }
    elseif ( $level -eq 2 )
    {
        $lines += "</ul></ul><p class=`"release_note`">"
    }
    elseif ( $level -eq 1 )
    {
        $lines += "</ul><p class=`"release_note`">"
    }

    return ($lines -join "")
}

#----------------------------------------------------------------
# 日付文字列
function Convert-DateStr( $dt )
{
    $month = "Jan","Feb","Mar","Apr","Mar","Jun","Jul","Aug","Sep","Oct","Nov","Dec"

    if ( $English -eq $true )
    {
        return $month[ [int]($dt.Substring(4,2)) - 1 ] + " " + $dt.Substring(6,2) + " " + $dt.Substring(0,4)
    }
    else
    {
        return $dt.Substring(0,4) + "年" + $dt.Substring(4,2) + "月" + $dt.Substring(6,2) + "日"
    }
}
#----------------------------------------------------------------
# 各行の処理
#     CVS纏めデータへの追加
#     isCheckHash が $true なら、同じハッシュを探しすでにあるならスキップする
#
function Add-Eachline( $line, $isCheckHash )
{
    #---- 行頭シャープならコメント
    if ( $line -match "^[ 	]*#" )
    {
        return
    }

    #---- その行を CSVデータとして分割
    $c = $line | ConvertFrom-Csv -Delimiter "," -Header HASH,LANG,KEY,CATEGORY,PRIORITY,PACKAGE,COMPONENT_JA,COMPONENT_EN,COMPONENT_CAT,TITLE,CONTENT,FIXVERSION,DATE

    #---- 除外判定
    if ( $c.CATEGORY -eq "None" -or $c.PRIORITY -eq "None" -or $c.PACKAGE -eq "None" -or $c.COMPONENT_JA -eq "None" -or $c.TITLE -eq "None" -or $c.CONTENT -eq "None" -or $null -eq $c.CONTENT )
    {
        return
    }

    #---- ハッシュ重複チェック
    if ( $isCheckHash -eq $true )
    {
        $isSkip = $false
        $script:AllCsv | ForEach-Object `
        {
            # すでに同じハッシュのものが含まれていたらスキップ
            if ( $_.HASH -eq $c.HASH )
            {
                Write-Detail( "FOUND SAME HASH={1}, SKIP." -f $c.HASH, $_.HASH )
                $isSkip = $true
                return
            }
        }
        if ( $isSkip -eq $true )
        {
            return
        }
    }

    $script:AllCsv += $c
}

#----------------------------------------------------------------
# 変更履歴作成(ライブラリパッケージ向け)
#
function Create-ContentsList2( $subject, $packageName, [ref]$topList1, [ref]$mainList1, [ref]$mainList2, $argIdPrefix, $idx, $linkTargetFilePath )
{
    #---- リンク文字列
    $LinkStr = ("{0}" -f $packageName) -replace '[ \.]','_' -replace '[\(\)\[\]\{\}]',''

	$script:LogListIndex = 1

    #---- 対象パッケージの該当リストのみ取り出す
    foreach( $ct in @( 0, 1, 2, 3 ) )  # 0:重要度が"3"のもの 1-3:カテゴリが1-3のもの、と0だけ意味が違います
    {
        if ( $ct -eq "0" )
        {
            $partCsv = $AllCsv | where{$_.PRIORITY -eq "3" }
            $partCsv = $partCsv | sort COMPONENT_CAT,COMPONENT_JA,TITLE               #---- コンポーネントを考慮しつつタイトルソート

            #---- カテゴリ文字列
            $CategoryStr = $otherTerm["nocomp"]  # "互換性の失われる仕様変更"
            $CategoryClass = "nocomp_style"
        }
        else
        {
            #---- 対象の CVSデータ
            $partCsv = $AllCsv | where{$_.PRIORITY -ne "3" -and $_.CATEGORY -eq $ct }
            $partCsv = $partCsv | sort COMPONENT_CAT,COMPONENT_JA,TITLE               #---- コンポーネントを考慮しつつタイトルソート

            #---- カテゴリ文字列(「不具合修正」「改善」「機能追加」)
            $CategoryStr = Convert-Category( $ct )
            $CategoryClass = Convert-CategoryToClass( $ct )
        }

        if ( $null -eq $partCsv )
        {
            continue
        }

        # topファイルのリスト
        $topList1.value += "  <h3 class=`"release_note`"><span class=`"{1}`"></span>{0}</h3>" -f $CategoryStr, $CategoryClass
        $topList1.value += "  <ul style=`"list-style-type:none;`" class=`"ul_rn`">"

        # mainファイルのリスト
        $mainList1.value += "<h3 id=`"catlist_{1}_{2}`" class=`"release_note`"><span class=`"{3}`"></span>{0}</h3>" -f $CategoryStr, $LinkStr, $ct, $CategoryClass
        $mainList1.value += "<ul style=`"list-style-type:none;`" class=`"ul_rn`">"

        # mainファイルの本体部
        $mainList2.value += "<h2 id=`"category_{1}_{2}`" class=`"release_note`"><span class=`"{3}`"></span>{0}</h2>" -f $CategoryStr, $LinkStr, $ct, $CategoryClass

        # 重要なものを先にする
        foreach( $prio in @( 3, 2, 1 ) )
        {
            $partCsv | where { $_.PRIORITY -eq $prio } |
            Foreach-object `
            {
                #---- インデックス生成
                $LogId = "{0}{1}" -f $argIdPrefix, ($script:LogListIndex).ToString("00")
                $script:LogListIndex ++
                Write-Detail( "** Create {0} {1}" -f $LogId, $_.TITLE )

                #---- 重要度の考慮
                if ( $_.PRIORITY -eq "2" -or $_.PRIORITY -eq "3" )
                {
                    $important1 = "<span class=`"important_style`"> {0}" -f $otherTerm["importantMark"]
                    $important2 = "</span>"
                }
                else
                {
                    $important1 = ""
                    $important2 = ""
                }

                #---- コンポーネント選択
                if ( $English -eq $true )
                {
                    $ComponentStr = $_.COMPONENT_EN
                    $ComponentDispStr = $_.COMPONENT_EN
                }
                else
                {
                    $ComponentStr = $_.COMPONENT_JA
                    $ComponentDispStr = $_.COMPONENT_JA
                }
                #---- コンポーネントが確定されていない場合目立たせる
                if ( $ComponentDispStr -match "not firm" )
                {
                    $ComponentDispStr = "<font color=`"red`">$ComponentDispStr</font>"
                }

                Write-Detail( $_.KEY,$LogId,$_.CATEGORY,$_.PRIORITY,$_.PACKAGE,$ComponentStr,$_.TITLE,$_.CONTENT )

                #---- キー変換
                $KeyComment = Convert-ProjectName $_.KEY

                #---- タイトルと内容の整形
                $html_title   = Convert-Contents $_.TITLE $false
                $html_content = "<p class=`"release_note`">" + (Convert-Itemization (Convert-EachLine $_.CONTENT $true)) + "</p>"

                #---- topファイルのリスト
                $topList1.value += "    <li><a href=`"{8}#note_{7}`">{3}{0}: ({1}) {2}{4}</a></li><!--{6}-->" -f
                                                 $LogId, $ComponentDispStr, $html_title, $important1, $important2, $script:OutputName_MainHtml, $KeyComment, $_.HASH, $linkTargetFilePath

                #---- mainファイルのリスト
                $mainList1.value += "<li><a href=`"#note_{6}`">{3}{0}: ({1}) {2}{4}</a></li><!--{5}-->" -f
                                                 $LogId, $ComponentDispStr, $html_title, $important1, $important2, $KeyComment, $_.HASH

                #---- mainファイルの本体部
                $mainList2.value += "<h3 id=`"note_{6}`" class=`"release_note`"><span class=`"{5}`"></span>{3}{0}: ({1}) {2}{4}</h3>" -f
                                                 $LogId, $ComponentDispStr, $html_title, $important1, $important2, $CategoryClass, $_.HASH
                $mainList2.value += "{0}<!--{1}-->" -f
                                                 $html_content, $KeyComment

                $hash = $_.HASH
                $ComCat = $_.COMPONENT_CAT # ソートにのみ使う
                $ComJa = $_.COMPONENT_JA # ソートにのみ使う
                $c = ("$LogId,$IndexPList,$VersionForCsvNum,$VersionForCsv,$ComponentStr,$ComCat,$ComJa,{0},$hash,$ct,$prio,{1}" `
                      -f ($html_title -replace ',','@@_COMMA_@@'), ($html_content -replace ',','@@_COMMA_@@') ) | `
                      ConvertFrom-Csv -Delimiter "," -Header ID,PACKAGENUM,VERSIONNUM,VERSION,COMPONENT,COMPONENT_CAT,COMPONENT_JA,TITLE,HASH,CATEGORY,PRIO,CONTENT
                $script:ListCsv += $c
            }
        }
        $topList1.value += "  </ul>"
        $mainList1.value += "  </ul>"
    }
}

#----------------------------------------------------------------
#  旧ログ(ライブラリパッケージ)
#
function Create-OldContentsList( $mainPackageName, $packageName, $argVersion, $argSupplementation, $argIdPrefix  )
{
    if ( $AllCsv.Length -eq 0 )
    {
        return
    }

    #---- リンク文字列
    $LinkStr = ("{0}_{1}" -f $packageName, $argVersion) -replace '[ \.]','_' -replace '[\(\)\[\]\{\}]',''

    #---- バージョン
    if ( $English -ne $true )
    {
       $script:ConVerListHtml += "<li><a href=`"#list_{3}`">{0} {1} の変更点 {2}</a></li>" -f $mainPackageName, $argVersion, $argSupplementation, $LinkStr
       $script:ConOldListHtml += "<h2 id=`"list_{3}`" class=`"list`"><a href=`"#detail_{3}`">{0} {1} {2}</a></h2>" -f $mainPackageName, $argVersion, $argSupplementation, $LinkStr
       $script:ConBodyHtml += "<h1 id=`"detail_{3}`">{0} {1} の変更点<br>{2}</h1>" -f $mainPackageName, $argVersion, $argSupplementation, $LinkStr
    }
    else
    {
       $script:ConVerListHtml += "<li><a href=`"#list_{3}`">Changes in {0} {1} {2}</a></li>" -f $mainPackageName, $argVersion, $argSupplementation, $LinkStr
       $script:ConOldListHtml += "<h2 id=`"list_{3}`" class=`"list`"><a href=`"#detail_{3}`">{0} {1} {2}</a></h2>" -f $mainPackageName, $argVersion, $argSupplementation, $LinkStr
       $script:ConBodyHtml += "<h1 id=`"detail_{3}`">Changes in {0} {1} <br>{2}</h1>" -f $mainPackageName, $argVersion, $argSupplementation, $LinkStr
    }

    #---- 対象パッケージの該当リストのみ取り出す
    foreach( $ct in @( 0, 1, 2, 3 ) )  # 0:重要度が"3"のもの 1-3:カテゴリが1-3のもの、と0だけ意味が違います
    {
        if ( $ct -eq "0" )
        {
            $partCsv = $AllCsv | where{$_.PRIORITY -eq "3"}
            $partCsv = $partCsv | sort COMPONENT_CAT,COMPONENT_JA,TITLE               #---- コンポーネントを考慮しつつタイトルソート

            #---- カテゴリ文字列
            $CategoryStr = $otherTerm["nocomp"]  # "互換性の失われる仕様変更"
            $CategoryClass = "nocomp_style"
        }
        else
        {
            #---- 対象の CVSデータ
            $partCsv = $AllCsv | where{$_.PRIORITY -ne "3" -and $_.CATEGORY -eq $ct}
            $partCsv = $partCsv | sort COMPONENT_CAT,COMPONENT_JA,TITLE               #---- コンポーネントを考慮しつつタイトルソート

            #---- カテゴリ文字列(「不具合修正」「改善」「機能追加」)
            $CategoryStr = Convert-Category( $ct )
            $CategoryClass = Convert-CategoryToClass( $ct )
        }

        if ( $null -eq $partCsv )
        {
            continue
        }

        # mainファイルのリスト
        $script:ConOldListHtml += "<h3 id=`"catlist_{1}_{2}`" class=`"release_note`"><span class=`"{3}`"></span>{0}  <font size=`"1`"><a href=`"#category_{1}_{2}`">&gt;</a></font></h3>" -f $CategoryStr, $LinkStr, $ct, $CategoryClass
        $script:ConOldListHtml += "<ul style=`"list-style-type:none;`" class=`"ul_rn`">"

        # mainファイルの本体部
        $script:ConBodyHtml += "<h2 id=`"category_{1}_{2}`" class=`"release_note`"><span class=`"{3}`"></span>{0}  <font size=`"1`"><a href=`"#catlist_{1}_{2}`">&lt;</a></font></h2>" -f $CategoryStr, $LinkStr, $ct, $CategoryClass

        # 重要なものを先にする
        foreach( $prio in @( 3, 2, 1 ) )
        {
            $partCsv | where { $_.PRIORITY -eq $prio } |
            Foreach-object `
            {
                #---- インデックス生成
                $LogId = "{0}{1}" -f $argIdPrefix, ($script:LogListIndex).ToString("00")
                $script:LogListIndex ++
                Write-Detail( "** Create {0} {1}" -f $LogId, $_.TITLE )

                #---- 重要度の考慮
                if ( $_.PRIORITY -eq "2" -or $_.PRIORITY -eq "3" )
                {
                    $important1 = "<span class=`"important_style`"> {0}" -f $otherTerm["importantMark"]
                    $important2 = "</span>"
                }
                else
                {
                    $important1 = ""
                    $important2 = ""
                }

                #---- コンポーネント選択
                if ( $English -eq $true )
                {
                    $ComponentStr = $_.COMPONENT_EN
                    $ComponentDispStr = $_.COMPONENT_EN
                }
                else
                {
                    $ComponentStr = $_.COMPONENT_JA
                    $ComponentDispStr = $_.COMPONENT_JA
                }
                #---- コンポーネントが確定されていない場合目立たせる
                if ( $ComponentDispStr -match "not firm" )
                {
                    $ComponentDispStr = "<font color=`"red`">$ComponentDispStr</font>"
                }

                Write-Detail( $_.KEY,$LogId,$_.CATEGORY,$_.PRIORITY,$_.PACKAGE,$ComponentStr,$_.TITLE,$_.CONTENT )

                #---- キー変換
                $KeyComment = Convert-ProjectName $_.KEY

                #---- タイトルと内容の整形
                $html_title   = Convert-Contents $_.TITLE $false
                $html_content = "<p class=`"release_note`">" + (Convert-Itemization (Convert-EachLine $_.CONTENT $false)) + "</p>"

                #---- mainファイルのリスト
                $script:ConOldListHtml += "<li><a href=`"#note_{6}`">{3}{0}: ({1}) {2}{4}</a></li><!--{5}-->" -f
                                                 $LogId, $ComponentDispStr, $html_title, $important1, $important2, $KeyComment, $_.HASH

                #---- mainファイルの本体部
                $script:ConBodyHtml += "<h3 id=`"note_{6}`" class=`"release_note`"><span class=`"{5}`"></span>{3}{0}: ({1}) {2}{4}</h3>" -f
                                                 $LogId, $ComponentDispStr, $html_title, $important1, $important2, $CategoryClass, $_.HASH
                $script:ConBodyHtml += "{0}<!--{1}-->" -f
                                                 $html_content, $KeyComment
            }
        }
        $script:ConOldListHtml += "</ul>"
    }
}

#----------------------------------------------------------------
# 変更履歴作成(ドキュメントパッケージ向け)
#
function Create-ContentsListForDocument( $subject, $packageName, $mainHtml, $idx, $argIdPrefix )
{
    $script:ConDescHtml += "<h2 id=`"module_idx_0_0_0_{0}`"> $subject </h2>" -f $idx

    #---- 登場する日付
    $existedDate = $AllCsv | sort -descending DATE | Select-Object DATE | foreach-object { $_.DATE } | Get-Unique
    if ( $existedDate -eq $null )
    {
        if ( $English -eq $false )
        {
            $script:ConListHtml += "ありません。"
            $script:ConDescHtml += "ありません。"
            $script:ConBodyHtml += "<p>まだありません。今後情報はここへ記載されていきます。</p>"
        }
        else
        {
            $script:ConListHtml += "No log."
            $script:ConDescHtml += "No log."
            $script:ConBodyHtml += "<p>No log.</p>"
        }
        return
    }

    foreach( $dt in $existedDate )
    {
        $script:LogListIndex = 1

        #---- 対象日付のみ取り出す
        $partCsv = $AllCsv | where{ $_.DATE -eq $dt}

        $dtStr = Convert-DateStr $dt

        $template = Get-Content -Encoding UTF8 -Path $Config.Configuration.Template.DocContentsList
        $template = $template `
            -replace "_\$\$@\(SUB_PACKAGE_NAME\)", $PackageName `
            -replace "_\$\$@\(DATE_VAL\)", $dt `
            -replace "_\$\$@\(DATE_STR\)", $dtStr
        $script:ConListHtml += $template

        # topファイルのリスト
        $script:ConDescHtml += "  <h3 id=`"date_{0}_{1}`">{2}</h3>" -f $packageName, $dt, $dtStr
        $script:ConDescHtml += "  <ul style=`"list-style-type:none;`" class=`"ul_rn`">"

        # mainファイルのリスト
        $script:ConBodyListHtml += "<ul class=`"itemlist_rn`"><li><a href=`"#date_{1}`">{0}</a>" -f $dtStr, $dt
        $script:ConBodyListHtml += " <ul style=`"list-style-type:none;`" class=`"ul_rn`">"

        # mainファイルの本体部
        $script:ConBodyHtml += "<h2 id=`"date_{1}`">{0}</h2>" -f $dtStr, $dt

        # 重要なものを先にする
        foreach( $prio in @( 3, 2, 1 ) )
        {
            $partCsv | where { $_.PRIORITY -eq $prio } |
            Foreach-object `
            {
                #---- インデックス生成
                if ( $null -eq $_.LOGID -or "" -eq $_.LOGID )
                {
                    $LogId = "{0}{1}-{2}" -f $argIdPrefix, $dt.Substring(2,6), ($script:LogListIndex).ToString("00")
                }
                else
                {
                    $LogId = $_.LOGID
                }

                $script:LogListIndex ++
                Write-Detail( "** Create {0} {1}" -f $LogId, $_.TITLE )
                Write-Log( "** Create  $LogId." -f $LogId )

                #---- 重要度の考慮
                if ( $_.PRIORITY -eq "2" -or $_.PRIORITY -eq "3" )
                {
                    $important1 = "<span class=`"important_style`"> {0}" -f $otherTerm["importantMark"]
                    $important2 = "</span>"
                }
                else
                {
                    $important1 = ""
                    $important2 = ""
                }

                #---- コンポーネント選択
                if ( $English -eq $true )
                {
                    $ComponentStr = $_.COMPONENT_EN
                }
                else
                {
                    $ComponentStr = $_.COMPONENT_JA
                }

                #---- キー変換
                $KeyComment = Convert-ProjectName $_.KEY

                #---- タイトルと内容の整形
                $html_title   = Convert-Title $_.TITLE
                $html_content = "<p class=`"release_note`">" + (Convert-Itemization (Convert-Contents $_.CONTENT $true)) + "</p>"

                #---- topファイルのリスト
                $script:ConDescHtml += "    <li><a href=`"ReleaseNotes/{5}#note_{7}`">{3}{0}: ({1}) {2}{4}</a></li><!--{6}-->" -f
                                                 $LogId, $ComponentStr, $html_title, $important1, $important2, $mainHtml, $KeyComment, $_.HASH

                #---- mainファイルのリスト
                $script:ConBodyListHtml += "  <li><a href=`"#note_{6}`">{3}{0}: ({1}) {2}{4}</a></li><!--{5}-->" -f
                                                 $LogId, $ComponentStr, $html_title, $important1, $important2, $KeyComment, $_.HASH

                #---- mainファイルの本体部
                $script:ConBodyHtml += "<h3 id=`"note_{6}`" class=`"release_note`"><span class=`"{5}`"></span>{3}{0}: ({1}) {2}{4}</h3>" -f
                                                 $LogId, $ComponentStr, $html_title, $important1, $important2, $CategoryClass, $_.HASH
                $script:ConBodyHtml += "{0}<!--{1}-->" -f
                                                 $html_content, $KeyComment
            }
        }

        $script:ConDescHtml += "  </ul>"
        $script:ConBodyListHtml += "  </ul>"

        # mainファイルのリスト
        $script:ConBodyListHtml += "</ul>"
    }
}

#----------------------------------------------------------------
#  旧ログ(ドキュメントパッケージ)
#
function Create-OldContentsListForDocument( $packageName, $argVersion, $argSupplementation, $argIdPrefix  )
{
    if ( $AllCsv.Length -eq 0 )
    {
        return
    }

    #---- 登場する日付
    $existedDate = $AllCsv | sort -descending DATE | Select-Object DATE | foreach-object { $_.DATE } | Get-Unique
    if ( $existedDate -eq $null )
    {
        return
    }

    foreach( $dt in $existedDate )
    {
        $script:LogListIndex = 1

        #---- 対象日付のみ取り出す
        $partCsv = $AllCsv | where{ $_.DATE -eq $dt}

        $dtStr = Convert-DateStr $dt

        # mainファイルのリスト
        $script:ConOldListHtml += "<ul class=`"itemlist_rn`"><li><a href=`"#date_{1}`">{0}</a>" -f $dtStr, $dt
        $script:ConOldListHtml += " <ul style=`"list-style-type:none;`" class=`"ul_rn`">"

        # mainファイルの本体部
        $script:ConBodyHtml += "<h2 id=`"date_{1}`">{0}</h2>" -f $dtStr, $dt

        # 重要なものを先にする
        foreach( $prio in @( 3, 2, 1 ) )
        {
            $partCsv | where { $_.PRIORITY -eq $prio } |
            Foreach-object `
            {
                #---- インデックス生成
                #$LogId = "{0}{1}" -f $argIdPrefix, ($script:LogListIndex).ToString("00")
                if ( $null -eq $_.LOGID -or "" -eq $_.LOGID )
                {
                    $LogId = "{0}{1}-{2}" -f $argIdPrefix, $dt.Substring(2,6), ($script:LogListIndex).ToString("00")
                }
                else
                {
                    $LogId = $_.LOGID
                }

                $script:LogListIndex ++
                Write-Detail( "** Create {0} {1}" -f $LogId, $_.TITLE )

                #---- 重要度の考慮
                if ( $_.PRIORITY -eq "2" -or $_.PRIORITY -eq "3" )
                {
                    $important1 = "<span class=`"important_style`"> {0}" -f $otherTerm["importantMark"]
                    $important2 = "</span>"
                }
                else
                {
                    $important1 = ""
                    $important2 = ""
                }

                #---- コンポーネント選択
                if ( $English -eq $true )
                {
                    $ComponentStr = $_.COMPONENT_EN
                }
                else
                {
                    $ComponentStr = $_.COMPONENT_JA
                }

                Write-Detail( $_.KEY,$LogId,$_.CATEGORY,$_.PRIORITY,$_.PACKAGE,$ComponentStr,$_.TITLE,$_.CONTENT )

                #---- キー変換
                $KeyComment = Convert-ProjectName $_.KEY

                #---- タイトルと内容の整形
                $html_title   = Convert-Title $_.TITLE
                $html_content = "<p class=`"release_note`">" + (Convert-Itemization (Convert-Contents $_.CONTENT $false)) + "</p>"

                #---- mainファイルのリスト
                $script:ConOldListHtml += "  <li><a href=`"#note_{6}`">{3}{0}: ({1}) {2}{4}</a></li><!--{5}-->" -f
                                                 $LogId, $ComponentStr, $html_title, $important1, $important2, $KeyComment, $_.HASH

                #---- mainファイルの本体部
                $script:ConBodyHtml += "<h3 id=`"note_{6}`" class=`"release_note`"><span class=`"{5}`"></span>{3}{0}: ({1}) {2}{4}</h3>" -f
                                                 $LogId, $ComponentStr, $html_title, $important1, $important2, $CategoryClass, $_.HASH
                $script:ConBodyHtml += "{0}<!--{1}-->" -f
                                                 $html_content, $KeyComment
            }
        }
        $script:ConOldListHtml += " </ul>"
        $script:ConOldListHtml += "</ul>"
    }
}

#--------------------------------------------------------------------------------
# 指定ファイルを CSVとして $AddCsv に追加
# isCheckHash が $true なら、同じハッシュを探しすでにあるならスキップする
#
function Add-CsvData( $f, $isCheckHash )
{
    Write-Detail( "** Processing {0}..." -f $f )

    #---- 各行に対して追加処理
    Get-Content $f | Foreach-Object `
    {
        Add-Eachline $_ $isCheckHash
    }
}

#--------------------------------------------------------------------------------
# 文字列の中の %DATE% を置換
#
function Replace-DateMacro( $str )
{
    if ( $str -match "(.*?)%DATE%(.*)" )
    {
        $str = $matches[1] + (Get-Date -Format "yyyy-MM-dd") + $matches[2]
    }
    return $str
}

#--------------------------------------------------------------------------------
# データ読み込み
#
function Read-CsvData( $isOutputDir, $isMainLanguage, $ProjectStr, $PackageStr, $VersionStr )
{
    if ( $isMainLanguage -eq $true )
    {
        $LangStr = $Lang
    }
    else
    {
        $LangStr = $Lang2
    }

    if ( $isOutputDir -eq $true )
    {
        $DataFile = "{0}\Outputs\Resources\ReleaseNotes\{1}\{2}_{3}_{4}_{5}.dat" -f $ResourceDir, $Lang, $ProjectStr, $PackageStr, $VersionStr, $LangStr
    }
    else
    {
        $DataFile = "{0}\Resources\ReleaseNotes\{1}\{2}_{3}_{4}_{5}.dat" -f $ResourceDir, $Lang, $ProjectStr, $PackageStr, $VersionStr, $LangStr
    }

    if ( Test-Path -path $DataFile )
    {
        Write-Log "** Read $DataFile ."
        Add-CsvData $DataFile $true
        return $true
    }

    return $false
}

#--------------------------------------------------------------------------------
# 各バージョンの変更履歴 HTML を出力
#
function Generate-EachVersionChangelog( [array]$VersionList, [string]$VersionString, [string]$Supplementation, [string]$OutputDirName, [string]$newVersion = "", [string]$oldVersion = "")
{
	[string[]]$WorkingContList = @()
	[string[]]$WorkingDescList = @()
	[string[]]$WorkingConEachVer = @()
	[string[]]$WorkingMixedConBodyList = @()
	[string[]]$WorkingConBodyList = @()
	[string[]]$WorkingVersionList = @()
	[string]$WorkingContentNew = ""
	[string]$WorkingContentOld = ""
	
	$WorkingContList = "<ul>"

	# この HTML ファイルの名前
	$eachVersionStr = $VersionList[0]  -replace "\.", "_"
	$eachVersionHtmlName = "Changelog-{0}.html" -f $eachVersionStr
	
	# ひとつ古いバージョンの HTML ファイルの名前
	$oldVersionStr = $oldVersion -replace "\.", "_"
	$oldVersionHtmlName = ""
	if ( "" -ne $oldVersionStr )
	{
		$oldVersionHtmlName = "Changelog-{0}.html" -f $oldVersionStr
		# TODO : 古いバージョンの HTML へのリンクを作る
		$WorkingContentOld = "<a href=`"{0}`">{1} &gt;&gt;</a>" -f $oldVersionHtmlName, ($oldVersion)
	}

	# ひとつ新しいバージョンの HTML ファイルの名前
	$newVersionStr = $newVersion -replace "\.", "_"
	$newVersionHtmlName = ""
	if ( "" -ne $newVersionStr )
	{
		$newVersionHtmlName = "Changelog-{0}.html" -f $newVersionStr
		# TODO : 新しいバージョンの HTML へのリンクを作る
		$WorkingContentNew = "<a href=`"{0}`">&lt;&lt; {1}</a>" -f $newVersionHtmlName, ($newVersion)
	}

	#---- 各パッケージを調べる
	$PackageListNum = Get-PackageListNum
	for( $IndexPList=0; $IndexPList -lt $PackageListNum; $IndexPList++ )
	{
		$CommonPackageName, $CommonSubject, $CommonIdPrefix, $CommonName = Get-PackageListCommonInfo( $IndexPList )

		$s_, $Versions, $IdPrefix, $Subject, $v_ = Get-PackageListPackageInfo $IndexPList 0

		#---- 対象パッケージに関するファイル名
		$OutputName_MainHtml = Get-MainHtmlName $CommonPackageName
		$OutputName_OldHtml  = Get-OldHtmlName $CommonPackageName

		if ( $IdPrefix -eq $null )
		{
			$IdPrefix = (Filter-AliasStr $idAliasRule $VersionList[0]) + $CommonIdPrefix
		}
		else
		{
			$IdPrefix = (Filter-AliasStr $idAliasRule $VersionList[0]) + $IdPrefix
		}

		$script:AllCsv = @()
		foreach ($cname in $CommonName)
		{
			if ( $cname.Length -gt 0 )
			{
				$cname0 = "SIGLO"
				$cname1 = $cname
			}
			else
			{
				$cname0 = $cname.Project
				$cname1 = ($cname."#text")
			}

			foreach ($vname in $VersionList)
			{
				# Write-Host("[ Loop ] : {0}, {1}, {2}" -f $IndexPList, $cname.Project, $vname)
				$cname1 = Filter-AliasStr $packageAliasRule $cname1
				$vname = Filter-AliasStr $versionAliasRule $vname
				$VersionForCsvNum = $i.ToString("000")
				$VersionForCsv = $vname

				$ProjectStr = Filter-AliasStr $projectFnameAliasRule $cname0
				$PackageStr = Filter-AliasStr $packageFnameAliasRule $cname1
				$VersionStr = Filter-AliasStr $versionFnameAliasRule $vname

				# (主言語データの読み込み)
				$r = Read-CsvData $true $true $ProjectStr $PackageStr $VersionStr
				if ( ! $r )
				{
					$r = Read-CsvData $false $true $ProjectStr $PackageStr $VersionStr
				}
				# (副言語データの読み込み)
				$r = Read-CsvData $true $false $ProjectStr $PackageStr $VersionStr
				if ( ! $r )
				{
					$r = Read-CsvData $false $false $ProjectStr $PackageStr $VersionStr
				}
			}
		}

		if ( $script:AllCsv.Length -ne 0 )
		{
			# Write-Host "[" $CommonSubject "]" "  IdStr=" $CommonIdPrefix "   Packages=" $CommonName
			# Write-Host ("{0} log gathered." -f $AllCsv.Length)
			if ( $Subject -eq $null )
			{
				$Subject = $CommonSubject
			}
			if ( $VersionString -eq $null )
			{
				$VersionString = $VersionList[0]
			}

			if ( $English -eq $true )
			{
				$WorkingContList += "  <li><a href=`"#module_idx_{0}_0_0_{1}`">Changes in {2}</a></li>" -f ($i + 1), ($IndexPList + 1), $Subject
				$WorkingDescList += "<h2 id=`"module_idx_{0}_0_0_{1}`"> {2} </h2>" -f ($i + 1), ($IndexPList + 1), $Subject
				$WorkingMixedConBodyList += "<h2 id=`"ret_{0}`" class=`"list`"><a href=`"#list_contents_{0}`"> {2} {3} {4}</a></h2>" -f ($i + 1), $IndexPList, $Subject, $VersionString, $Supplementation
				$WorkingConBodyList += "<h1 id=`"list_contents_{0}`">Changes in  {2} {3}</h1>" -f ($i + 1), $IndexPList, $Subject, $VersionString, $Supplementation
				$WorkingVersionList += "   <li><a href=`"#ret_{0}`">Changes in {2} {3} {4}</a></li>" -f ($i+1), $IndexPList, $Subject, $VersionString, $Supplementation
			}
			else
			{
				$WorkingContList += "  <li><a href=`"#module_idx_{0}_0_0_{1}`">{2} の変更点</a></li>" -f ($i + 1), ($IndexPList + 1), $Subject
				$WorkingDescList += "<h2 id=`"module_idx_{0}_0_0_{1}`"> {2} </h2>" -f ($i + 1), ($IndexPList + 1), $Subject
				$WorkingMixedConBodyList += "<h2 id=`"ret_{0}`" class=`"list`"><a href=`"#list_contents_{0}`"> {2} {3} {4}</a></h2>" -f ($i + 1), $IndexPList, $Subject, $VersionString, $Supplementation
				$WorkingConBodyList += "<h1 id=`"list_contents_{0}`"> {2} {3} の変更点</h1>" -f ($i + 1), $IndexPList, $Subject, $VersionString, $Supplementation
				$WorkingVersionList += "   <li><a href=`"#ret_{0}`"> {2} {3} の変更点 {4}</a></li>" -f ($i+1), $IndexPList, $Subject, $VersionString, $Supplementation
			}

			#---- リンク追加
			$LogListIndex = 1
			Create-ContentsList2 $Subject $CommonPackageName ([ref]$WorkingDescList) ([ref]$WorkingMixedConBodyList) ([ref]$WorkingConBodyList) $IdPrefix $IndexPList
			$script:LogNum += ($script:LogListIndex - 1)
			Write-Host ("Working Desc List {0}" -f $WorkingDescList)
		}
	}
	$WorkingContList += "</ul>"
	$WorkingDescList += $WorkingConBodyList
		
	#---- 各バージョン HTML のテンプレート読み込み
	$TemplateName_EachHtml = $Config.Configuration.Template.EachHtml
	$template = Get-Content -Encoding UTF8 -Path $TemplateName_EachHtml

	#---- テンプレート変数の置き換え
	$template = $template `
		-replace "_\$\$@\(PACKAGE_NAME\)", $VersionString `
		-replace "_\$\$@\(RELEASE_DATE\)", $Supplementation `
		-replace "_\$\$@\(RELEASE_YEAR\)", $ReleaseYear `
		-replace "_\$\$@\(TOP_INFORMATION\)", $Config_TopInformation `
		-replace "_\$\$@\(PACKAGE_SUPPLEMENTATION\)", $Config_PackageSupplementation `
		-replace "_\$\$@\(CONTENTS_LIST\)", [string]::join("`r`n", $WorkingContList) `
		-replace "_\$\$@\(CONTENTS_EACH_VERSIONS\)", [string]::join("`r`n", $WorkingConEachVer ) `
		-replace "_\$\$@\(CONTENTS_DESCRIPTION\)", [string]::join("`r`n", $WorkingDescList ) `
		-replace "_\$\$@\(CONTENTS_HISTORY\)", [string]::join("`r`n", $ConHistHtml ) `
		-replace "_\$\$@\(CONTENTS_NAVI_NEW_VER\)", [string]::join("`r`n", $WorkingContentNew ) `
		-replace "_\$\$@\(CONTENTS_NAVI_OLD_VER\)", [string]::join("`r`n", $WorkingContentOld )
		
	Write-Host(" WorkingContList : {0}" -f $WorkingContList)

	#---- HTML ファイルを出力
	$OutFile = $OutputDirName + "\" + $eachVersionHtmlName
	if ( "" -ne $OutputDir )
	{
		$OutFile = $OutputDir + "\" + $OutFile
	}
	New-Item $OutFile -itemType File -Force | Out-Null
	$template | Out-File -Encoding UTF8 $OutFile

	Write-Log "************** $OutFile output. **************"
}


#================================================================================

#---- 現在の日付を取得
$ReleaseYear  = (Get-Date).ToString("yyyy")
$ReleaseMonth = (Get-Date).ToString("MM")
$ReleaseDay   = (Get-Date).ToString("dd")

#---- コンフィグデータ
$Config_PackageName            = $Config.Configuration.PackageName
$Config_ReleaseDate            = Get-ReleaseDate

#---- 日付情報を置換
$Config_ReleaseDate = $Config_ReleaseDate `
         -replace "{{year}}", $ReleaseYear `
         -replace "{{month}}", $ReleaseMonth `
         -replace "{{day}}", $ReleaseDay

. .\readTopInformation.ps1
if ( $Config.Configuration.Template.TopInformation -ne $null -and $Config.Configuration.Template.TopInformation -ne "" )
{
    $Config_TopInformation = Read-TopInformation $Config.Configuration.Template.TopInformation
}
else
{
    $Config_TopInformation = ""
}

#---- トップHTMLのファイル名
$OutputName_TopHtml  = "ChangeLog.html"

#---- 内容の HTML部
#  (Topファイル)
[string[]]$ConListHtml = @()
[string[]]$ConDescHtml = @()
[string[]]$ConHistHtml = @()

[string[]]$WorkingContList = @()
[string[]]$WorkingDescList = @()
[string[]]$WorkingConEachVer = @()
[string[]]$WorkingMixedConBodyList = @()
[string[]]$WorkingConBodyList = @()
[string[]]$WorkingVersionList = @()

#---- トップページのバージョン個数
$TopVersionNum = Get-NumberOfTopVersion_Confluence
if ( $TopVersionNum -eq 0 )
{
    Write-Error "No top version specified."
    exit 1
}

#---- 各バージョンの変更履歴数(key = each version)
$EachVersionChangelogNum = @{}

#----------------------------------------------------------------
# トップページ作成
#----------------------------------------------------------------
$AllCsv = @()

$majorVersionList = "";
[array]$majorVersionList = Get-MajorVersionList
Write-Host("Major Version List Length : {0}" -f $majorVersionList.Length)

$WorkingContList += "<ul>"

# メジャーバージョンごとにループして1枚のトップページHTMLを作る
for( $i=0; $i -lt $majorVersionList.Length; $i++)
{
    $PackageVersionList, $PackageVersionString, $PackageSupplementation = Get-EachMajorVersionsList $majorVersionList[$i] 0
	
	$majorVersionStr = $majorVersionList[$i]  -replace "\.", "_"
	$OutputVersionDirName = $OutputDirName + "\Changelog-" + $majorVersionStr
	$majorVersionHtmlName = $OutputVersionDirName + "\Changelog-{0}-Top.html" -f $majorVersionStr


	Write-Host("PackageVersionList : {0}, PackageVersionString : {1}, PackageSupplementation : {2}" -f $PackageVersionList[0], $PackageVersionString, $PackageSupplementation)

    if ( $English -eq $true )
    {
		$WorkingContList += "<li><a href='#subject_changelog_{0}'>{1} ({2})</a> - <a href='.\ReleaseNotes{3}'>{4} Change Log List</a></li>" -f $i, $PackageVersionString, $PackageSupplementation, $majorVersionHtmlName, $majorVersionList[$i]
        $WorkingConEachVer += "<h1 id=`"subject_changelog_{0}`" class=`"indexed_head`">Changes in $Config_PackageName $PackageVersionString</h1>" -f $i
    }
    else
    {
		$WorkingContList += "<li><a href='#subject_changelog_{0}'>{1} ({2})</a> - <a href='.\ReleaseNotes{3}'>{4} の履歴一覧</a></li>" -f $i, $PackageVersionString, $PackageSupplementation, $majorVersionHtmlName, $majorVersionList[$i]
        $WorkingConEachVer += "<h1 id=`"subject_changelog_{0}`" class=`"indexed_head`">$Config_PackageName $PackageVersionString の変更点</h1>" -f $i
    }
	
	$majorVersionStr = $majorVersionList[$i]  -replace "\.", "_"
	$majorVersionDirName = ".\ReleaseNotes\Changelog-" + $majorVersionStr
	
	$eachVersionStr = $PackageVersionList[0]  -replace "\.", "_"
	$eachVersionHtmlName = $majorVersionDirName + "\Changelog-{0}.html" -f $eachVersionStr

    #---- 各パッケージを調べる
    $PackageListNum = Get-PackageListNum
    for( $IndexPList=0; $IndexPList -lt $PackageListNum; $IndexPList++ )
    {
        $CommonPackageName, $CommonSubject, $CommonIdPrefix, $CommonName = Get-PackageListCommonInfo( $IndexPList )
		Write-Host("  各パッケージを調べる {0}, {1}, {2}, {3}" -f $CommonPackageName, $CommonSubject, $CommonIdPrefix, $CommonName)

        #---- 対象パッケージに関するファイル名
        $OutputName_MainHtml = Get-MainHtmlName $CommonPackageName
        $OutputName_OldHtml  = Get-OldHtmlName $CommonPackageName

        #---- そのパッケージのバージョンを走査する
		$PackageListPackageNum = Get-EachMajorVersionsListNum ( $majorVersionList[$i] )
		Write-Host("    パッケージのバージョンを走査：{0}" -f $PackageListPackageNum)

		# メジャーバージョンごとに最新のバージョンしか見ないため、index には 0 を与える
        $not_use, $not_use, $IdPrefix, $Subject, $not_use = Get-PackageListPackageInfo $IndexPList 0

        if ( $IdPrefix -eq $null )
        {
            $IdPrefix = (Filter-AliasStr $idAliasRule $PackageVersionList[0]) + $CommonIdPrefix
        }
        else
        {
            $IdPrefix = (Filter-AliasStr $idAliasRule $PackageVersionList[0]) + $IdPrefix
        }

        $AllCsv = @()
        foreach ($cname in $CommonName)
        {
            if ( $cname.Length -gt 0 )
            {
                $cname0 = "SIGLO"
                $cname1 = $cname
            }
            else
            {
                $cname0 = $cname.Project
                $cname1 = ($cname."#text")
            }

            foreach ($vname in $PackageVersionList)
            {
                $cname1 = Filter-AliasStr $packageAliasRule $cname1
                $vname = Filter-AliasStr $versionAliasRule $vname
                $VersionForCsvNum = $i.ToString("000")
                $VersionForCsv = $vname

                $ProjectStr = Filter-AliasStr $projectFnameAliasRule $cname0
                $PackageStr = Filter-AliasStr $packageFnameAliasRule $cname1
                $VersionStr = Filter-AliasStr $versionFnameAliasRule $vname

                # (主言語データの読み込み)
                $r = Read-CsvData $true $true $ProjectStr $PackageStr $VersionStr
                if ( ! $r )
                {
                    $r = Read-CsvData $false $true $ProjectStr $PackageStr $VersionStr
                }
                # (副言語データの読み込み)
                $r = Read-CsvData $true $false $ProjectStr $PackageStr $VersionStr
                if ( ! $r )
                {
                    $r = Read-CsvData $false $false $ProjectStr $PackageStr $VersionStr
                }
            }
        }

        if ( $AllCsv.Length -ne 0 )
        {
            # Write-Host "[" $CommonSubject "]" "  IdStr=" $CommonIdPrefix "   Packages=" $CommonName
            # Write-Host ("{0} log gathered." -f $AllCsv.Length)
            if ( $Subject -eq $null )
            {
                $Subject = $CommonSubject
            }
            if ( $PackageVersionString -eq $null )
            {
                $PackageVersionString = $PackageVersionList[0]
            }

            if ( $English -eq $true )
            {
                #$WorkingContList += "      <li><a href=`"#module_idx_{0}_0_0_{1}`">Changes in {2}</a></li>" -f ($i + 1), ($IndexPList + 1), $Subject
                $WorkingConEachVer += "<h2 id=`"module_idx_{0}_0_0_{1}`"> {2} </h2>" -f ($i + 1), ($IndexPList + 1), $Subject
                $WorkingMixedConBodyList += "{1}::<h2 id=`"ret_{0}`" class=`"list`"><a href=`"#list_contents_{0}`"> {2} {3} {4}</a></h2>" -f ($i + 1), $IndexPList, $Subject, $PackageVersionString, $Supplementation
                $WorkingConBodyList += "{1}::<h1 id=`"list_contents_{0}`">Changes in  {2} {3}</h1>" -f ($i + 1), $IndexPList, $Subject, $PackageVersionString, $Supplementation
                $WorkingVersionList += "{1}::   <li><a href=`"#ret_{0}`">Changes in {2} {3} {4}</a></li>" -f ($i+1), $IndexPList, $Subject, $PackageVersionString, $Supplementation
            }
            else
            {
                #$WorkingContList += "      <li><a href=`"#module_idx_{0}_0_0_{1}`">{2} の変更点</a></li>" -f ($i + 1), ($IndexPList + 1), $Subject
                $WorkingConEachVer += "<h2 id=`"module_idx_{0}_0_0_{1}`"> {2} </h2>" -f ($i + 1), ($IndexPList + 1), $Subject
                $WorkingMixedConBodyList += "{1}::<h2 id=`"ret_{0}`" class=`"list`"><a href=`"#list_contents_{0}`"> {2} {3} {4}</a></h2>" -f ($i + 1), $IndexPList, $Subject, $PackageVersionString, $Supplementation
                $WorkingConBodyList += "{1}::<h1 id=`"list_contents_{0}`"> {2} {3} の変更点</h1>" -f ($i + 1), $IndexPList, $Subject, $PackageVersionString, $Supplementation
                $WorkingVersionList += "{1}::   <li><a href=`"#ret_{0}`"> {2} {3} の変更点 {4}</a></li>" -f ($i+1), $IndexPList, $Subject, $PackageVersionString, $Supplementation
            }

            #---- リンク追加
            $LogListIndex = 1
            Create-ContentsList2 $Subject $CommonPackageName ([ref]$WorkingConEachVer) ([ref]$WorkingMixedConBodyList) ([ref]$WorkingConBodyList) $IdPrefix $IndexPList $eachVersionHtmlName
        }
    }
}
#---- 過去の履歴一覧
if ( $English -eq $true )
{
	$WorkingContList += "<li><a href='.\ReleaseNotes\Changelog-Old\Changelog-Old.html'>Old Change Log List</a></li>"
}
else
{
	$WorkingContList += "<li><a href='.\ReleaseNotes\Changelog-Old\Changelog-Old.html'>過去の履歴一覧</a></li>"
}

$WorkingContList += "</ul>"

#---- トップ HTML のテンプレート読み込み
if ( $false -eq $Document )
{
    $TemplateName_TopHtml = $Config.Configuration.Template.TopHtml
}
else
{
    $TemplateName_TopHtml = $Config.Configuration.Template.DocTopHtml
}
$template = Get-Content -Encoding UTF8 -Path $TemplateName_TopHtml

#---- テンプレート変数の置き換え
$template = $template `
    -replace "_\$\$@\(PACKAGE_NAME\)", $Config_PackageName `
    -replace "_\$\$@\(RELEASE_DATE\)", $Config_ReleaseDate `
	-replace "_\$\$@\(RELEASE_YEAR\)", $ReleaseYear `
    -replace "_\$\$@\(TOP_INFORMATION\)", $Config_TopInformation `
    -replace "_\$\$@\(PACKAGE_SUPPLEMENTATION\)", $Config_PackageSupplementation `
    -replace "_\$\$@\(CONTENTS_LIST\)", [string]::join("`r`n", $WorkingContList) `
	-replace "_\$\$@\(CONTENTS_EACH_VERSIONS\)", [string]::join("`r`n", $WorkingConEachVer ) `
    -replace "_\$\$@\(CONTENTS_DESCRIPTION\)", [string]::join("`r`n", $WorkingDescList ) `
    -replace "_\$\$@\(CONTENTS_HISTORY\)", [string]::join("`r`n", $ConHistHtml )

#---- 出力
$OutFile = $OutputName_TopHtml
if ( "" -ne $OutputDir )
{
    $OutFile = $OutputDir + "\" + $OutFile
}
New-Item $OutFile -itemType File -Force | Out-Null
$template | Out-File -Encoding UTF8 $OutFile

Write-Log "************** $OutFile output. **************"


#----------------------------------------------------------------
# 各バージョンの HTML 作成
#----------------------------------------------------------------
Write-Host("

各バージョンの HTML 作成")

# メジャーバージョンごとにループ
for( $i=0; $i -lt $majorVersionList.Length; $i++)
{
	$majorVersionStr = $majorVersionList[$i]  -replace "\.", "_"
	$OutputVersionDirName = "ReleaseNotes\Changelog-" + $majorVersionStr
	
	$versionListNum = Get-EachMajorVersionsListNum($majorVersionList[$i])
	$newVersion = ""
	$oldVersion = ""

	# メジャーバージョンの中の各バージョンでループしてそれぞれの HTML を出力する
	for( $j=0; $j -lt $versionListNum; $j++)
	{
		$script:LogNum = 0
		# 次のバージョンの値をとるためだけの処理
		if ( $j + 1 -lt $versionListNum )
		{
			$PackageVersionList, $PackageVersionString, $PackageSupplementation = Get-EachMajorVersionsList $majorVersionList[$i] ($j + 1)
			$oldVersion = $PackageVersionList[0]
		}
		else
		{
			$oldVersion = ""
		}

		$PackageVersionList, $PackageVersionString, $PackageSupplementation = Get-EachMajorVersionsList $majorVersionList[$i] $j
		Generate-EachVersionChangelog $PackageVersionList $PackageVersionString $PackageSupplementation $OutputVersionDirName $newVersion $oldVersion

		# 今のバージョンを「次のバージョン」として次回使う
		$newVersion = $PackageVersionList[0]

		# 各バージョンの変更履歴数を保持
		$EachVersionChangelogNum.Add($PackageVersionList[0], $script:LogNum)
		Write-Host ( " Log Num : {0}" -f $script:LogNum)
	}
}

#----------------------------------------------------------------
# 各メジャーバージョンの HTML 作成
#----------------------------------------------------------------
Write-Host("
	
各メジャーバージョンの HTML 作成")

$OutputDirName = "ReleaseNotes"
# メジャーバージョンごとにループしてそれぞれの HTML を作る
for( $i=0; $i -lt $majorVersionList.Length; $i++)
{
	[string[]]$WorkingContList = @()
	$majorVersionStr = $majorVersionList[$i]  -replace "\.", "_"
	$OutputVersionDirName = $OutputDirName + "\Changelog-" + $majorVersionStr
	
	$WorkingContList += "<ul>"
	$versionListNum = Get-EachMajorVersionsListNum($majorVersionList[$i])
	$majorVersionHtmlName = $OutputVersionDirName + "\Changelog-{0}-Top.html" -f $majorVersionStr

	for( $j=0; $j -lt $versionListNum; $j++)
	{
		$PackageVersionList, $PackageVersionString, $PackageSupplementation = Get-EachMajorVersionsList $majorVersionList[$i] $j
		$eachVersionStr = $PackageVersionList[0]  -replace "\.", "_"
		$eachVersionHtmlName = "\Changelog-{0}.html" -f $eachVersionStr
		# Write-Host("{0}, {1}, {2}, {3}" -f $PackageVersionList, $PackageVersionString, $PackageSupplementation, $eachVersionHtmlName)
		if ( $English -eq $true )
		{
			$WorkingContList += "<li><a href='.{0}'>Changes in {1} ({2})</a> - ({3})</li>" -f $eachVersionHtmlName, $PackageVersionString, $PackageSupplementation, $EachVersionChangelogNum[$PackageVersionList[0]]
		}
		else
		{
			$WorkingContList += "<li><a href='.{0}'>{1} の変更点 ({2})</a> - ({3})</li>" -f $eachVersionHtmlName, $PackageVersionString, $PackageSupplementation, $EachVersionChangelogNum[$PackageVersionList[0]]
		}
	}
	$WorkingContList += "</ul>"

	#---- メジャーバージョン HTML のテンプレート読み込み
	$TemplateName_MajorHtml = $Config.Configuration.Template.MajorHtml
	$template = Get-Content -Encoding UTF8 -Path $TemplateName_MajorHtml

	#---- テンプレート変数の置き換え
	$template = $template `
		-replace "_\$\$@\(PACKAGE_NAME\)", $majorVersionList[$i] `
		-replace "_\$\$@\(RELEASE_DATE\)", $Config_ReleaseDate `
		-replace "_\$\$@\(RELEASE_YEAR\)", $ReleaseYear `
		-replace "_\$\$@\(TOP_INFORMATION\)", $Config_TopInformation `
		-replace "_\$\$@\(PACKAGE_SUPPLEMENTATION\)", $Config_PackageSupplementation `
		-replace "_\$\$@\(CONTENTS_LIST\)", [string]::join("`r`n", $WorkingContList) `
		-replace "_\$\$@\(CONTENTS_EACH_VERSIONS\)", [string]::join("`r`n", $WorkingConEachVer ) `
		-replace "_\$\$@\(CONTENTS_DESCRIPTION\)", [string]::join("`r`n", $WorkingDescList ) `
		-replace "_\$\$@\(CONTENTS_HISTORY\)", [string]::join("`r`n", $ConHistHtml )

	#---- HTML ファイルを出力
	$OutFile = $majorVersionHtmlName
	if ( "" -ne $OutputDir )
	{
		$OutFile = $OutputDir + "\" + $OutFile
	}
	New-Item $OutFile -itemType File -Force | Out-Null
	$template | Out-File -Encoding UTF8 $OutFile

	Write-Log "************** $OutFile output. **************"
}


#----------------------------------------------------------------
# 過去の各バージョンの HTML 作成
#----------------------------------------------------------------
Write-Host("

過去の各バージョンの HTML 作成")
$OutputDirName = "ReleaseNotes\Changelog-Old"
$versionListNum = Get-OldChangelogListNum
$newVersion = ""
$oldVersion = ""

# 過去の各バージョンそれぞれの HTML を出力する
for( $i=0; $i -lt $versionListNum; $i++)
{
	$script:LogNum = 0
	# 次のバージョンの値をとるためだけの処理
	if ( $i + 1 -lt $versionListNum )
	{
		$PackageVersionList, $PackageVersionString, $PackageSupplementation = Get-OldChangelogList ($i + 1)
		$oldVersion = $PackageVersionList[0]
	}
	else
	{
		$oldVersion = ""
	}

	$PackageVersionList, $PackageVersionString, $PackageSupplementation = Get-OldChangelogList $i
	Generate-EachVersionChangelog $PackageVersionList $PackageVersionString $PackageSupplementation $OutputDirName $newVersion $oldVersion

	# 今のバージョンを「次のバージョン」として次回使う
	$newVersion = $PackageVersionList[0]

	# 各バージョンの変更履歴数を保持
	$EachVersionChangelogNum.Add($PackageVersionList[0], $script:LogNum)
	Write-Host ( " Log Num : {0}" -f $script:LogNum)
}


#----------------------------------------------------------------
# 古い変更履歴のトップ HTML 作成
#----------------------------------------------------------------
Write-Host("
	
過去の変更履歴の HTML 作成")

$OutputDirName = "ReleaseNotes\Changelog-Old"
$OldVersionHtmlName = $OutputDirName + "\Changelog-Old.html"

[string[]]$WorkingContList = @()
	
$oldVersionListNum = Get-OldChangelogListNum
$WorkingContList += "<ul>"

for( $i=0; $i -lt $oldVersionListNum; $i++)
{
	$PackageVersionList, $PackageVersionString, $PackageSupplementation = Get-OldChangelogList $i
	# $PackageVersionList, $PackageVersionString, $PackageSupplementation = Get-EachMajorVersionsList $majorVersionList[$i] $j
	$eachVersionStr = $PackageVersionList[0]  -replace "\.", "_"
	$eachVersionHtmlName = "\Changelog-{0}.html" -f $eachVersionStr
	# Write-Host("{0}, {1}, {2}, {3}" -f $PackageVersionList, $PackageVersionString, $PackageSupplementation, $eachVersionHtmlName)
	if ( $English -eq $true )
	{
		$WorkingContList += "<li><a href='.{0}'>Changes in {1} ({2})</a> - ({3})</li>" -f $eachVersionHtmlName, $PackageVersionString, $PackageSupplementation, $EachVersionChangelogNum[$PackageVersionList[0]]
	}
	else
	{
		$WorkingContList += "<li><a href='.{0}'>{1} の変更点 ({2})</a> - ({3})</li>" -f $eachVersionHtmlName, $PackageVersionString, $PackageSupplementation, $EachVersionChangelogNum[$PackageVersionList[0]]
	}
}
$WorkingContList += "</ul>"

#---- メジャーバージョン HTML のテンプレート読み込み
$TemplateName_OldHtml = $Config.Configuration.Template.OldHtml
$template = Get-Content -Encoding UTF8 -Path $TemplateName_OldHtml

#---- テンプレート変数の置き換え
$template = $template `
	-replace "_\$\$@\(PACKAGE_NAME\)", $majorVersionList[$i] `
	-replace "_\$\$@\(RELEASE_DATE\)", $Config_ReleaseDate `
	-replace "_\$\$@\(RELEASE_YEAR\)", $ReleaseYear `
	-replace "_\$\$@\(TOP_INFORMATION\)", $Config_TopInformation `
	-replace "_\$\$@\(PACKAGE_SUPPLEMENTATION\)", $Config_PackageSupplementation `
	-replace "_\$\$@\(CONTENTS_LIST\)", [string]::join("`r`n", $WorkingContList) `
	-replace "_\$\$@\(CONTENTS_EACH_VERSIONS\)", [string]::join("`r`n", $WorkingConEachVer ) `
	-replace "_\$\$@\(CONTENTS_DESCRIPTION\)", [string]::join("`r`n", $WorkingDescList ) `
	-replace "_\$\$@\(CONTENTS_HISTORY\)", [string]::join("`r`n", $ConHistHtml )

#---- HTML ファイルを出力
$OutFile = $OldVersionHtmlName
if ( "" -ne $OutputDir )
{
	$OutFile = $OutputDir + "\" + $OutFile
}
New-Item $OutFile -itemType File -Force | Out-Null
$template | Out-File -Encoding UTF8 $OutFile

Write-Log "************** $OutFile output. **************"

#---- 終了表示
Write-Log "Done."

