﻿namespace Nintendo.PlatformCodeFilter

open System

/// #if,#elif の式の要素の種別です。
type IfExprType =

    /// 式が無い(未定義なもの)
    | Null  = 0

    /// AND
    | And   = 1

    /// OR
    | Or    = 2

    /// NOT
    | Not   = 3

    /// 未対応2項演算子
    | Usop2 = 4

    /// defined(マクロ)
    | DefinedMacro  = 5

    /// 数値
    | Num   = 6


/// #if,#elif の式の要素を保持する基本クラスです。
type IfExpr (defExprType: IfExprType) =

    /// デフォルトコンストラクタです。
    new () = IfExpr (IfExprType.Null)

    /// IfExprType です。
    member this.Type = defExprType


/// defined (マクロ) を保持するクラスです。
type IfExprDefinedMacro (macro: string) =
    inherit IfExpr (IfExprType.DefinedMacro)

    do
        if macro = null then
            raise <| new ArgumentNullException "Macro"

    /// マクロを取得します。
    member this.Macro = macro


/// 数値を保持するクラスです。
type IfExprNum (num: int64) =
    inherit IfExpr (IfExprType.Num)

    /// 数値を取得します。
    member this.Num = num


/// 単項演算子を保持するクラスです。
type IfExprOp1 (defExprType, expr: IfExpr) =
    inherit IfExpr (defExprType)

    /// IfExpr を取得します。
    member this.Expr = expr


/// 2項演算子を保持するクラスです。
type IfExprOp2 (defExprType, left: IfExpr, right: IfExpr) =
    inherit IfExpr (defExprType)

    /// 左辺の IfExpr を取得します。
    member this.Left = left

    /// 右辺の IfExpr を取得します。
    member this.Right = right


/// 文字列の範囲を保持する構造体です。
type StringRange (offset: int, length: int) =
    struct
        static member Empty = StringRange (0, 0)

        /// オフセットを取得します。
        member this.Offset = offset

        /// 長さを取得します。
        member this.Length = length
    end

/// コードブロックを表す抽象クラスです。
[<AbstractClass>]
type CCodeBlock (isIfBlock: bool) =

    /// #if/#if(n)def ～ #endif ブロックかどうかを判別する値を取得します。
    member this.IsIfBlock = isIfBlock


/// #if/if(n)def, #elif, #else の行と式、後続するコードブロックを保持するクラスです。
type IfInfo (ifLine: StringRange, expr: IfExpr, cCodeBlocks: CCodeBlock seq) =

    /// #if/if(n)def, #elif, #else の行の内容を取得します。
    member this.IfLine = ifLine

    /// #if/if(n)def, #elif, #else の式の内容を取得します。
    member this.Expr = expr

    /// #if/if(n)def, #elif, #else に後続するコードブロックを取得します。
    member this.CCodeBlocks = cCodeBlocks


/// #if/#if(n)def ～ #endif を保持するクラスです。
type IfEndifBlock (ifInfos: IfInfo seq, endifLine: StringRange) =
    inherit CCodeBlock true

    /// IfInfo オブジェクトの Enumerable を取得します。
    /// #if/if(n)def, #elif, #else の分存在します。
    /// 上述の #if 等に続くコードブロックを再帰的に持ちます。
    member this.IfInfos = ifInfos

    /// #endif 行の内容を取得します。
    member this.EndifLine = endifLine


/// #if/#if(n)def ～ #endif でない1行を保持するクラスです。
type CLineBlock (line: StringRange) =
    inherit CCodeBlock false

    /// 1行の内容を取得します。
    member this.Line = line


/// parseDirective の結果を保持するクラスです。
type ParseDirectiveResult (success: bool, cCodeBlocks: CCodeBlock seq, errorMessage: string, index: int) =

    new (cLineBases) =
        ParseDirectiveResult (true, cLineBases, null, 0)

    new (errorMessage, index) =
        ParseDirectiveResult (false, Seq.empty, errorMessage, index) 

    /// 解析が成功したか失敗したかを判別する値を取得します。
    member this.Success= success

    /// 解析が成功している場合、解析したコードツリーを取得します。
    member this.CCodeBlocks = cCodeBlocks

    /// 解析が失敗している場合、エラーのメッセージを取得します。
    member this.ErrorMessage = errorMessage

    member this.Index = index
