﻿namespace Nintendo.PlatformCodeFilter

open System
open System.Collections.Generic
open System.IO
open System.Text
open FParsec

/// CharStreamに保持させる UserState のレコードです。
type CFilterUserState = {

    /// 行頭を保持します。
    IndexToken: CharStreamIndexToken;
    }


/// Cコードをパースするモジュールです。
module CFilter =

    /// <summary>
    /// 空白文字かどうかを判定します。
    /// </summary>
    /// <param name="c">判定する文字。</param>
    /// <returns>スペースかタブ文字の場合は true、そうでない場合は false。</returns>
    let internal isSpace c = c = ' ' || c = '\t' 

    /// 連続する0個以上のスペースまたはタブ文字をスキップします。
    let internal skipSpaces<'u> : Parser<_, 'u> = skipManySatisfy isSpace

    /// コメントをスキップします。
    let internal skipComment<'u> : Parser<_, 'u> =
        (skipString "/*" >>. skipCharsTillString "*/" true Int32.MaxValue)  // 複数行コメント
        <|> (skipString "//" >>. skipRestOfLine false)                      // 1行コメント

    /// スペースとコメントをスキップします。
    let internal skipSCs<'u> : Parser<_, 'u>  =
        skipSepBy skipSpaces skipComment

    /// '0'や'\'の後の、8進文字列をパースします。
    /// 1文字も消費しない場合を許容し、その場合は 0 を返します。
    let internal parseOctal<'u> : Parser<_, 'u> =
        manySatisfy isOctal
        |>> fun str ->
                if str.Length = 0 then
                    0L
                else
                    Convert.ToInt64 (str, 8)
        <?> "octal"

    /// '0'や'\'の後の、8進文字列をパースします。
    let internal parseOctal1<'u> : Parser<_, 'u> =
        many1SatisfyL isOctal "octal"
        |>> fun str -> Convert.ToInt64 (str, 8)

    /// '0'や'\'の後の、8進文字列をスキップします。
    /// 1文字も消費しない場合を許容します。
    let internal skipOctal<'u> : Parser<_, 'u> =
        skipManySatisfy isOctal <?> "octal"

    /// '0'や'\'の後の、8進文字列をスキップします。
    let internal skipOctal1<'u> : Parser<_, 'u> =
        skipMany1SatisfyL isOctal "octal"

    /// '0'や'\'の後の、'x' とこれに続く16数字列をパースします。
    let internal parseHex1<'u> : Parser<_, 'u> =
        skipStringCI "x" >>. many1SatisfyL isHex "hex" |>> fun s -> Convert.ToInt64 (s, 16)

    /// '0'や'\'の後の、'x' とこれに続く16数字列をスキップします。
    let internal skipHex1<'u> : Parser<_, 'u> =
        skipStringCI "x" >>. skipMany1SatisfyL isHex "hex"

    /// '1' から '9' の文字かどうかを判定します。
    let internal isDigitNonZero c =
        uint32 c - uint32 '1' <= uint32 '9' - uint32 '1'

    /// 数値の後の 'u'や'l'があればスキップします。
    let internal skipUL<'u> : Parser<_, 'u> =
        optional (skipStringCI "U") .>> optional (skipStringCI "L")

    /// 数字をパースします。
    let internal parseNumber<'u> : Parser<_, 'u> =
        (
            (skipChar '0' >>. (parseHex1 <|> parseOctal))
            <|> (many1Satisfy2L isDigitNonZero isDigit "digit" |>> Int64.Parse)
        )
        .>> skipUL
        <?> "number"

    /// 数字をスキップします。
    let internal skipNumber<'u> : Parser<_, 'u> =
        (
            (skipChar '0' >>. (skipHex1 <|> skipOctal))
            <|> skipMany1Satisfy2L isDigitNonZero isDigit "digit"
        )
        .>> skipUL
        <?> "number"

    /// 識別子の最初の文字として使用可能な文字かどうかを判定します。
    /// 具体的には、アスキー文字かアンダーバーかどうかを判定します。
    let internal isIdentifierFirstChar c = isAsciiLetter c || c = '_'

    /// 識別子の文字として使用可能な文字かどうかを判定します。
    /// アスキー文字かアンダーバーか数字どうかを判定します。
    let internal isIdentifierChar c = isIdentifierFirstChar c || isDigit c 

    /// 識別子をパースします。
    let internal parseId<'u> : Parser<_, 'u> =
        let isIdFirstNonL c = c <> 'L' && isIdentifierFirstChar c

        pipe2
            (
                (
                    // L で始まるときは、'や" が直後に来た場合に
                    // バックトラックする。
                    // 常にバックトラックの準備をするのを避けるために
                    // Lで始まるかどうかを確認してからにしているが、
                    // それほど重要ではない。
                    followedByString "L"
                    >>. (pchar 'L' .>>? nextCharSatisfiesNot (fun c -> c = '\'' || c = '"'))
                )
                <|> satisfy isIdFirstNonL
            )
            (manySatisfy isIdentifierChar)
            (fun c str -> string c + str)
        <?> "identifier"

    let internal parseIdStringReturn str result : Parser<_, 'u> =
        let expectedIdentifier = expected str
        fun stream ->
            let state = stream.State
            let reply = parseId stream
            if reply.Status = Ok then
                if reply.Result = str then  // 期待する文字列か?
                    Reply result
                else
                    // 期待した文字列ではない場合、元の位置に戻してエラーにする。
                    stream.BacktrackTo state
                    Reply (Error, expectedIdentifier)
            else
                Reply (reply.Status, reply.Error)

    /// 特定の識別子をパースします。
    let internal parseIdString str = parseIdStringReturn str str

    /// 特定の識別子をスキップします。
    let internal skipIdString str = parseIdStringReturn str ()

    /// 文字列で使われるエスケープされる文字を判定します。
    /// (注: isAnyOf は実行時にコードを生成します。)
    let internal isEscapeCharInString = isAnyOf "abnrftv\¥?'\""

    /// 特定の文字で囲まれた文字列をパースするパーサの実装部です。
    let internal parseQuoteStringImpl quote (f: int -> char seq -> 'Result) label =
        let skipQuoteChar = skipChar quote

        let parseNormalCharSnippet = manySatisfy (fun c -> c <> quote && c <> '\\')

        let parseEscapedChar =
            skipChar '\\'
            >>. choice [
                (satisfy isEscapeCharInString
                |>> function
                    | 'a' -> '\u0007'
                    | 'b' -> '\u0008'
                    | 't' -> '\u0009'
                    | 'n' -> '\u000A'
                    | 'v' -> '\u000B'
                    | 'f' -> '\u000C'
                    | 'r' -> '\u000D'
                    | c -> c
                    )
                parseHex1 |>> (uint16 >> char)
                parseOctal1 |>> (uint16 >> char)    // '\0' もここで処理
            ]
        let pnum sf =
            Inline.SepBy(
                (fun (x: string) -> StringBuilder x),
                (fun xs (ec: char) x -> (xs.Append ec).Append x),
                (fun sb -> sb.ToString ()),
                parseNormalCharSnippet,
                parseEscapedChar,
                resultForEmptySequence = fun () -> "")
            |>> f sf

        (between skipQuoteChar  skipQuoteChar (pnum 8))
        <|> (between (skipChar 'L' >>? skipQuoteChar) skipQuoteChar (pnum 16))
        <?> label

    /// 特定の文字で囲まれた文字列をスキップするパーサの実装部です。
    let internal skipQuoteStringImpl quote label =
        let skipQuoteChar = skipChar quote
        let skipOpen = optional (skipChar 'L') >>? skipQuoteChar

        let skipNormalCharSnippet = skipManySatisfy (fun c -> c <> quote && c <> '\\')

        let skipEscapedChar =
            skipChar '\\'
            .>> choice [
                skipSatisfy isEscapeCharInString
                skipHex1
                skipOctal1      // '\0' もここで処理
            ]

        between
            skipOpen
            skipQuoteChar
            (skipSepBy skipNormalCharSnippet skipEscapedChar)
        <?> label

    /// シングルクォートで囲まれた文字列をパースします。
    let internal parseSingleQuote<'u> : Parser<_, 'u> =
        parseQuoteStringImpl
            '\''
            (fun sf chars -> chars |> Seq.fold (fun num c -> (num <<< sf) + (int64 c)) 0L)
            "character-constant"

    /// シングルクォートで囲まれた文字列をスキップします。
    let internal skipSingleQuote<'u> : Parser<_, 'u> =
        skipQuoteStringImpl '\'' "character-constant"

    /// ダブルクォートで囲まれた文字列をスキップします。
    let internal skipDoubleQuote<'u> : Parser<_, 'u> =
        skipQuoteStringImpl '"'  "string-literal"

    /// 文字・文字列のクォーテーションかコメントで使う文字かを判定します。
    /// (注: isAnyOf は実行時にコードを生成します。)
    let internal isQuoteCommentChar = isAnyOf "'\"\\/\n"

    /// シンボル文字等を判定する関数。
    /// 次の文字列を処理するためにそれらを除く文字を判定する。
    /// ・数値
    /// ・識別子
    /// ・エスケープされた改行
    /// ・文字
    /// ・文字列定数
    /// ・コメント
    let internal isSymbolChar c = not (isDigit c || isAsciiLetter c || c = '_' || isQuoteCommentChar c)

    let internal skipText<'u> : Parser<_, 'u> =
        skipSepBy
            (skipManySatisfy isSymbolChar)
            (choice [
                skipNumber
                parseId |>> ignore
                skipSingleQuote
                skipDoubleQuote
                skipComment
                (skipChar '/')
                ])

    let internal setIndexToken : Parser<unit, CFilterUserState> =
        fun stream ->
            stream.UserState <- { IndexToken = stream.IndexToken }
            Reply (())

    let internal getSkippedLine : Parser<_, CFilterUserState> =
        fun stream ->
            let userState = stream.UserState
            let startIdx = userState.IndexToken.GetIndex stream
            let len = stream.Index - startIdx
            Reply (StringRange (int startIdx, int len))

    let internal createCLineBlock line = CLineBlock line :> CCodeBlock

    let internal parseCLineBlock =
        skipText >>. getSkippedLine |>> createCLineBlock

    // 改行コードまでスキップします。
    let internal parseRestOfLine =
        skipRestOfLine false >>. getSkippedLine |>> createCLineBlock

    let internal parsePpDirInclude =
        let pIncludePath =
            let skipPath cOpen cClose =
                between (skipChar cOpen)
                        (skipChar cClose)
                        (skipManySatisfy ((<>) cClose))

            choice [
                skipPath '<' '>' .>> skipSCs
                skipPath '"' '"' .>> skipSCs
                parseId >>. skipText
            ]

        skipSCs >>. pIncludePath >>. getSkippedLine |>> createCLineBlock

    let internal notOpFunc (expr: IfExpr) : IfExpr =
        IfExprOp1 (IfExprType.Not, expr) :> IfExpr

    let internal orOpFunc expr1 expr2 =
        IfExprOp2 (IfExprType.Or, expr1, expr2) :> IfExpr

    let internal andOpFunc expr1 expr2 =
        IfExprOp2 (IfExprType.And, expr1, expr2) :> IfExpr

    // プラットフォーム定義マクロでないもの同士をまとめるだけの関数。
    // プラットフォーム定義マクロでないもの以外が来た場合は例外をスローする。
    let internal noneOpFunc (expr1: IfExpr) (expr2: IfExpr) =
        IfExprOp2 (IfExprType.Usop2, expr1, expr2) :> IfExpr

    /// "#ifdef" の後のマクロをパースします。
    let internal parseIfdefMacro =
        parseId .>> skipSCs |>> (fun id -> IfExprDefinedMacro id :> IfExpr)

    /// "#ifndef" の後のマクロをパースします。
    let internal parseIfndefMacro =
        parseId .>> skipSCs |>> (fun id -> IfExprDefinedMacro id :> IfExpr |> notOpFunc)

    /// マクロをスキップします。
    let internal skipMacro =
        let skipMacro, skipMacroRef = createParserForwardedToRef ()

        let skipMacroParam =
            choice [
                skipNumber .>> skipSCs;
                skipSingleQuote .>> skipSCs;
                skipMacro
            ]

        skipMacroRef :=
            parseId .>> skipSCs |>> ignore
            .>> optional
                (between
                    (skipChar '(' >>. skipSCs)
                    (skipChar ')' >>. skipSCs)
                    (skipSepBy skipMacroParam (skipChar ',' >>. skipSCs)))

        !skipMacroRef

    /// defined (マクロ)をパースします。
    let internal parseDefined =
        let paren = between (skipChar '(' >>. skipSCs ) (skipChar ')' >>. skipSCs) (parseId .>> skipSCs)
        skipIdString "defined" .>> skipSCs >>. (parseId .>> skipSCs <|> paren)

    /// "#if" の後の式をパースします。
    let internal parseIfExpression =
        let opp = OperatorPrecedenceParser ()

        // ( 式 ) のパース
        let parenExpr = between (skipChar '(' >>. skipSCs) (skipChar ')' >>. skipSCs) opp.ExpressionParser

        opp.TermParser <-
            choice [
                parseNumber .>> skipSCs |>> fun value -> IfExprNum value :> IfExpr;
                parseSingleQuote .>> skipSCs |>> fun value -> IfExprNum value :> IfExpr;
                parseDefined |>> fun id -> IfExprDefinedMacro id :> IfExpr;
                skipMacro >>% IfExpr ();
                parenExpr
            ]

        opp.AddOperator <| InfixOperator ("||", skipSCs, 1, Associativity.Left, orOpFunc)
        opp.AddOperator <| InfixOperator ("&&", skipSCs, 2, Associativity.Left, andOpFunc)
        // == , != , <, <=, >, >=, ! の演算子は、プラットフォーム定義マクロではエラーとし、
        // プラットフォーム定義マクロ以外でもまとめるだけにする。
        opp.AddOperator <| InfixOperator ("==", skipSCs, 3, Associativity.Left, noneOpFunc)
        opp.AddOperator <| InfixOperator ("!=", skipSCs, 3, Associativity.Left, noneOpFunc)
        opp.AddOperator <| InfixOperator ("<", skipSCs, 4, Associativity.Left, noneOpFunc)
        opp.AddOperator <| InfixOperator (">", skipSCs, 4, Associativity.Left, noneOpFunc)
        opp.AddOperator <| InfixOperator ("<=", skipSCs, 4, Associativity.Left, noneOpFunc)
        opp.AddOperator <| InfixOperator (">=", skipSCs, 4, Associativity.Left, noneOpFunc)
        opp.AddOperator <| PrefixOperator ("!", skipSCs, 5, true, notOpFunc)

        opp.ExpressionParser

    /// #if の式をパースし、プラットフォームのマクロのみのリストを取得します。
    let internal parsePlatformMacros =
        let paren =
            between
                (skipChar '(' .>> skipSCs)
                (skipChar ')')
                (parseId .>> skipSCs)

        /// defined () のパース
        let parseDefinedOrId =
            parseId
            >>= function
                | "defined" -> skipSCs >>. (parseId <|> paren) |>> Some
                | _ -> preturn None

        many
            (choice [
                (skipMany1Satisfy isSymbolChar >>% None)
                (skipNumber >>% None)
                parseDefinedOrId
                (skipSingleQuote >>% None)
                (skipDoubleQuote >>% None)
                (skipComment >>% None)
                (skipChar '/' >>% None)
                ])
        |>> List.choose id
        <?> "#if expression"

    /// パーサ test が true を返すときに、元の位置から パーサ p を適用します。
    /// パーサ test が false を返すときに falseResult の Reply を返します。
    let internal testParse (test: Parser<bool,_>) p falseResult : Parser<_,'u> =
        fun stream ->
            let state = stream.State
            let reply = test stream
            if reply.Status = Ok then
                if reply.Result then
                    stream.BacktrackTo state
                    p stream
                else
                    Reply falseResult
            else
                Reply (reply.Status, reply.Error)

    /// "#else" 以降をパースします。(何もパースしません)
    let internal parseElseExpr = preturn (IfExpr ())

    /// #if の 式と#elif, #else, #endifのいずれかに遭遇するまで文をパースします。
    let internal parseIfExprAndLines parseIfExpr parseLine =
        skipSCs    // 行頭のスペース、コメントをスキップ
        >>. pipe3
            parseIfExpr
            (getSkippedLine .>> skipNewline .>> setIndexToken .>> skipSCs)
            (many (parseLine .>> skipNewline .>> setIndexToken .>> skipSCs))
            (fun expr lineStr codeBlocks ->
                IfInfo (lineStr, expr, List.toArray codeBlocks)
            )

    /// 1つのコードブロックをパースします。
    ///
    /// コードブロックは次のものを指します。
    ///   ・#if/#ifdef/#ifndef がある行から対応する #endif のある行(複数行)
    ///   ・上記に当てはまらない行(1行)
    ///
    /// いずれも、末尾の改行の直前までパースし、改行のスキップはしません。
    /// #if/#ifdef/#ifndefの場合は、#endifまでの間のコードブロックを再帰的に
    /// パースします。#endifまでの間の改行はスキップされますが、
    /// #endifがある行の改行はスキップされません。
    let internal parseCCodeBlock enabledMacro =
        let parseLine, parseLineRef = createParserForwardedToRef ()

        /// #if の式の計算
        /// 本来は全てマクロを展開してから式を計算すべきだが、この処理ではそれができないので、
        /// プラットフォームの defined があるときのみ式を計算するようにしている。
        /// 逆に言うと、 プラットフォームの defined を含む場合は #if の式に
        /// 制限が発生している。
        let parseTestIfExpr =
            testParse
                (parsePlatformMacros |>> List.exists enabledMacro)
                parseIfExpression
                (IfExpr ())

        let parseIfBlockCommon parseIfPart =
            pipe4
                parseIfPart // #if の式以降の処理
                (many (pchar '#' .>> skipSCs >>? parseIdString "elif" .>> skipSCs >>. parseIfExprAndLines parseTestIfExpr parseLine))  // # elif の処理
                (opt (pchar '#' .>> skipSCs >>? parseIdString "else" .>> skipSCs >>. parseIfExprAndLines parseElseExpr parseLine)) // #else の処理
                (pchar '#' .>> skipSCs >>? parseIdString "endif" .>> skipSCs >>. getSkippedLine) // #endif の処理
                (fun ifInfo elifInfos elseInfoOpt endifLine ->
                    let list = ifInfo :: elifInfos
                    let list =
                        match elseInfoOpt with
                        | Some elseInfo -> list @ [elseInfo]
                        | _ -> list
                    IfEndifBlock (list |> List.toArray, endifLine) :> CCodeBlock)

        /// "#ifdef" 以降のマクロから "#endif" までパースします。
        let parseIfdefBlock = parseIfBlockCommon (parseIfExprAndLines parseIfdefMacro parseLine)

        /// "#ifndef" 以降のマクロから "#endif" までパースします。
        let parseIfndefBlock = parseIfBlockCommon (parseIfExprAndLines parseIfndefMacro parseLine)

        /// "#if" 以降の式から "#endif" までパースします。
        let parseIfBlock = parseIfBlockCommon (parseIfExprAndLines parseTestIfExpr parseLine)

        /// キーワードの内容に応じて、後続に呼び出すパーサを返します。
        /// elif, else, endif はパース対象としません。
        let getPpParser =
            function
                | "ifdef"  -> preturn parseIfdefBlock
                | "ifndef" -> preturn parseIfndefBlock
                | "if"     -> preturn parseIfBlock
                // エラーにしてバックトラックさせて、'#'の位置に戻させます。
                | "elif" | "else" | "endif" -> pzero
                | "include" -> preturn parsePpDirInclude
                | "error" | "warning" -> preturn parseRestOfLine
                | _        -> preturn parseCLineBlock   // 上の対象とするプリプロセッサディレクティブに当てはまらない場合
                    
        // 実体をセット。
        // クロージャの中でパーサの構築を行うと呼び出されるたびにパーサを
        // 構築してしまうため、事前にパーサを構築しています。
        do parseLineRef := 
                (skipChar '#' .>> skipSCs >>. parseId >>=? getPpParser >>= id)
            <|> (notFollowedByString "#" >>. parseCLineBlock)

        !parseLineRef

    /// 複数のコードブロックをパースします。
    let internal parseCCodeBlocks enabledMacro =
        let parseCLine = parseCCodeBlock enabledMacro

        setIndexToken >>. skipSCs
        >>. sepBy parseCLine (skipNewline >>. setIndexToken >>. skipSCs)
        .>> eof

    /// p がスキップする範囲をStringRangeで取得します。
    let getRange p =
        pipe2 (getPosition .>> p) getPosition (fun p1 p2 ->
            StringRange (int p1.Index, int (p2.Index - p1.Index)))

    /// p がパースする範囲をStringRangeでパースした結果とタプルで取得します。
    let parseWithRange p =
        pipe3 getPosition p getPosition (fun p1 result p2 ->
            (result, StringRange (int p1.Index, int (p2.Index - p1.Index))))

    /// idをパースし、idの値と範囲を取得します。
    let internal parseIdStr =
        parseWithRange parseId

    /// SimpleStringSnippet を作成します。
    let internal createSimpleStringSnippet str =
        SimpleStringSnippet str :> StringSnippet

    /// StringRangeSnippet を作成します。
    let internal createStringRangeSnippet range =
        StringRangeSnippet range :> StringSnippet

    /// DefinedSnippet を作成します。
    let internal createDefinedSnippet def (defAft: StringRange) (openRng: StringRange, (macroStr, macro), close) =
        DefinedSnippet (macroStr, def, StringRange (defAft.Offset, defAft.Length + openRng.Length), macro, close)
        :> StringSnippet

    /// defined () のパース
    let internal parseDefinedOrIdSnippet =
        parseIdStr
        >>= function
            | "defined", defRange ->
                let simpleId = parseIdStr |>> fun tp -> (StringRange.Empty, tp, StringRange.Empty)

                let parenId =
                    tuple3
                        (skipChar '(' .>> skipSCs |> getRange)
                        parseIdStr
                        (skipSCs >>. skipChar ')' |> getRange)

                pipe2
                    (getRange skipSCs)
                    (parenId <|> simpleId)
                    (createDefinedSnippet defRange)

            | _, range ->
                createStringRangeSnippet range |> preturn

    /// スキップした部分の StringRangeSnippet を取得します。
    let getRangeSnippet p = getRange p |>> createStringRangeSnippet

    /// #ifdef のマクロをパースします。
    let internal parseIfdefSnippets =
        pipe2
            (parseIdStr |>> fun tp -> createDefinedSnippet StringRange.Empty StringRange.Empty (StringRange.Empty, tp, StringRange.Empty))
            (getRangeSnippet skipSCs)
            (fun tp macroAftSCsSnippet -> [ tp; macroAftSCsSnippet ])

    // #if の式に含まれる defined(マクロ)を解析するパーサ
    let internal parseIfSnippets =
        choice [
            getRangeSnippet (skipMany1Satisfy isSymbolChar);
            parseDefinedOrIdSnippet;
            getRangeSnippet skipNumber;
            getRangeSnippet skipSingleQuote;
            getRangeSnippet skipDoubleQuote;
            getRangeSnippet skipComment;
            getRangeSnippet (skipChar '/');
            ]
        |> many

    /// #if/#if(n)def/#elif の行をパースします。
    let internal parseIfDirLine =
        pipe2
            (skipSCs >>. skipChar '#' .>> skipSCs |> getRangeSnippet)
            (parseIdStr .>>. getRangeSnippet skipSCs
                >>= fun ((ifDir, ifDirRange), ifDirAftSni) ->
                    match ifDir with
                    | "ifdef" ->    preturn (createSimpleStringSnippet "if", ifDirAftSni, None) .>>. parseIfdefSnippets
                    | "ifndef" ->   preturn (createSimpleStringSnippet "if", ifDirAftSni, createSimpleStringSnippet  "!" |> Some) .>>. parseIfdefSnippets
                    | "if" | "elif" -> preturn (createStringRangeSnippet ifDirRange, ifDirAftSni, None) .>>. parseIfSnippets
                    | _ ->  raise <| InvalidOperationException ())
            (fun ifDirPreSni ((ifDirSni, ifDirAftSni, exStrSniOpt), list) ->
                let list =
                    match exStrSniOpt with
                    | None -> list
                    | Some exStrSni -> exStrSni :: list
                ifDirPreSni :: ifDirSni :: ifDirAftSni :: list
                |> List.toArray)
        .>> eof

    /// <summary>
    /// #if ～ #endif プリプロセッサディレクティブを解析し、
    /// コードブロックを#if ～ #endif とそうでないものの単位で分けます。
    ///
    /// コードブロックは次の2つに大別されます。
    ///   ・#if 行から #endif の行を含むのコードブロック
    ///     (#ifdef または #ifndefも対象です。)
    ///   ・#if ～ #endif でないコード1行
    ///
    /// #if ～ #endif は、次のようにコードブロックを含むため再帰構造になります。
    ///     #if 式
    ///         コードブロック
    ///     #elif 式
    ///         コードブロック
    ///     #else
    ///         コードブロック
    ///     #endif
    /// </summary>
    /// <param name="str">解析対象の文字列。</param>
    /// <param name="macroMap">マクロが有効かどうかを判定するデリゲート。</param>
    /// <returns>ParseDirectiveResultオブジェクトを返します。</returns>
    [<CompiledName("ParseDirective")>]
    let parseDirective (str: string, enabledMacro: Func<string, bool>) : ParseDirectiveResult =
        let enabledMacroF id = enabledMacro.Invoke id
        let initState = { IndexToken = CharStreamIndexToken () }
        match runParserOnString (parseCCodeBlocks enabledMacroF) initState "" str with
        | Success (result, _, _) -> ParseDirectiveResult (result :> CCodeBlock seq)
        | Failure (errorMsg, parserError, _) ->
            ParseDirectiveResult (errorMsg, int parserError.Position.Index)

    /// <summary>
    /// #if の式をパースし、文字列の断片の配列を取得します。
    /// </summary>
    [<CompiledName("ParseDirectiveLine")>]
    let parseDirectiveLine (str: string) (range: StringRange) : StringSnippet array =
        match runParserOnSubstring parseIfDirLine () "" str (int range.Offset) (int range.Length) with
        | Success (result, _, _)   ->
            result
        | Failure (errorMsg, _, _) ->
            // ParseDirective が成功した文字列がくるので、基本的にはここに来ないはず。
            let msg = sprintf "Failure: %s" errorMsg
            raise <| NotImplementedException msg

