﻿namespace PackageCreator
{
    using System;
    using System.Collections.Generic;
    using System.IO;
    using System.Linq;
    using System.Text;
    using System.Threading.Tasks;

    /// <summary>
    /// パッケージ定義ファイルを読み込むクラスです。
    /// </summary>
    public class PackageDefinition
    {
        #region コンストラクタ
        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="packagePath">パッケージ定義ファイルのパス</param>
        /// <param name="sdkRoot">SDK のルート</param>
        /// <param name="extraFiles">追加ファイルリスト</param>
        /// <param name="fileTreeSelection">ファイルツリーの選択方法</param>
        /// <param name="needsPackageInfo">PackageInfo を生成するかどうか</param>
        public PackageDefinition(string packagePath, string sdkRoot, IFileGetter fileGetter, bool needsPackageInfo)
        {
            this.PackageDefinitionFileName = packagePath;
            this.ParsePackageDefinition(packagePath);
            this.ReadBundles(sdkRoot, fileGetter);
            // SIGLO-2096 SetupFiles のチェックを一時的に無くす
            // this.CheckSetupFiles();
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="packagePath">パッケージ定義ファイルのパス</param>
        /// <param name="sdkRoot">SDK のルート</param>
        public PackageDefinition(string packagePath, string sdkRoot)
            : this(packagePath, sdkRoot, new FileTreeByFileSystem(), true)
        {
        }

        /// <summary>
        /// コンストラクタ
        /// パッケージ定義ファイルのパースのみを行う
        /// </summary>
        /// <param name="packagePath">パッケージ定義ファイルのパス</param>
        private PackageDefinition(string packagePath)
        {
            this.ParsePackageDefinition(packagePath);
        }
        #endregion

        #region プロパティ
        private static readonly string[] AllowedExtension = { ".bat", ".wsf", ".js" };

        /// <summary>
        /// パッケージ定義ファイル名を表します。
        /// </summary>
        public string PackageDefinitionFileName { get; private set; }

        /// <summary>
        /// 公開タイプを表します。
        /// </summary>
        public PackageType.Type Type { get; private set; }

        /// <summary>
        /// パッケージ名を表します。
        /// </summary>
        public string PackageName { get; private set; }

        /// <summary>
        /// パッケージのルートディレクトリ名を表します。
        /// </summary>
        public string RootDirName { get; private set; }

        /// <summary>
        /// パッケージに含むバンドル名のリストを表します。
        /// </summary>
        public IEnumerable<string> BundleNames { get; private set; }

        /// <summary>
        /// パッケージに含むバンドルのリストです。
        /// </summary>
        public IEnumerable<BundleDefinition> BundleList { get; private set; }

        #endregion

        #region パッケージ定義ファイルの内容をパース
        private void ParsePackageDefinition(string packagePath)
        {
            try
            {
                dynamic data;
                using (StreamReader reader = new StreamReader(packagePath))
                {
                    data = YamlDataReader.Read(reader, "Package");
                }

                this.Type = PackageType.Convert(data["Type"]);
                this.PackageName = data["Name"];
                this.RootDirName = data.ContainsKey("RootDirName") ? data["RootDirName"] : this.PackageName;
                // SIGLO-2730 暫定的に Bundles 未指定でもエラーにしない
                this.BundleNames = data.ContainsKey("Bundles") ? (data["Bundles"] as List<object>).ConvertFromSequencialNode() : new List<string>();
            }
            catch (Exception)
            {
                Console.Error.WriteLine("ERROR: Package definition file has invalid format.");
                Console.Error.WriteLine($"\tFailed to read '${packagePath}'.");
                throw;
            }
        }
        #endregion

        #region パッケージ定義ファイルに書かれたバンドルファイルの読み込み
        private void ReadBundles(string rootPath, IFileGetter fileTree)
        {
            var bundleList = new List<BundleDefinition>();
            foreach (var bundleName in this.BundleNames)
            {
                try
                {
                    var bundle = new BundleDefinition(Path.Combine(rootPath, bundleName), rootPath, fileTree);
                    if ((this.Type | bundle.Type) != this.Type)
                    {
                        throw new FormatException(string.Format("ERROR: Cannot include following bundles in this package.\n"
                                                    + "  PackageType : {0}\n  BundleType : {1}({2})", this.Type, bundle.Type, bundleName));
                    }

                    bundleList.Add(bundle);
                }
                catch (Exception)
                {
                    Console.Error.WriteLine($"ERROR: Failed to read '${bundleName}'.");
                    throw;
                }
            }

            this.BundleList = bundleList;
        }
        #endregion

    }
}
