﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

using YamlDotNet.RepresentationModel;
using System.IO;

namespace NxAgingHelper
{
    public class CommandYamlParser
    {
        public bool EnableHdmi { get; set; } = false;
        public bool EnableMcci { get; set; } = false;
        public bool EnablePowerUsb { get; set; } = false;
        public bool EnableSdevUsb { get; set; } = false;
        public bool EnablePowerButton { get; set; } = false;
        public bool EnableBatteryLevelEmulation { get; set; } = false;

        private class ParseException : Exception
        {
            public ParseException(string message) : base(message)
            {
            }
        }

        private class Util
        {
            public static string GetNodePositionString(YamlNode node)
            {
                if (node.Start.Line == node.End.Line)
                {
                    return string.Format("at line {0}", node.Start.Line);
                }
                else
                {
                    return string.Format("between line {0} and {1}", node.Start.Line, node.End.Line);
                }
            }

            public static YamlNode GetChildNode(YamlMappingNode node, string name)
            {
                try
                {
                    return node.Children[new YamlScalarNode(name)];
                }
                catch
                {
                    throw new ParseException(string.Format("No '{0}' element {1}.", name, GetNodePositionString(node)));
                }
            }

            public static string GetStringChild(YamlMappingNode node, string name)
            {
                return GetChildNode(node, name).ToString();
            }

            public static int GetIntegerChild(YamlMappingNode node, string name)
            {
                try
                {
                    return int.Parse(GetChildNode(node, name).ToString());
                }
                catch
                {
                    throw new ParseException(string.Format("Unable to convert to integer {1}.", GetNodePositionString(node)));
                }
            }

            public static YamlSequenceNode GetListChild(YamlMappingNode node, string name)
            {
                var list = node.Children[new YamlScalarNode(name)];
                if (list == null)
                {
                    throw new ParseException(string.Format("No '{0}' element {1}.", name, GetNodePositionString(node)));
                }

                var sequenceNode = list as YamlSequenceNode;
                if (sequenceNode == null)
                {
                    throw new ParseException(string.Format("'{0}' must be a sequence node {1}.", GetNodePositionString(node)));
                }

                return sequenceNode;
            }
        }

        private ICommand MakeCommandFrom(YamlMappingNode node)
        {
            var mapping = node as YamlMappingNode;
            if (mapping == null)
            {
                throw new ParseException(string.Format("Yaml node must be dictionary{0}.", Util.GetNodePositionString(node)));
            }

            string command = Util.GetStringChild(node, "Command");

            switch (command)
            {
                case "Hdmi":
                    return MakeHdmiCommandFrom(node);
                case "Mcci":
                    return MakeMcciCommandFrom(node);
                case "PowerUsb":
                    return MakePowerUsbCommandFrom(node);
                case "SdevUsb":
                    return MakeSdevUsbCommandFrom(node);
                case "Battery":
                    return MakeBatteryLevelCommandFrom(node);
                case "PowerButton":
                    return MakePowerButtonCommandFrom(node);
                case "HostProcess":
                    return MakeHostProcessCommandFrom(node);
                case "Wait":
                    return MakeWaitCommandFrom(node);
                case "Execute":
                    return MakeExecuteCommandFrom(node);
                case "ParallelExecute":
                    return MakeParallelExecuteCommandFrom(node);
                case "Repeat":
                    return MakeRepeatCommandFrom(node);
                case "Loop":
                    return MakeLoopCommandFrom(node);
                default:
                    throw new ParseException(string.Format("Unknown command '{0}' {1}", command, Util.GetNodePositionString(node)));
            }
        }

        private IList<ICommand> MakeCommandListFrom(YamlSequenceNode node)
        {
            var list = new List<ICommand>();
            foreach (var child in node.Children)
            {
                var childMappingNode = child as YamlMappingNode;
                if (childMappingNode == null)
                {
                    throw new ParseException(string.Format("Yaml node must be dictionary {0}.", Util.GetNodePositionString(node)));
                }

                var command = MakeCommandFrom(childMappingNode);
                if (command != null)
                {
                    list.Add(command);
                }
            }

            return list;
        }

        private ICommand MakeHdmiCommandFrom(YamlMappingNode node)
        {
            if (!EnableHdmi)
            {
                return null;
            }

            string operation = Util.GetStringChild(node, "Operation");

            switch (operation)
            {
                case "Connect":
                    return new HdmiCommand() { Enable = true };
                case "Disconnect":
                    return new HdmiCommand() { Enable = false };
                default:
                    throw new ParseException(string.Format("Unknown operation '{0}' {1}", operation, Util.GetNodePositionString(node)));
            }
        }

        private ICommand MakeMcciCommandFrom(YamlMappingNode node)
        {
            if (!EnableMcci)
            {
                return null;
            }

            string operation = Util.GetStringChild(node, "Operation");

            switch (operation)
            {
                case "Connect":
                    return new McciCommand() { Enable = true };
                case "Disconnect":
                    return new McciCommand() { Enable = false };
                default:
                    throw new ParseException(string.Format("Unknown operation '{0}' {1}", operation, Util.GetNodePositionString(node)));
            }
        }

        private ICommand MakePowerUsbCommandFrom(YamlMappingNode node)
        {
            if (!EnablePowerUsb)
            {
                return null;
            }

            var port = Util.GetIntegerChild(node, "Port");
            var operation = Util.GetStringChild(node, "Operation");

            switch (operation)
            {
                case "Connect":
                    return new PowerUsbCommand() { Enable = true, Port = port };
                case "Disconnect":
                    return new PowerUsbCommand() { Enable = false, Port = port };
                default:
                    throw new ParseException(string.Format("Unknown operation '{0}' {1}", operation, Util.GetNodePositionString(node)));
            }
        }

        private ICommand MakeSdevUsbCommandFrom(YamlMappingNode node)
        {
            if (!EnableSdevUsb)
            {
                return null;
            }

            var operation = Util.GetStringChild(node, "Operation");

            switch (operation)
            {
                case "Connect":
                    return new SdevUsbCommand() { Enable = true };
                case "Disconnect":
                    return new SdevUsbCommand() { Enable = false };
                default:
                    throw new ParseException(string.Format("Unknown operation '{0}' {1}", operation, Util.GetNodePositionString(node)));
            }
        }

        private ICommand MakeBatteryLevelCommandFrom(YamlMappingNode node)
        {
            if (!EnableBatteryLevelEmulation)
            {
                return null;
            }
            var percentage = Util.GetStringChild(node, "Percentage").Replace("0x", "");
            var voltage = Util.GetStringChild(node, "Voltage").Replace("0x", "");
            return new BatteryLevelCommand()
            {
                Voltage = Int32.Parse(voltage, System.Globalization.NumberStyles.HexNumber),
                Percentage = Int32.Parse(percentage, System.Globalization.NumberStyles.HexNumber)
            };
        }

        private ICommand MakePowerButtonCommandFrom(YamlMappingNode node)
        {
            if (!EnablePowerButton)
            {
                return null;
            }

            var holdTime = Util.GetIntegerChild(node, "HoldTime");
            if (holdTime <= 100)
            {
                Console.WriteLine("Warning: HoldTime is very small (recommended value is > 100).  Possibly miss the event. -- {0}", Util.GetNodePositionString(node));
            }

            return new PowerButtonCommand() { HoldTime = holdTime };
        }

        private ICommand MakeHostProcessCommandFrom(YamlMappingNode node)
        {
            string path = Util.GetStringChild(node, "Path");
            string argument;
            try
            {
                argument = Util.GetStringChild(node, "Argument");
            }
            catch
            {
                // Argument を省略したら空文字列になる
                argument = string.Empty;
            }

            return new HostProcessCommand() { Path = path, Argument = argument };
        }

        private WaitCommand MakeWaitCommandFrom(YamlMappingNode node)
        {
            var time = Util.GetIntegerChild(node, "Time");
            if (time <= 100)
            {
                Console.WriteLine("Warning: WaitTime is very small (the recommended value is > 100).  Possibly miss the event. -- {0}", Util.GetNodePositionString(node));
            }
            return new WaitCommand(int.Parse(time.ToString()));
        }

        private ExecuteCommand MakeExecuteCommandFrom(YamlMappingNode node)
        {
            var sequenceNode = Util.GetListChild(node, "List");
            return new ExecuteCommand(MakeCommandListFrom(sequenceNode));
        }

        private ParallelExecuteCommand MakeParallelExecuteCommandFrom(YamlMappingNode node)
        {
            var sequenceNode = Util.GetListChild(node, "List");
            return new ParallelExecuteCommand(MakeCommandListFrom(sequenceNode));
        }

        private RepeatCommand MakeRepeatCommandFrom(YamlMappingNode node)
        {
            var sequenceNode = Util.GetListChild(node, "List");
            var repeatCount = Util.GetIntegerChild(node, "Count");
            return new RepeatCommand(MakeCommandListFrom(sequenceNode), repeatCount);
        }

        private LoopCommand MakeLoopCommandFrom(YamlMappingNode node)
        {
            var sequenceNode = Util.GetListChild(node, "List");
            return new LoopCommand(MakeCommandListFrom(sequenceNode));
        }

        public ExecuteCommand Parse(string ymlPath)
        {
            try
            {
                using (StreamReader reader = new StreamReader(ymlPath))
                {
                    var yml = new YamlStream();
                    yml.Load(reader);

                    var mappingNode = yml.Documents[0].RootNode as YamlMappingNode;
                    if (mappingNode == null)
                    {
                        throw new ParseException(string.Format("Yaml node must be dictionary {0}.", Util.GetNodePositionString(mappingNode)));
                    }

                    var command = MakeCommandFrom(mappingNode);
                    return new ExecuteCommand(new List<ICommand>() { command });
                }
            }
            catch (ParseException e)
            {
                throw new Exception(string.Format("Failed to parse configuration YAML file. -- {0}", ymlPath)
                    + Environment.NewLine + Environment.NewLine
                    + e.Message,
                    e);
            }
            catch (YamlDotNet.Core.YamlException e)
            {
                throw new Exception(string.Format("Failed to parse configuration YAML file. -- {0}", ymlPath)
                    + Environment.NewLine + Environment.NewLine
                    + e.Message,
                    e);
            }
            catch (IOException e)
            {
                throw new Exception(string.Format("Failed to open configuration file. -- {0}", ymlPath), e);
            }
        }
    }
}
