﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using Nintendo.NintendoSdkVsExtension.Models;
using Nintendo.NintendoSdkVsExtension.Shell;
using Microsoft.VisualStudio.Shell;
using System.Diagnostics;
using Nintendo.NintendoSdkVsExtension.LocalResources;
using Nintendo.NintendoSdkVsExtension.Resources;
using Nintendo.NintendoSdkVsExtension.Base;

namespace Nintendo.NintendoSdkVsExtension.Logic
{
    public class SdkConfigurationManagerOperation
    {
        /// <summary>
        /// SdkConfigurationManagerOperation が使う UI
        /// </summary>
        public interface ISdkConfigurationManagerUI
        {
            /// <summary>
            /// 「旧仕様のプロパティシートが登録されています以下のプロパティシートの登録を解除します」確認ダイアログ
            /// </summary>
            /// <returns>続行するなら true、キャンセルなら false</returns>
            bool QueryConfirmUpgrading();
            /// <summary>
            /// SDK 構成を編集するダイアログ
            /// </summary>
            /// <returns>編集が完了したら true、キャンセルされたら false</returns>
            bool EditSdkConfiguration(Models.SdkSolution sol);
            /// <summary>
            /// 「構成が適用できなかった」警告ダイアログ
            /// </summary>
            void WarnFailedToApplyConfiguration(IEnumerable<FailedToApplyConfiguration> failedToApplyConfigurations);
        }

        public class DefaultSdkConfigurationManagerUI : ISdkConfigurationManagerUI
        {
            private readonly IServiceProvider _serviceProvider;

            public DefaultSdkConfigurationManagerUI(IServiceProvider serviceProvider)
            {
                this._serviceProvider = serviceProvider;
            }

            public bool QueryConfirmUpgrading()
            {
                var result = VsShellUtilities.ShowMessageBox(
                    this._serviceProvider,
                    SdkConfigurationManagerResources.ConfirmUpgrading,
                    Strings.Common_DialogTitleConfirmation,
                    Microsoft.VisualStudio.Shell.Interop.OLEMSGICON.OLEMSGICON_WARNING,
                    Microsoft.VisualStudio.Shell.Interop.OLEMSGBUTTON.OLEMSGBUTTON_OKCANCEL,
                    Microsoft.VisualStudio.Shell.Interop.OLEMSGDEFBUTTON.OLEMSGDEFBUTTON_SECOND);
                return result == ShellConstants.OKButton;
            }

            public bool EditSdkConfiguration(SdkSolution solution)
            {
                // SdkConfigurationViewModel はモデルのイベントを購読するので、その購読を解除する必要がある (Dispose でやる)
                using (var vm = new ViewModels.SdkConfigurationDialogViewModel(solution))
                {
                    var dialog = new Views.SdkConfigurationDialog()
                    {
                        Owner = System.Windows.Application.Current.MainWindow,
                        ShowInTaskbar = false,
                        WindowStartupLocation = System.Windows.WindowStartupLocation.CenterOwner,
                        DataContext = vm,
                    };
                    dialog.ShowDialog();
                    return vm.DialogResult ?? false;
                }
            }

            public void WarnFailedToApplyConfiguration(IEnumerable<FailedToApplyConfiguration> failedToApplyConfigurations)
            {
                foreach (var x in failedToApplyConfigurations)
                {
                    Global.Logger.WriteGeneralLog(
                        Strings.FailedToApplyConfiguration_ValueMismatch,
                        x.Project.ProjectFilePath,
                        x.ConfigurationPair,
                        x.ExpectedConfiguration.Spec,
                        x.ExpectedConfiguration.BuildType,
                        x.ExpectedConfiguration.SdkRoot.ToEvaluatedPath(),
                        x.ActualSpec,
                        x.ActualBuildType,
                        x.ActualSdkRoot);
                }

                Global.Logger.WriteGeneralLog(Strings.FailedToApplyConfiguration);

                var result = VsShellUtilities.ShowMessageBox(
                     this._serviceProvider,
                     Strings.FailedToApplyConfiguration,
                     null,
                     Microsoft.VisualStudio.Shell.Interop.OLEMSGICON.OLEMSGICON_CRITICAL,
                     Microsoft.VisualStudio.Shell.Interop.OLEMSGBUTTON.OLEMSGBUTTON_OK,
                     Microsoft.VisualStudio.Shell.Interop.OLEMSGDEFBUTTON.OLEMSGDEFBUTTON_FIRST);
            }
        }

        private readonly IServiceProvider _serviceProvider;
        private readonly ISdkConfigurationManagerUI _sdkConfigurationManagerUI;

        public SdkConfigurationManagerOperation(IServiceProvider serviceProvider, ISdkConfigurationManagerUI ui)
        {
            this._serviceProvider = serviceProvider;
            this._sdkConfigurationManagerUI = ui;
        }

        public void Execute()
        {
            Debug.Assert(VsSolutionUtil.SolutionHasVcxproj(this._serviceProvider), "no vcxproj");

            // vcxproj を直接ロードするので、保存しておく（ビルドコマンド等と同様）
            VsSolutionUtil.SaveSolution(this._serviceProvider, Microsoft.VisualStudio.Shell.Interop.__VSSLNSAVEOPTIONS.SLNSAVEOPT_SaveIfDirty);

            // 旧仕様のNintendoSDK のプロパティシートがあったら削除するので、確認メッセージを出す
            if (SdkConfigurationManagerUtil.IsUpgradingRequired(this._serviceProvider))
            {
                if (!this._sdkConfigurationManagerUI.QueryConfirmUpgrading())
                {
                    return;
                }
            }

            var projectStates = SdkConfigurationManagerUtil.ReadProjectSettings(this._serviceProvider);
            var solutionModel = SdkConfigurationManagerUtil.ToSolutionModel(projectStates);
            if (!this._sdkConfigurationManagerUI.EditSdkConfiguration(solutionModel))
            {
                return;
            }

            // 旧仕様のプロパティシートが残っていたら削除する
            SdkConfigurationManagerUtil.UpgradePropertySheets(this._serviceProvider, solutionModel);

            // ソリューションを閉じる
            var solutionFile = VsSolutionUtil.GetSolutionInfo(this._serviceProvider).SolutionFile;
            VsSolutionUtil.CloseSolution(this._serviceProvider, Microsoft.VisualStudio.Shell.Interop.__VSSLNSAVEOPTIONS.SLNSAVEOPT_ForceSave);

            // 設定を保存する（プロジェクトへの反映、XML 書き出し）
            SdkConfigurationManagerUtil.SaveProjectSettings(solutionModel);

            // ソリューションを開く
            VsSolutionUtil.OpenSolution(this._serviceProvider, solutionFile);
            VsSolutionUtil.EnsureSolutionIsLoaded(this._serviceProvider);

            // プロパティが反映されていることを確認する
            var projectHierarchies = VsSolutionUtil.EnumerateVcxproj(this._serviceProvider);
            var failedToApplyConfigurations = SdkConfigurationManagerUtil.EnumerateFailedToApplyConfigurations(
                projectStates, projectHierarchies).ToList();
            if (failedToApplyConfigurations.Any())
            {
                // 反映されていなかったら警告を出す
                this._sdkConfigurationManagerUI.WarnFailedToApplyConfiguration(failedToApplyConfigurations);
            }
        }

        public static bool CanExecute(IServiceProvider serviceProvider)
        {
            return VsSolutionUtil.SolutionHasVcxproj(serviceProvider);
        }
    }
}
