﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <utility>
#include <new>

#include <nn/nlib/Nlist.h>
#include <nn/nlib/threading/Future.h>
#include <nn/nlib/threading/ThreadPool.h>
#include <nn/nlib/StringView.h>

#include "./AhoCorasickBuilder.h"
#include "./../../../../Programs/Eris/Sources/Libraries/ngc/succinct/ngc_AhoCorasickPrivate.h"


using std::pair;
using std::make_pair;
using std::sort;

namespace nn { namespace ngc { namespace detail {

/**
 * @brief   与えられたポインタのデータを 1 バイトごとに char に格納するクラスです
 */
class StringHolder final
{
public:
    StringHolder() NN_NOEXCEPT : m_Cur(kBufSize) {}
    ~StringHolder() NN_NOEXCEPT;

    /**
     * @brief       データをバイト単位で区切り格納します
     * @param[in]   ptr  対象のデータへのポインタ
     * @param[in]   len  対象データの長さ(bytes)
     * @details     一度に格納する長さは 8192 バイトまで許容
     *              返り値として格納された文字列の先頭ポインタを返します
    */
    char* MakeString(const void* ptr, size_t len) NN_NOEXCEPT;
private:
    static const size_t kBufSize = 1024 * 1024 * 4U;

    nn::ngc::detail::ReallocVec<char*> m_Data;  // 格納している文字列
    nn::ngc::detail::ReallocVec<char*> m_Buf;   // 格納している文字列(m_Data に入れる前の整列されていないデータ)
    size_t m_Cur;                               // m_Buf 内の次のポインタ位置

    StringHolder(const StringHolder&) = delete;
    void operator=(const StringHolder&) = delete;
};

/**
 * @brief   各ノードが表す部分文字列を表すクラスです。
 *          各ノードはその時点までの単語の接頭辞になる
 */
class PrefixString final
{
public:
    PrefixString() NN_NOEXCEPT : m_pRbegin(NULL) {}
    ~PrefixString() NN_NOEXCEPT {}
    PrefixString(const unsigned char* first, const unsigned char* last) NN_NOEXCEPT
    {
        NN_UNUSED(first);
        NLIB_ASSERT(last);
        m_pRbegin = last - 1;
    }

    /**
     * @brief   辞書順比較
     * @return  自身が rhs より小さければ true
     */
    bool ReverseLexOrderLess(const PrefixString& rhs) const NN_NOEXCEPT
    {
        NLIB_ASSERT(m_pRbegin && rhs.m_pRbegin);
        // NOTE:
        // lexicographical_compare between reversed strings
        // lexicographical_compare is very slow in DEBUG build of MSVC
        const unsigned char* rbeg = m_pRbegin;
        const unsigned char* rbeg_rhs = rhs.m_pRbegin;
        for (;;)
        {
            unsigned char c = *rbeg;
            unsigned char c_rhs = *rbeg_rhs;
            if (c < c_rhs) return true;
            if (c > c_rhs) return false;
            if (c == 0) return false;
            --rbeg;
            --rbeg_rhs;
        }
    }
    /**
     * @brief   接頭辞の先頭を返す
     */
    const unsigned char* begin() const NN_NOEXCEPT
    {
        NLIB_ASSERT(m_pRbegin);
        return m_pRbegin - length() + 1;
    }
    /**
     * @brief   接頭辞の末尾を返す
     */
    const unsigned char* end() const NN_NOEXCEPT
    {
        NLIB_ASSERT(m_pRbegin);
        return m_pRbegin + 1;
    }
    /**
     * @brief   接頭辞が空かどうか
     */
    bool empty() const NN_NOEXCEPT
    {
        NLIB_ASSERT(m_pRbegin);
        return *m_pRbegin == 0;
    }
    /**
     * @brief   接頭辞の長さを返す
     */
    size_t length() const NN_NOEXCEPT
    {
        NLIB_ASSERT(m_pRbegin);
        const unsigned char* p;
        const unsigned char* rbeg = m_pRbegin;
        for (p = rbeg; *p; --p)
        {   // NOTE: 接頭辞の始まりは 0 (root)であることを前提としている
        }
        return rbeg - p;
    }
    /**
     * @brief   葉かどうか・・ではなくこれが単語の終わりかどうかを返します
     */
    bool IsTerminal() const NN_NOEXCEPT
    {
        NLIB_ASSERT(m_pRbegin);
        return !*(m_pRbegin + 1);
    }

private:
    const unsigned char* m_pRbegin;
};

/**
 * @brief   AC 法の各ノードを表すクラスです。
 *          ノードの id は接頭辞の昇順で付けられます
 */
class AcNode final
{
public:
     // ノードが持つエッジ
     // 単方向リストで管理
    struct EdgeList
    {
        unsigned char label;    // エッジのラベル（1 文字）
        AcNode* dest;           // エッジの伸び先ノード
        EdgeList* next;
    };

public:
    // コンストラクタ時の m_pFailure は NULL
    AcNode() NN_NOEXCEPT : m_NodeId(0), m_pFailure(NULL), m_pEdges(NULL) {}
    ~AcNode() NN_NOEXCEPT {}
    bool operator<(const AcNode& rhs) const NN_NOEXCEPT
    {
        return m_Prefix.ReverseLexOrderLess(rhs.m_Prefix);
    }
    /**
     * @brief   level の深さの文字は何かを返します
     */
    unsigned char GetLevelValue(size_t level) const NN_NOEXCEPT
    {
        return level < m_NodeId ? *(m_Prefix.end() - 1 - level) : 0;
    }
    /**
     * @brief   ノード番号を返します
     */
    uint32_t GetNodeId() const NN_NOEXCEPT
    {
        return m_NodeId;
    }
    /**
     * @brief   単語の完結かどうかを返します
     */
    bool IsTerminal() const NN_NOEXCEPT
    {
        return m_NodeId != 0 && m_Prefix.IsTerminal();
    }
    /**
     * @brief       goto() を追加します
     * @param[in]   c       与えられた文字
     * @param[out]  next    c の時の行先ノード
     * @param[out]  holder  木の中にあるすべてのエッジを管理するリスト
     *                      エッジのメモリ確保のために必要
     */
    bool AddGoto(unsigned char c, AcNode* next,
                 nlib_ns::Nlist<AcNode::EdgeList>* holder) NN_NOEXCEPT;
    /**
     * @brief   ノードの描画
     */
    void print() NN_NOEXCEPT;
    /**
     * @brief   goto() 関数です
     * @return  c を与えたときの行先ノードを返します。
     */
    const AcNode* Goto(const unsigned char c) const NN_NOEXCEPT;
    /**
     * @brief   first - last の文字列がこのノード以下にあるかどうか探します
     * @details 探して一致したノードまで行き、その位置（ノード）を返します
     *          （一つも見つからなければこのノードが返る）
     */
    AcNode* Find(const unsigned char* first,
                 const unsigned char* last) NN_NOEXCEPT;
    /**
     * @brief   failure() で得られるノードをたどります
     */
    AcNode* GetReportArc() const NN_NOEXCEPT
    {
        // NOTE: たどり続けると root に行くのでは？
        //       → 途中で単語の終端にたどり着く可能性があるのでそうは限らない
        AcNode* failure;
        for (failure = m_pFailure;
             failure != NULL && !failure->IsTerminal();
             failure = failure->m_pFailure) {
        }
        return failure;
    }
    /**
     * @brief   failure() 関数です
     * @details goto() 失敗時の行先です
     */
    AcNode* GetFailureArc() const NN_NOEXCEPT
    {
        return m_pFailure;
    }
    /**
     * @brief   このノードが持つエッジのリストを返します
     */
    const EdgeList* GetEdgeList() const NN_NOEXCEPT
    {
        return m_pEdges;
    }
    /**
     * @brief   c をラベルに持つエッジがあるかを探します
     */
    AcNode* FindArc(unsigned char c) const NN_NOEXCEPT
    {
        for (EdgeList* e = m_pEdges; e != NULL; e = e->next)
        {
            if (e->label == c)
            {
                return e->dest;
            }
        }
        return NULL;
    }
    /**
     * @brief   このノードが表す接頭辞を返します
     */
    PrefixString GetPrefix() const NN_NOEXCEPT
    {
        return m_Prefix;
    }

private:
    void SetNodeId(uint32_t nodeid) NN_NOEXCEPT
    {
        m_NodeId = nodeid;
    }
    void SetFailureArc(AcNode* failure) NN_NOEXCEPT
    {
        m_pFailure = failure;
    }
    /**
     * @brief   根からこのノードまでで表される文字列（接頭辞）を設定します
     */
    void SetPrefix(PrefixString prefix) NN_NOEXCEPT
    {
        m_Prefix = prefix;
        // ノード id は一旦接頭辞の長さが入れられる
        // (ACBuilder で後程接頭辞の昇順でソートされる)
        m_NodeId = static_cast<uint32_t>(m_Prefix.length());
    }

private:
    uint32_t m_NodeId;  // suffix-lexicographic order, m_NodeId == 0 if root
    PrefixString m_Prefix;
    AcNode* m_pFailure;
    EdgeList* m_pEdges;     // ノードが持つエッジのリスト

    friend class AhoCorasickBuilder;
    AcNode(const AcNode&) = delete;
    void operator=(const AcNode&) = delete;
};

StringHolder::~StringHolder() NN_NOEXCEPT
{
    for (size_t i = 0; i < m_Buf.Size(); ++i)
    {
        nlib_free_size(m_Buf[i], kBufSize);
    }
}

char* StringHolder::MakeString(const void* ptr, size_t len) NN_NOEXCEPT
{
    // ensure '\0' retval '\0'
    if (len >= 8192)
    {
        NLIB_ASSERT(len < 8192);
        return NULL;
    }
    // kBufSize を超える場合は新たな kBufSize 分を m_Buf に用意する
    if (m_Cur + len + 1 > kBufSize)
    {
        char* mem = reinterpret_cast<char*>(nlib_malloc(kBufSize));
        if (!mem)
        {
            return NULL;
        }
        if (!m_Buf.PushBack(mem))
        {
            nlib_free_size(mem, kBufSize);
            return NULL;
        }
        mem[0] = '\0';
        m_Cur = 1;
    }
    // NOTE: m_Buf は m_Cur + len + 1 > kBufSize が続くとデータはとびとびになる？
    char* buf = m_Buf.Back();
    memcpy(buf + m_Cur, ptr, len);
    buf[m_Cur + len] = '\0';
    char* rval = &buf[m_Cur];
    m_Cur += len + 1;
    if (!m_Data.PushBack(rval))
    {
        return NULL;
    }
    return rval;
}

bool AcNode::AddGoto(unsigned char c, AcNode* next,
                     nlib_ns::Nlist<EdgeList>* holder) NN_NOEXCEPT
{
    EdgeList* e = holder->push_back();
    if (!e)
    {
        NLIB_ASSERT(e);
        return false;
    }
    e->label = c;
    e->dest = next;
    e->next = m_pEdges;
    m_pEdges = e;         // 新たに生成したエッジをエッジのリストに追加
    return true;
}

// ノードの描画
// Build() 後にやること
// 深さ: [このノードまでの接頭辞] (持つエッジのラベル, 行先) ...
// } #  単語の終点か（葉であるとは限らない）
// | F: failure の進み先
void AcNode::print() NN_NOEXCEPT
{
    nlib_printf("%" PRIu32 ": [", GetNodeId());
    for (size_t i = 0; i < GetPrefix().length(); ++i)
    {
        nlib_printf("%x,", static_cast<unsigned char>(*(GetPrefix().begin() + i)));
    }
    nlib_printf("] ");

    EdgeList* e = m_pEdges;
    while (e)
    {
        nlib_printf("('%x',%" PRIu32 "),", static_cast<int>(e->label), e->dest->GetNodeId());
        e = e->next;
    }
    if (m_pFailure)
    {
        nlib_printf("| F:%" PRIu32, m_pFailure->GetNodeId());
    }
    AcNode* report = GetReportArc();
    if (report)
    {
        nlib_printf("| R:%" PRIu32, report->GetNodeId());
    }
    if (IsTerminal())
    {
        nlib_printf("| #");
    }
    nlib_printf("\n");
}

// goto() 関数です
// c を与えたときの行先ノードを返します。
// failure もたどる
const AcNode* AcNode::Goto(const unsigned char c) const NN_NOEXCEPT
{
    const AcNode* x;
    for (const AcNode* tmp = this; (x = tmp->FindArc(c)) == NULL; tmp = tmp->m_pFailure)
    {
        if (!tmp->m_pFailure) return tmp;
    }
    return x;
}

AcNode* AcNode::Find(const unsigned char* first, const unsigned char* last) NN_NOEXCEPT
{
    AcNode* x = this;
    for (; first != last; ++first)
    {
        x = x->FindArc(*first);
        if (!x) break;
    }
    return x;
}

// #define MSG(...) nlib_printf(__VA_ARGS__)
#define MSG(...)

typedef nn::ngc::detail::ReallocVec<AcNode*> AcNodeHolder;

// AhoCorasickBuilder の内部実装構造体
struct AhoCorasickBuilder::AhoCorasickBuilderPrivate
{
    size_t numWords;      // 検索できる単語の数(≠葉の数)
    size_t numBytes;      // 検索できる単語の総バイト数
    AcNodeHolder acnodeHolder;    // 作成された木構造

    // ノードのメモリ領域を管理するためのインスタンス
    // 全ノードは acnodeMem.push_back() で生成される
    nlib_ns::Nlist<AcNode> acnodeMem;

    // ノードから生えるエッジのメモリ領域を管理するためのインスタンス
    // 全エッジは edgeMem.push_back() で生成される
    nlib_ns::Nlist<AcNode::EdgeList> edgeMem;

    StringHolder stringHolder;
    nlib_ns::threading::ThreadPool threadPool;    // 並列処理用スレッドプール
};

AhoCorasickBuilder::~AhoCorasickBuilder() NN_NOEXCEPT
{
    delete m_pPrv;
}

bool AhoCorasickBuilder::Init() NN_NOEXCEPT
{
    if (m_pPrv)
    {
        return true;
    }
    std::unique_ptr<AhoCorasickBuilderPrivate> impl(new (std::nothrow) AhoCorasickBuilderPrivate());
    if (!impl)
    {
        return false;
    }
    // スレッドプールの作成
    // 作成するスレッドの数はハードウェアスレッド数
    if (impl->threadPool.Init() != 0)
    {
        return false;
    }

    // ルートノードを追加
    static const unsigned char dummy[] = {0, 0};

    if (!impl->acnodeHolder.Empty())
    {
        return false;
    }
    AcNode* p = impl->acnodeMem.push_back();  // 空要素を push -> 返り値として取得
    if (!p)
    {
        return false;
    }
    p->SetPrefix(PrefixString(&dummy[1], &dummy[1]));   // ルートノードの prefix は 0
    if (!impl->acnodeHolder.PushBack(p))
    {
        impl->acnodeMem.pop_back();
        return false;
    }
    impl->numWords = 0;
    impl->numBytes = 0;

    m_pPrv = impl.release();
    return true;
}

size_t AhoCorasickBuilder::GetNumWords() const NN_NOEXCEPT
{
    return m_pPrv ? m_pPrv->numWords : 0;
}
size_t AhoCorasickBuilder::GetNumBytes() const NN_NOEXCEPT
{
    return m_pPrv ? m_pPrv->numBytes : 0;
}
size_t AhoCorasickBuilder::GetNumNodes() const NN_NOEXCEPT
{
    return m_pPrv ? m_pPrv->acnodeHolder.Size() : 0;
}

bool AhoCorasickBuilder::AddWord(const char* str) NN_NOEXCEPT
{
    // NULL, ""は登録できない。
    if (!str || str[0] == '\0')
    {
        return false;
    }
    return this->AddPattern(str, nlib_strlen(str));
}

bool AhoCorasickBuilder::AddPattern(const void* ptr, size_t n) NN_NOEXCEPT
{
    if (!ptr || n == 0)
    {
        return false;
    }
    if (!m_pPrv)
    {
        return false;
    }

    AcNode& root = *m_pPrv->acnodeHolder[0];
    // ptr を文字列として取得（\0 も付加される）
    char* str = m_pPrv->stringHolder.MakeString(ptr, n);
    if (!str)
    {
        return false;
    }
    const unsigned char* beg = reinterpret_cast<const unsigned char*>(str);
    const unsigned char* end = beg + strlen(str);   // ここには \0 が入っている
    const unsigned char* p = beg;
    AcNode* node = &root;

    // 文字列を 1 文字ずつ見ていく
    for (; p != end; ++p)
    {
        AcNode* next = node->Find(p, p + 1);    // ノードから該当文字のエッジが出ているか
        if (!next)
        {   // エッジがまだ作成されていないならエッジを作成する
            PrefixString prefix(beg, p + 1);            // 作成するエッジまでのルートからの文字列(接頭辞)
            next = m_pPrv->acnodeMem.push_back();     // ノード用のメモリ領域を取得
            if (!next)
            {
                return false;
            }
            if (!m_pPrv->acnodeHolder.PushBack(next))
            {
                m_pPrv->acnodeMem.pop_back();
                return false;
            }
            next->SetPrefix(prefix);
            // AddGoto で作成したノードと現在のノードをエッジでつなぐ
            if (!node->AddGoto(*p, next, &m_pPrv->edgeMem))
            {
                return false;
            }
        }
        node = next;
    }
    node->SetPrefix(PrefixString(beg, end));  // 終端記号としてマークする
    NLIB_ASSERT(PrefixString(beg, end).IsTerminal());
    // 登録数のインクリメント
    ++m_pPrv->numWords;
    m_pPrv->numBytes += n;
    return true;
}

struct AhoCorasickBuilder::BuildLenHolderTh
{
    bool operator()() NN_NOEXCEPT;
    AhoCorasickBuilder::AhoCorasickBuilderPrivate* This;
    nn::ngc::detail::AhoCorasick::AhoCorasickPrivate* obj;
};

bool AhoCorasickBuilder::BuildLenHolderTh::operator()() NN_NOEXCEPT
{
    nlib_time base, t;
    nlib_epochtime(&base);
    // ターミナルになるNodeId -> マッチした長さ
    // を求める辞書を作成する。
    // NodeIdがターミナルでない場合は-1を返す。
    AcNodeHolder::const_iterator it;
    AcNodeHolder::const_iterator it_end = This->acnodeHolder.End();
    nn::ngc::detail::ReallocVec<const AcNode*> vec_terminal;
    for (it = This->acnodeHolder.Begin(); it != it_end; ++it)
    {
        if ((*it)->IsTerminal())
        {
            if (!vec_terminal.PushBack(*it))
            {
                return false;
            }
        }
    }
    size_t tm_size = vec_terminal.Size();
    if (!obj->lenHolder.Init(This->acnodeHolder.Size()))
    {
        return false;
    }
    // どのインデックス(NodeId)に値を入れるか決定する
    for (size_t i = 0; i < tm_size; ++i)
    {
        uint32_t len = static_cast<uint32_t>(vec_terminal[i]->GetPrefix().length());
        obj->lenHolder.TurnOn(vec_terminal[i]->GetNodeId(), len);
    }
    bool result = obj->lenHolder.Build();
    nlib_epochtime(&t);
    MSG("BuildLenHolder: %" PRId64 "\n", (t - base) / 10000);
    return result;
}

// AC のオートマトンの goto() 部分を簡潔データ構造で作るための構造体
struct AhoCorasickBuilder::BuildGotoArcHolderTh
{
    bool operator()() NN_NOEXCEPT;
    AhoCorasickBuilder::AhoCorasickBuilderPrivate* This;    // 元となるオートマトン
    AhoCorasick::AhoCorasickPrivate* obj;                   // 出来上がる簡潔データ構造
};

// AC のオートマトンの goto() 部分だけを抜き出した GotoArcHolder クラスを
// 簡潔データ構造で作成する
bool AhoCorasickBuilder::BuildGotoArcHolderTh::operator()() NN_NOEXCEPT
{
    nlib_time base, t;
    nlib_epochtime(&base);
    obj->gotoarcHolder.Init(static_cast<uint32_t>(This->acnodeHolder.Size()));
    AcNodeHolder::iterator it;
    AcNodeHolder::iterator it_end = This->acnodeHolder.End();
    // goto を表す簡潔データ構造の生成
    // 各ノードについて、該当部分のビットを立てていく
    for (it = This->acnodeHolder.Begin(); it != it_end; ++it)
    {
        AcNode* node = *it;
        uint32_t idx = node->GetNodeId();
        for (const AcNode::EdgeList* e = node->GetEdgeList(); e != NULL; e = e->next)
        {
            obj->gotoarcHolder.TurnOn(idx, e->label);
        }
    }
    bool result = obj->gotoarcHolder.Build();
    nlib_epochtime(&t);
    MSG("BuildGotoArcHolder: %" PRId64 "\n", (t - base) / 10000);
    return result;
}

// failure で移動した場合に単語の終端にたどり着くかどうかを表す簡潔データ構造(Map) を作成する構造体です。
struct AhoCorasickBuilder::BuildReportTreeHolderTh
{
    bool operator()() NN_NOEXCEPT;
    AhoCorasickBuilder::AhoCorasickBuilderPrivate* This;
    AhoCorasick::AhoCorasickPrivate* obj;
};

// 以下の Map を作成します
// サイズ : ノードの数
// キー : ノードid
// 値 : report で移動した際にたどり着く単語の終端のノード id
bool AhoCorasickBuilder::BuildReportTreeHolderTh::operator()() NN_NOEXCEPT
{
    nlib_time t, base;
    nlib_epochtime(&base);

    obj->reporttreeHolder.Init(This->acnodeHolder.Size());
    AcNodeHolder::const_iterator it;
    AcNodeHolder::const_iterator it_end = This->acnodeHolder.End();
    for (it = This->acnodeHolder.Begin(); it != it_end; ++it)
    {
        const AcNode& node = **it;
        const AcNode* report = node.GetReportArc();
        if (report)
        {
            obj->reporttreeHolder.TurnOn(node.GetNodeId(), report->GetNodeId());
        }
    }
    bool result = obj->reporttreeHolder.Build();

    nlib_epochtime(&t);
    MSG("BuildReportTreeHolder: %" PRId64 "\n", (t - base) / 10000);
    return result;
}

namespace {

// AC の有限オートマトンの failure についてを辿るためのイテレータです
// Bp の作成で利用されます
// 接頭辞の昇順でソートすると、ちょうど failure で構成したい木構造における
// 深さ優先探索の順にノード id が振られることになるので、 nodeid を単純にインクリメントするだけで
// ノードをたどることができる
class TreeIter
{
public:
    explicit TreeIter(const nn::ngc::detail::ReallocVec<detail::AcNode*>& rhs) NN_NOEXCEPT : m_NodeId(0),
                                                                              m_pHolder(&rhs) {}
    // ソートされているのでFailureで構成される木に対する深さ優先探索になる。
    bool MoveNext() NN_NOEXCEPT
    {
        if (m_NodeId + 1 < m_pHolder->Size())
        {
            ++m_NodeId;
            return true;
        }
        return false;
    }
    const void* NodePtr() const NN_NOEXCEPT
    {
        return (*m_pHolder)[m_NodeId];
    }
    // 親を failure の戻り先に設定する
    const void* ParentNodePtr() const NN_NOEXCEPT
    {
        return (*m_pHolder)[m_NodeId]->GetFailureArc();
    }

private:
    size_t m_NodeId;
    const nn::ngc::detail::ReallocVec<detail::AcNode*>* m_pHolder;
};

}  // namespace

// AhoCorasickBuilderPrivate の木構造に対して failure 関数を設定するための構造体
// スレッドプールで並列処理するため構造体として定義されている
struct AhoCorasickBuilder::BuildFailureArcTh
{
    bool operator()() NN_NOEXCEPT;
    AhoCorasickBuilder::AhoCorasickBuilderPrivate* This;
};

// ACBuilderPrivate の木構造に対して failure 関数を構築します
bool AhoCorasickBuilder::BuildFailureArcTh::operator()() NN_NOEXCEPT
{
    nlib_time t, base;
    nlib_epochtime(&base);

    AcNode* root = This->acnodeHolder[0];

    // 幅優先検索してfailureリンクを作成する。
    nlib_ns::Nlist<AcNode*> curLevel, nextLevel;
    AcNode** p = nextLevel.push_back();
    if (!p)
    {
        return false;
    }
    *p = This->acnodeHolder[0];   // p に root を入れる

    while (!nextLevel.empty())
    {
        std::swap(curLevel, nextLevel); // 次の階層のノード一覧を取得
        nlib_ns::Nlist<AcNode*>::iterator it;
        nlib_ns::Nlist<AcNode*>::iterator it_end = curLevel.end();
        for (it = curLevel.begin(); it != it_end; ++it)
        {
            AcNode* node = *it;
            // 各エッジについてみていく
            for (const AcNode::EdgeList* e = node->GetEdgeList();
                 e != NULL;
                 e = e->next)
            {
                AcNode* failure;
                // rootのm_pFailureはNULLのまま
                // 親のfailureリンクからit->firstを辿れる場合はそこを
                // 自身のfailureリンクとする
                for (failure = node->GetFailureArc();
                     failure != NULL;
                     failure = failure->GetFailureArc())
                {
                    AcNode* dest = failure->FindArc(e->label);
                    if (dest)
                    {
                        e->dest->SetFailureArc(dest);
                        break;
                    }
                }
                if (!failure)
                {
                    // failure が NULL なら failure に root を設定する
                    e->dest->SetFailureArc(root);
                }
                p = nextLevel.push_back();     // nextLevel にこのエッジの 1 つ下の子をためる
                if (!p)
                {
                    return false;
                }
                *p = e->dest;
            }
        }
        curLevel.clear();  // 使い終わったのでクリア
    }
    NLIB_ASSERT(!root->GetFailureArc());
    nlib_epochtime(&t);
    MSG("BuildFailureArc: %" PRId64 "\n", (t - base) / 10000);
    return true;
}

// failure に関する木構造を作成するための構造体
struct AhoCorasickBuilder::BuildFailureTreeHolderTh
{
    bool operator()() NN_NOEXCEPT;
    AhoCorasickBuilder::AhoCorasickBuilderPrivate* This;
    AhoCorasick::AhoCorasickPrivate* obj;
};

bool AhoCorasickBuilder::BuildFailureTreeHolderTh::operator()() NN_NOEXCEPT
{
    nlib_time base, t;
    nlib_epochtime(&base);
    TreeIter iter(This->acnodeHolder);
    bool result = obj->failuretreeHolder.Init(
        iter, static_cast<uint32_t>(This->acnodeHolder.Size()));
    nlib_epochtime(&t);
    MSG("BuildFailureTreeHolder: %" PRId64 "\n", (t - base) / 10000);
    return result;
}

namespace {
inline bool LessFunc(const detail::AcNode* lhs, const detail::AcNode* rhs) NN_NOEXCEPT
{
    return (*lhs) < (*rhs);
}

inline unsigned char GetLevelValue(const detail::AcNode* lhs, size_t level) NN_NOEXCEPT
{
    return lhs->GetLevelValue(level);
}
}  // namespace

static void MultiKeySort(detail::AcNode** first, detail::AcNode** last, size_t level)
{
    // マルチキークイックソートを実装している
    using std::swap;
    int n = static_cast<int>(last - first);
    if (n <= 8)
    {
        std::sort(first, last, LessFunc);
        return;
    }
    // firstにpivotを持ってくる
    swap(*first, *(first + n / 2));
    unsigned char pivot = GetLevelValue(*first, level);
    int a, b, c, d, r;
    for (a = b = 1, c = d = n - 1;; ++b, --c)
    {
        for (; b <= c && (r = GetLevelValue(*(first + b), level) - pivot) <= 0; ++b)
        {
            if (r == 0)
            {
                swap(*(first + a), *(first + b));
                ++a;
            }
        }
        for (; b <= c && (r = GetLevelValue(*(first + c), level) - pivot) >= 0; --c)
        {
            if (r == 0)
            {
                swap(*(first + c), *(first + d));
                --d;
            }
        }
        if (b > c) break;
        swap(*(first + b), *(first + c));
    }
    {
        r = std::min(a, b - a);
        NLIB_ASSUME(r >= 0);
        NLIB_ASSERT(r >= 0);
        detail::AcNode** p0 = first;
        detail::AcNode** p1 = first + b - r;
        detail::AcNode** p0end = first + r;
        for (; p0 != p0end; ++p0, ++p1)
        {
            swap(*p0, *p1);
        }
    }

    {
        r = std::min(d - c, n - d - 1);
        NLIB_ASSUME(r >= 0);
        NLIB_ASSERT(r >= 0);
        detail::AcNode** p0 = first + b;
        detail::AcNode** p1 = first + n - r;
        detail::AcNode** p0end = first + b + r;
        for (; p0 != p0end; ++p0, ++p1)
        {
            swap(*p0, *p1);
        }
    }

    r = b - a;
    MultiKeySort(first, first + r, level);

    if (GetLevelValue(*(first + r), level) != 0)
    {
        MultiKeySort(first + r, first + r + a + n - d - 1, level + 1);
    }

    r = d - c;
    MultiKeySort(first + n - r, first + n, level);
}

// スレッドプールを使ったソートの並列化をするための構造体です
struct AhoCorasickBuilder::SortNodesTh
{
    void operator()() NN_NOEXCEPT;
    AhoCorasickBuilder::AhoCorasickBuilderPrivate* This;
    int from;
    int to;
    const int* count;
};

void AhoCorasickBuilder::SortNodesTh::operator()() NN_NOEXCEPT
{
    int prev = from == 0 ? 0 : count[from - 1];
    for (int j = from; j < to; ++j)
    {
        if (prev != count[j])
        {
            MultiKeySort(This->acnodeHolder.Begin() + prev,
                         This->acnodeHolder.Begin() + count[j], 2);
            prev = count[j];
        }
    }
}

// ノードの情報を接頭辞の昇順でソートし、ノード id もその順番で振り直します
bool AhoCorasickBuilder::SortNodes() NN_NOEXCEPT
{
    // 最初の2文字をカウンティングソートして、その後マルチキークイックソートする。
    const int kRange = 256 * 256;
    std::unique_ptr<int[]> count(new int[kRange]);
    memset(count.get(), 0, sizeof(count[0]) * kRange);
    {
        AcNodeHolder::iterator it;
        AcNodeHolder::iterator it_end = m_pPrv->acnodeHolder.End();
        for (it = m_pPrv->acnodeHolder.Begin(); it != it_end; ++it)
        {
            size_t idx = ((*it)->GetLevelValue(0) << 8) + (*it)->GetLevelValue(1);
            ++count[idx];
        }

        int total = 0;
        int old_count;
        for (int i = 0; i < kRange; ++i)
        {
            old_count = count[i];
            count[i] = total;
            total += old_count;
        }

        {
            AcNodeHolder output;
            output.Resize(m_pPrv->acnodeHolder.Size());  // sorry, overhead!!
            for (it = m_pPrv->acnodeHolder.Begin(); it != it_end; ++it)
            {
                size_t idx = ((*it)->GetLevelValue(0) << 8) + (*it)->GetLevelValue(1);
                output[count[idx]] = *it;
                ++count[idx];
            }
            using std::swap;
            swap(m_pPrv->acnodeHolder, output);
        }
    }

    bool mt_fail = false;
    unsigned int num_cpu;
    if (nlib_thread_getconcurrency(&num_cpu) != 0)
    {
        num_cpu = 1;
    }
    int batch_size = count[kRange - 1] / num_cpu;
    {
        typedef nlib_ns::Nlist<nlib_ns::threading::Future<void> > FutureVec;
        FutureVec futures;
        if (!futures.reserve(num_cpu))
        {
            return false;
        }
        int from = 0;
        int count_from = 0;
        for (int i = 0; i < kRange; ++i)
        {
            int to = i;
            if (to == kRange - 1 ||
                count[to] - count_from > batch_size)
            {
                SortNodesTh th;
                th.count = count.get();
                th.from = from;
                th.to = to;
                th.This = this->m_pPrv;
                if (to != kRange - 1)
                {
                    futures.push_back();
                    if (0 != m_pPrv->threadPool.Submit(&futures.back(), th))
                    {
                        futures.pop_back();
                        mt_fail = true;
                        break;
                    }
                    from = to;
                    count_from = count[to];
                }
                else
                {
                    th();
                }
            }
        }
        FutureVec::iterator it;
        FutureVec::iterator it_end = futures.end();
        for (it = futures.begin(); it != it_end; ++it)
        {
            if (0 != it->Wait())
            {
                mt_fail = true;
                break;
            }
        }
    }
    if (mt_fail)
    {
        // ダメだったらシングルスレッドで計算する
        int prev = 0;
        for (int i = 0; i < kRange; ++i)
        {
            MultiKeySort(m_pPrv->acnodeHolder.Begin() + prev,
                         m_pPrv->acnodeHolder.Begin() + count[i], 2);
            prev = count[i];
        }
    }
    {
        uint32_t nodeid = 0;
        AcNodeHolder::iterator it2 = m_pPrv->acnodeHolder.Begin();
        AcNodeHolder::iterator it2_end = m_pPrv->acnodeHolder.End();
        for (; it2 != it2_end; ++it2)
        {
            // NodeIdの設定
            (*it2)->SetNodeId(nodeid++);
        }
    }
    return true;
}   // NOLINT(impl/function_size)

AhoCorasick* AhoCorasickBuilder::Build() NN_NOEXCEPT
{
    if (!m_pPrv) return NULL;
    m_pPrv->acnodeMem.shrink_to_fit();
    m_pPrv->edgeMem.shrink_to_fit();

    AhoCorasick* obj = new (std::nothrow) AhoCorasick();
    if (!obj)
    {
        return NULL;
    }
    std::unique_ptr<AhoCorasick> pObj(obj);
    obj->m_pPrv = new (std::nothrow) AhoCorasick::AhoCorasickPrivate();
    if (!obj->m_pPrv)
    {
        return NULL;
    }

    nlib_time t;
    nlib_time base;

    nlib_epochtime(&base);
    MSG("AhoCorasickBuilder::Build()\n");

    // ノードの情報を接頭辞の昇順でソート
    if (!this->SortNodes())
    {
        return NULL;
    }
    nlib_epochtime(&t);
    MSG("Sorting Nodes: %" PRId64 "\n", (t - base) / 10000);

    // 依存関係図
    // Sort -> GotoArc
    //      -> LenHolder
    //      -> FailureArc -> FailureTree
    //                    -> ReportTree
    errno_t e;

    // 以下スレッドプールを使って並列処理

    // failure 関数の構築
    BuildFailureArcTh failuerArcTh;
    failuerArcTh.This = this->m_pPrv;
    nlib_ns::threading::Future<bool> futureFailureArc;
    e = m_pPrv->threadPool.Submit(&futureFailureArc, failuerArcTh);
    if (e != 0)
    {
        return NULL;
    }

    // 簡潔データ構造(SparseSet) を用いて goto を省メモリ化し AhoCorasick クラスへ格納
    BuildGotoArcHolderTh gotoArcHolderTh;
    gotoArcHolderTh.obj = obj->m_pPrv;
    gotoArcHolderTh.This = this->m_pPrv;
    nlib_ns::threading::Future<bool> futureGoto;
    e = m_pPrv->threadPool.Submit(&futureGoto, gotoArcHolderTh);
    if (e != 0)
    {
        return NULL;
    }

    // 完結する単語を持つ接頭辞の長さを返す簡潔データ構造(Map)を作成するための構造体
    BuildLenHolderTh lenHolderTh;
    lenHolderTh.obj = obj->m_pPrv;
    lenHolderTh.This = this->m_pPrv;
    nlib_ns::threading::Future<bool> futureLen;
    e = m_pPrv->threadPool.Submit(&futureLen, lenHolderTh);
    if (e != 0)
    {
        return NULL;
    }

    // ここから下は failure 構築済みになってからやる
    if (!futureFailureArc.Get())
    {
        return NULL;
    }

    // failure で表される木構造を構築する
    BuildFailureTreeHolderTh failureTreeHolderTh;
    failureTreeHolderTh.obj = obj->m_pPrv;
    failureTreeHolderTh.This = this->m_pPrv;
    nlib_ns::threading::Future<bool> futureFailure;
    e = m_pPrv->threadPool.Submit(&futureFailure, failureTreeHolderTh);
    if (e != 0)
    {
        return NULL;
    }

    // report 用の簡潔データ構造(Map) を作成
    BuildReportTreeHolderTh reportTreeHolderTh;
    reportTreeHolderTh.obj = obj->m_pPrv;
    reportTreeHolderTh.This = this->m_pPrv;
    if (!reportTreeHolderTh())
    {
        return NULL;
    }

    if (!futureGoto.Get())
    {
        return NULL;
    }
    if (!futureLen.Get())
    {
        return NULL;
    }
    if (!futureFailure.Get())
    {
        return NULL;
    }
    return pObj.release();
}

#undef MSG

void AhoCorasickBuilder::MatchByBuilder(const char* pDoc, MatchCallback callback,
                                        void* pUserObj) NN_NOEXCEPT
{
    const char* p = pDoc;
    const AcNode* node = m_pPrv->acnodeHolder[0];

    for (; *p; ++p)
    {
        unsigned char c = *reinterpret_cast<const unsigned char*>(p);
        node = node->Goto(c);
        if (node->IsTerminal())
        {
            if (callback)
            {
                size_t len = node->GetPrefix().length();
                if (!(*callback)(p - len + 1, p + 1, node->GetNodeId(), pUserObj))
                {
                    return;
                }
            }
        }
        const AcNode* report = node;
        while ((report = report->GetReportArc()) != NULL)
        {
            if (callback)
            {
                size_t len = report->GetPrefix().length();
                if (!(*callback)(p - len + 1, p + 1, report->GetNodeId(), pUserObj))
                {
                    return;
                }
            }
        }
    }
}

bool AhoCorasickBuilder::AddWords(const char* str, size_t len) NN_NOEXCEPT
{
    nlib_ns::StringView sview(str, len);
    while (!sview.empty())
    {
        nlib_ns::StringView line = nlib_ns::GetLine(sview);
        size_t line_len = line.length();
        if (line_len >= 8191)
        {
            return false;
        }
        if (line_len > 0)
        {
            if (!this->AddPattern(&*line.begin(), line_len))
            {
                return false;
            }
        }
    }
    return true;
}

}}} // nn::ngc::detail
