﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cassert>

#include <nn/util/util_CharacterEncoding.h>

#include <nn/nlib/Config.h>

#include "./../../../../Programs/Eris/Sources/Libraries/ngc/succinct/ngc_BinaryWriter.h"
#include "./../../../../Programs/Eris/Sources/Libraries/ngc/succinct/ngc_AhoCorasick.h"
#include "./succinct/FileOutputStream.h"
#include "./succinct/AhoCorasickBuilder.h"

#include "NgWordConverter.h"

namespace {

// NG ワードリストを読み込む
//  - pOut          読み込み先バッファ
//  - pOutSize      読み込みサイズ
//  - path          読み込み元ファイルパス
//  - fileName      読み込み元ファイル名
errno_t ReadNgWord(std::unique_ptr<char16_t[]>& pOut, size_t* pOutSize, std::string path, std::string fileName)
{
    // NG ワードリストの読み込み
    // NG ワードリストは UTF-16 BE で記述されている
    nlib_fd ngListFd = 0;       // NG ワードリストのファイルディスクリプタ
    size_t readSize = 0;

    errno_t result = nlib_fd_open(&ngListFd, (path + fileName + ".txt").c_str(),
        NLIB_FD_O_RDONLY);
    if (result != 0)
    {
        nlib_printf("Cannot open %s\n", (fileName + ".txt").c_str());
        return result;
    }

    nlib_offset ngListSizeTmp = 0;
    result = nlib_fd_getsize(&ngListSizeTmp, ngListFd);
    if (result != 0)
    {
        nlib_printf("Cannot get file size %s\n", (fileName + ".txt").c_str());
        return result;
    }
    assert(ngListSizeTmp > 0);
    *pOutSize = static_cast<size_t>(ngListSizeTmp);
    std::unique_ptr<char16_t[]> pPattern(new (std::nothrow) char16_t[*pOutSize / sizeof(char16_t)]);
    if (!pPattern)
    {
        return ENOMEM;
    }

    result = nlib_fd_read(&readSize, ngListFd, pPattern.get(), *pOutSize);
    if (result != 0)
    {
        nlib_printf("Cannot read %s\n", (fileName + ".txt").c_str());
        return result;
    }
    assert(readSize == *pOutSize);
    result = nlib_swapendian_16(reinterpret_cast<uint16_t*>(pPattern.get()), *pOutSize / sizeof(uint16_t));
    if (pPattern[0] != 0xFEFF)    // ファイルの先頭は BOM のはず
    {
        nlib_printf("%s is invalid format\n", (fileName + ".txt").c_str());
        return EINVAL;
    }

    result = nlib_fd_close(ngListFd);
    if (result != 0)
    {
        nlib_printf("Cannot close %s\n", (fileName + ".txt").c_str());
        return result;
    }

    pOut = std::move(pPattern);
    return 0;
}

// pWord が区切り文字を含むかどうか判定
// 含む場合は true を返す
bool CheckBoundaryMark(char* pWord, size_t size)
{
    char pBoundary[2] = { 0x5C, 0x62 };
    for (int i = 0; i + 1 < static_cast<int>(size); ++i)
    {
        if (pWord[i] == pBoundary[0] && pWord[i + 1] == pBoundary[1])
        {
            return true;
        }
    }
    return false;
}

// ACBuilder にパターンを追加
//  - pStr      登録元パターン文字列
//  - length    pStr の長さ
//  - pBuilder  登録先 AhoCorasickBuilder
//  - isCut     \b を省いて登録するかどうか
bool AddPaternToAcBuilder(char* pStr, size_t length, nn::ngc::detail::AhoCorasickBuilder* pBuilder, bool isCut)
{
    std::unique_ptr<unsigned char[]> pReg(new (std::nothrow) unsigned char[length]);
    size_t regLength = 0;           // 登録する長さ

    for (size_t i = 0; i < length; ++i)
    {
        // もしエスケープシーケンスが出るなら次も見る
        if (pStr[i] == '\\' && i + 1 < length)
        {
            if (pStr[i + 1] == 'b')
            {
                if (isCut)
                {
                    // \b をスキップする設定なら飛ばす
                    i++;
                }
                else
                {
                    pReg[regLength] = pStr[i];
                    regLength++;
                }
            }
            else if (pStr[i + 1] == '.')
            {
                // AhoCorasick では . のエスケープは必要ない
                i++;
                pReg[regLength] = pStr[i];
                regLength++;
            }
            else
            {
                // 他は未対応
                return false;
            }
        }
        else
        {
            pReg[regLength] = pStr[i];
            regLength++;
        }
    }
    pBuilder->AddPattern(pReg.get(), regLength);
    return true;
}

// 正規表現を一定の規則に従って変換する
// 具体的には .* を 取り除き、 ^|$ を \b(単語境界) にする
// その他特定の条件で \b を付加する
//  - pConvertedPattern     変換先の正規表現パターンを受け取るためのポインタ
//  - convertedLength       変換先バッファの char16_t での長さ
//  - pPattern              変換元になる正規表現パターン
//  - length                変換元になる正規表現パターンの長さ
//
// NGワードリストの正規表現は単語に対して完全一致したときに不正文字列として
// 判定出来るように最適化されている。これをこのまま長文のチェックのために利用すると
// 正しくフィルタリングさせることが出来ないため、この関数によって変換
bool ConvertRegexPatternForText(char16_t* pConvertedPattern, size_t convertedLength,
    const char16_t* pPattern, size_t length)
{
    assert(pConvertedPattern);
    assert(pPattern);

    size_t i;

    //-----------------------------------------------------
    // .*hoge.* => hoge
    //-----------------------------------------------------
    if (4 < length && length < convertedLength + 4 + 0 &&
        pPattern[0] == L'.' &&
        pPattern[1] == L'*' &&
        pPattern[length - 2] == L'.' &&
        pPattern[length - 1] == L'*')
    {
        for (i = 0; i < length - 4; i++)
        {
            pConvertedPattern[i] = pPattern[i + 2];
        }
        assert(i < convertedLength);
        pConvertedPattern[i] = L'\0';
        return true;
    }

    //-----------------------------------------------------
    // .*hoge$ => hoge\b
    //-----------------------------------------------------
    if (3 < length && length < convertedLength + 3 - 2 &&
        pPattern[0] == L'.' &&
        pPattern[1] == L'*' &&
        pPattern[length - 1] == L'$')
    {
        for (i = 0; i < length - 3; i++)
        {
            pConvertedPattern[i] = pPattern[i + 2];
        }
        assert(i + 2 < convertedLength);
        pConvertedPattern[i + 0] = L'\\';
        pConvertedPattern[i + 1] = L'b';
        pConvertedPattern[i + 2] = L'\0';
        return true;
    }

    //-----------------------------------------------------
    // ^hoge.* => \bhoge
    //-----------------------------------------------------
    if (3 < length && length < convertedLength + 3 - 2 &&
        pPattern[0] == L'^' &&
        pPattern[length - 2] == L'.' &&
        pPattern[length - 1] == L'*')
    {
        pConvertedPattern[0] = L'\\';
        pConvertedPattern[1] = L'b';
        for (i = 0; i < length - 3; i++)
        {
            pConvertedPattern[i + 2] = pPattern[i + 1];
        }
        assert(i + 2 < convertedLength);
        pConvertedPattern[i + 2] = L'\0';
        return true;
    }

    //-----------------------------------------------------
    // ^hoge$ => \bhoge\b
    //-----------------------------------------------------
    if (2 < length && length < convertedLength + 2 - 4 &&
        pPattern[0] == L'^' &&
        pPattern[length - 1] == L'$')
    {
        pConvertedPattern[0] = L'\\';
        pConvertedPattern[1] = L'b';
        for (i = 0; i < length - 2; i++)
        {
            pConvertedPattern[i + 2] = pPattern[i + 1];
        }
        assert(i + 4 < convertedLength);
        pConvertedPattern[i + 2] = L'\\';
        pConvertedPattern[i + 3] = L'b';
        pConvertedPattern[i + 4] = L'\0';
        return true;
    }

    //-----------------------------------------------------
    // ^hoge => \bhoge\b
    //-----------------------------------------------------
    if (1 < length && length < convertedLength + 1 - 4 &&
        pPattern[0] == L'^')
    {
        pConvertedPattern[0] = L'\\';
        pConvertedPattern[1] = L'b';
        for (i = 0; i < length - 1; i++)
        {
            pConvertedPattern[i + 2] = pPattern[i + 1];
        }
        assert(i + 4 < convertedLength);
        pConvertedPattern[i + 2] = L'\\';
        pConvertedPattern[i + 3] = L'b';
        pConvertedPattern[i + 4] = L'\0';
        return true;
    }

    //-----------------------------------------------------
    // hoge$ => \bhoge\b
    //-----------------------------------------------------
    if (1 < length && length < convertedLength + 1 - 4 &&
        pPattern[length - 1] == L'$')
    {
        pConvertedPattern[0] = L'\\';
        pConvertedPattern[1] = L'b';
        for (i = 0; i < length - 1; i++)
        {
            pConvertedPattern[i + 2] = pPattern[i];
        }
        assert(i + 4 < convertedLength);
        pConvertedPattern[i + 2] = L'\\';
        pConvertedPattern[i + 3] = L'b';
        pConvertedPattern[i + 4] = L'\0';
        return true;
    }

    //-----------------------------------------------------
    // hoge.* => \bhoge
    //-----------------------------------------------------
    if (2 < length && length < convertedLength + 2 - 2 &&
        pPattern[length - 2] == L'.' &&
        pPattern[length - 1] == L'*')
    {
        pConvertedPattern[0] = L'\\';
        pConvertedPattern[1] = L'b';
        for (i = 0; i < length - 2; i++)
        {
            pConvertedPattern[i + 2] = pPattern[i];
        }
        assert(i + 2 < convertedLength);
        pConvertedPattern[i + 2] = L'\0';
        return true;
    }

    //-----------------------------------------------------
    // .*hoge => hoge\b
    //-----------------------------------------------------
    if (2 < length && length < convertedLength + 2 - 2 &&
        pPattern[0] == L'.' &&
        pPattern[1] == L'*')
    {
        for (i = 0; i < length - 2; i++)
        {
            pConvertedPattern[i] = pPattern[i + 2];
        }
        assert(i + 2 < convertedLength);
        pConvertedPattern[i + 0] = L'\\';
        pConvertedPattern[i + 1] = L'b';
        pConvertedPattern[i + 2] = L'\0';
        return true;
    }

    //-----------------------------------------------------
    // hoge => \bhoge\b
    //-----------------------------------------------------
    if (length < convertedLength + 0 - 4)
    {
        pConvertedPattern[0] = L'\\';
        pConvertedPattern[1] = L'b';
        for (i = 0; i < length; i++)
        {
            pConvertedPattern[i + 2] = pPattern[i];
        }
        assert(i + 4 < convertedLength);
        pConvertedPattern[i + 2] = L'\\';
        pConvertedPattern[i + 3] = L'b';
        pConvertedPattern[i + 4] = L'\0';
        return true;
    }

    return false;
}   // NOLINT(impl/function_size)

// UTF-16 文字列を AC に登録
errno_t RegisterToAcBuilder(nn::ngc::detail::AhoCorasickBuilder* pOutbuilder, const char16_t* pConvertedPattern, nlib_fd debugFd, int option)
{
    int utf8Length = 0;
    int utf16Length = static_cast<int>(nlib_utf16len(reinterpret_cast<nlib_utf16_t*>(const_cast<char16_t*>(pConvertedPattern))));
    nn::util::GetLengthOfConvertedStringUtf16NativeToUtf8(&utf8Length,
        reinterpret_cast<uint16_t*>(const_cast<char16_t*>(pConvertedPattern)),
        utf16Length);
    std::unique_ptr<char[]> pConvertedUtf8Pattern(new (std::nothrow) char[utf8Length]);
    nn::util::ConvertStringUtf16NativeToUtf8(pConvertedUtf8Pattern.get(), utf8Length,
        reinterpret_cast<uint16_t*>(const_cast<char16_t*>(pConvertedPattern)), utf16Length);

    // ACBuilder に追加

    // \b (区切り文字) があるかどうかチェック
    if (CheckBoundaryMark(pConvertedUtf8Pattern.get(), utf8Length))
    {
        // 区切り文字がある場合
        // \b を省いたものを 1 に登録
        if (!AddPaternToAcBuilder(pConvertedUtf8Pattern.get(), utf8Length, &pOutbuilder[1], true))
        {
            nlib_printf("The file is invalid format\n");
            return EINVAL;
        }

        // \b を省かずに 2 に登録
        if (!AddPaternToAcBuilder(pConvertedUtf8Pattern.get(), utf8Length, &pOutbuilder[2], false))
        {
            nlib_printf("The file is invalid format\n");
            return EINVAL;
        }
    }
    else
    {
        // 区切り文字がない場合
        // ACBuilder には正規表現をうまく消したものを追加する
        if (!AddPaternToAcBuilder(pConvertedUtf8Pattern.get(), utf8Length, &pOutbuilder[0], true))
        {
            nlib_printf("The file is invalid format\n");
            return EINVAL;
        }
    }

    // デバッグ用としてメモ
    if (option & NgWordConverter::ExportAcBinOption_DebugOutputs)
    {
        size_t writeSize;
        nlib_fd_write(&writeSize, debugFd, pConvertedUtf8Pattern.get(), utf8Length);
        errno_t result = nlib_fd_write(&writeSize, debugFd, "\n", nlib_strnlen("\n", 8));
        if (result != 0)
        {
            nlib_printf("Cannot write to debug file\n");
            return result;
        }
    }

    return 0;
}

// NX 用 AhoCorasick のバイナリ出力
//  - acTypeSize    1 言語に対して何種類の AhoCorasick を生成するか
//  - pBuilder      生成元 AhoCorasickBuilder
//  - pAcStr        AC バイナリの生成先パス
errno_t ExportAcBinary(int acTypeSize, nn::ngc::detail::AhoCorasickBuilder* pBuilder, const std::string* pAcStr)
{
    for (int i = 0; i < acTypeSize; ++i)
    {
        nn::ngc::detail::AhoCorasick* pAc = pBuilder[i].Build();
        nn::ngc::detail::BinaryWriter writer;
        nn::ngc::detail::FileOutputStream outputStream;
        errno_t result = outputStream.Init();
        if (result != 0)
        {
            nlib_printf("Cannot initialize FileOutputStream\n");
            return result;
        }
        result = outputStream.Open(pAcStr[i].c_str());
        if (result != 0)
        {
            nlib_printf("Cannot open %s\n", pAcStr[i].c_str());
            return result;
        }
        result = writer.Init();
        if (result != 0)
        {
            nlib_printf("Cannot initialize BinaryWriter\n");
            return result;
        }
        result = writer.Open(&outputStream);
        if (result != 0)
        {
            nlib_printf("Cannot open FileOutputStream\n");
            return result;
        }
        if (!pAc->Export(&writer))
        {
            nlib_printf("AhoCorasick export error\n");
            return EIO;
        }
        if (!writer.Close())
        {
            nlib_printf("Cannot close BinaryWriter\n");
            return EIO;
        }
        if (!outputStream.Close())
        {
            nlib_printf("Cannot close FileOutputStream\n");
            return EIO;
        }
    }
    return 0;
}

}   // unnamed namespace

// HorizonOS 用の AhoCorasick バイナリを作成
errno_t NgWordConverter::CreatePlainAcBinary(std::string fileName, const std::string* pAcStr)
{
    errno_t result = 0;
    nn::ngc::detail::AhoCorasickBuilder pBuilder[m_AcTypeSize];

    for (auto& builder : pBuilder)
    {
        if (!builder.Init())
        {
            nlib_printf("AhoCorasickBuilder couldn't initialize\n");
            return -1;
        }
    }

    // デバッグ用中間ファイルの生成
    nlib_fd debugFd = 0;
    if (m_Option & ExportAcBinOption_DebugOutputs)
    {
        // ファイルパス
        const std::string debugPath = m_NgResourceOutputPath + "ac_" + fileName + "_for_debug.txt";

        // 既にあるファイルは削除
        result = nlib_remove(debugPath.c_str());
        if (result != 0 && result != ENOENT)
        {
            nlib_printf("Cannot remove %s\n", debugPath.c_str());
            return result;
        }

        result = nlib_fd_creat(&debugFd, debugPath.c_str(), 0);
        if (result != 0)
        {
            nlib_printf("Cannot create %s\n", debugPath.c_str());
            return result;
        }
    }

    std::unique_ptr<char16_t[]> pPattern; // 読み込んだ NG ワードリスト
    size_t ngListSize = 0;
    result = ReadNgWord(pPattern, &ngListSize, m_NgResourceInputPath, fileName);
    if (result != 0)
    {
        return result;
    }

    // 読み込んだ NG ワードリストの解析
    size_t listBufIndex = 1;   // 現在解析しているポジション(0はBOMなので1から)
    size_t wordStartIndex = 1; // 現在解析中のNGワードが開始されるインデックス位置

    // 正規表現を長文用に変換し、 UTF-8 にして AC に登録する関数オブジェクト
    auto ConvertAndRegister = [&]() -> errno_t
    {
        size_t convertedTextLength = listBufIndex - 1 - wordStartIndex + 4;     // 変換後に増える文字数は最大でも 4
        std::unique_ptr<char16_t[]> pConvertedPattern(new (std::nothrow) char16_t[convertedTextLength]);
        if (!pConvertedPattern)
        {
            return ENOMEM;
        }
        memset(pConvertedPattern.get(), 0, sizeof(char16_t) * convertedTextLength);
        if (ConvertRegexPatternForText(pConvertedPattern.get(), convertedTextLength,
            &pPattern[wordStartIndex],
            listBufIndex - 1 - wordStartIndex))
        {
            // 変換された文字列を UTF-8 にして AC に登録
            result = RegisterToAcBuilder(pBuilder, pConvertedPattern.get(), debugFd, m_Option);
            if (result != 0)
            {
                return result;
            }
        }
        return 0;
    };

    for (; listBufIndex * sizeof(char16_t) < ngListSize; listBufIndex++)
    {
        // nullptrが現れたら誤ってバイナリファイルを読み込んでいる可能性がある
        if (pPattern[listBufIndex] == L'\0')
        {
            nlib_printf("%s is invalid format\n", (fileName + ".txt").c_str());
            return EINVAL;
        }

        // 改行コードであるかどうかの判定
        if (pPattern[listBufIndex] == L'\n')
        {
            listBufIndex++;
        }
        else
        {
            continue;
        }

        // 単語用に作成された正規表現を、長文用に変換
        result = ConvertAndRegister();
        if (result != 0)
        {
            return result;
        }

        // 次の文字のためにセット
        wordStartIndex = listBufIndex;
    }

    // 最後が空行でないことへの対処
    if (listBufIndex - 1 > wordStartIndex)
    {
        result = ConvertAndRegister();
        if (result != 0)
        {
            return result;
        }
    }

    if (m_Option & ExportAcBinOption_DebugOutputs)
    {
        result = nlib_fd_flush(debugFd);
        if (result != 0)
        {
            nlib_printf("Cannot flush %s\n", ("ac_" + fileName + "_for_debug.txt").c_str());
            return result;
        }
        result = nlib_fd_close(debugFd);
        if (result != 0)
        {
            nlib_printf("Cannot close %s\n", ("ac_" + fileName + "_for_debug.txt").c_str());
            return result;
        }
    }

    // NX 用 AhoCorasick のバイナリ出力
    result = ExportAcBinary(m_AcTypeSize, pBuilder, pAcStr);
    if (result != 0)
    {
        return result;
    }

    return 0;
}   // NOLINT(impl/function_size)
