﻿using System.Collections.Generic;
using System.Text;
using System.IO;

namespace IFFRefGen.Src
{
    // 警告チェッカー
    public class WarningChecker
    {
        // 警告文字列取得
        public static string CheckWarning()
        {
            WarningChecker checker = new WarningChecker();
            checker.AllCheck();
            return checker.ToString();
        }

        // コンストラクタ
        public WarningChecker()
        {
            this.TabString = "  ";
        }

        // 全チェック
        public void AllCheck()
        {
            // 共通の xsd に含まれている simpleType は使う必要はないので、警告として扱わないでおく
            this.HasError |= CheckUnusedElement();
            this.HasError |= CheckUnusedComplexType();
            this.HasError |= CheckUnusedAttributeGroup();
            this.HasError |= CheckUnusedDocElement();
        }

        // 文字列化
        public override string ToString()
        {
            StringBuilder builder = new StringBuilder();

            GetUnusedElementString(builder);
            GetUnusedComplexTypeString(builder);
            GetUnusedSimpleTypeString(builder);
            GetUnusedAttributeGroupString(builder);
            GetUnusedDocElementString(builder);
            GetUnusedDocAttributeString(builder);

            return builder.ToString();
        }

        // エラーがあるか
        public bool HasError { get; private set; }
        // タブ文字
        public string TabString { get; set; }

        private int TabDepth { get; set; }
        private void AppendTab(StringBuilder builder)
        {
            for (int i = 0; i < this.TabDepth; i++) { builder.Append(this.TabString); }
        }

        #region 未使用のスキーマ要素
        // 未使用の element
        public bool CheckUnusedElement()
        {
            bool hasError = false;
            foreach (XsdElement element in GXsd.Elements)
            {
                if (!element.IsUsed)
                {
                    this.UnusedElement.Add(element);
                    hasError = true;
                }
            }
            return hasError;
        }

        public void GetUnusedElementString(StringBuilder builder)
        {
            GetUnusedItemString(builder, this.UnusedElement,
                "未使用の element があります\n");
        }

        public readonly List<XsdItem> UnusedElement = new List<XsdItem>();

        // 未使用の complexType
        public bool CheckUnusedComplexType()
        {
            bool hasError = false;
            foreach (XsdComplexType complexType in GXsd.ComplexTypes)
            {
                if (!complexType.IsUsed)
                {
                    this.UnusedComplexType.Add(complexType);
                    hasError = true;
                }
            }
            return hasError;
        }

        public void GetUnusedComplexTypeString(StringBuilder builder)
        {
            GetUnusedItemString(builder, this.UnusedComplexType,
                "未使用の complexType があります\n");
        }

        public readonly List<XsdItem> UnusedComplexType = new List<XsdItem>();

        // 未使用の simpleType
        public bool CheckUnusedSimpleType()
        {
            bool hasError = false;
            foreach (XsdSimpleType simpleType in GXsd.SimpleTypes)
            {
                if (!simpleType.IsUsed)
                {
                    this.UnusedSimpleType.Add(simpleType);
                    hasError = true;
                }
            }
            return hasError;
        }

        public void GetUnusedSimpleTypeString(StringBuilder builder)
        {
            GetUnusedItemString(builder, this.UnusedSimpleType,
                "未使用の simpleType があります\n");
        }

        public readonly List<XsdItem> UnusedSimpleType = new List<XsdItem>();

        // 未使用の attributeGroup
        public bool CheckUnusedAttributeGroup()
        {
            bool hasError = false;
            foreach (XsdAttributeGroup attributeGroup in GXsd.AttributeGroups)
            {
                if (!attributeGroup.IsUsed)
                {
                    this.UnusedAttributeGroup.Add(attributeGroup);
                    hasError = true;
                }
            }
            return hasError;
        }

        public void GetUnusedAttributeGroupString(StringBuilder builder)
        {
            GetUnusedItemString(builder, this.UnusedAttributeGroup,
                "未使用の attributeGroup があります\n");
        }

        public readonly List<XsdItem> UnusedAttributeGroup = new List<XsdItem>();

        public void GetUnusedItemString(StringBuilder builder,
            List<XsdItem> unusedItems, string message)
        {
            if (unusedItems.Count != 0)
            {
                builder.Append(message);
                this.TabDepth++;
                foreach (XsdItem item in unusedItems)
                {
                    AppendTab(builder);
                    builder.AppendFormat(
                        "{0} ({1})\n",
                        item.Name,
                        item.FileName);
                }
                this.TabDepth--;
                builder.Append("\n");
            }
        }
        #endregion

        #region 未使用のドキュメント
        // 未使用の DocElement と DocAttribute
        public bool CheckUnusedDocElement()
        {
            bool hasError = false;
            foreach (DocElement element in GDoc.Elements.Values)
            {
                if (!element.IsUsed)
                {
                    this.UnusedDocElement.Add(element);
                    hasError = true;
                }
                if (element.Attributes == null) { continue; }
                foreach (DocAttribute attribute in element.Attributes)
                {
                    if (!attribute.IsUsed)
                    {
                        this.UnusedDocAttribute.Add(attribute);
                        hasError = true;
                    }
                }
            }
            return hasError;
        }

        public void GetUnusedDocElementString(StringBuilder builder)
        {
            if (this.UnusedDocElement.Count == 0) { return; }
            builder.Append("未使用の要素ドキュメントがあります\n");
            this.TabDepth++;
            foreach (DocElement element in this.UnusedDocElement)
            {
                AppendTab(builder);
                builder.AppendFormat(
                    "{0} ({1})\n",
                    element.Name,
                    element.FileName);
            }
            this.TabDepth--;
            builder.Append("\n");
        }

        public readonly List<DocElement> UnusedDocElement = new List<DocElement>();

        public void GetUnusedDocAttributeString(StringBuilder builder)
        {
            if (this.UnusedDocAttribute.Count == 0) { return; }
            builder.Append("未使用の属性ドキュメントがあります\n");
            this.TabDepth++;
            foreach (DocAttribute attribute in this.UnusedDocAttribute)
            {
                AppendTab(builder);
                builder.AppendFormat(
                    "{0} ({1})\n",
                    attribute.Name,
                    attribute.FileName);
            }
            this.TabDepth--;
            builder.Append("\n");
        }

        public readonly List<DocAttribute> UnusedDocAttribute = new List<DocAttribute>();
        #endregion
    }
}
