﻿using System;
using System.Diagnostics;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.IO;
using System.Reflection;
using Nintendo.Foundation.IO;

namespace IFFRefGen.Src
{
    // プログラム
    public static class Program
    {
        class CommandLineParameters
        {
            [CommandLineOption("settings-file", Description = "設定ファイルのパスを指定します。", IsRequired = true, ValueName = "ファイルパス")]
            public string SettingFilePath { get; set; }

            [CommandLineOption('o', "output", Description = "出力フォルダのパスを指定します。このオプションを指定した場合、設定ファイルの出力フォルダは無視されます。", ValueName = "フォルダパス")]
            public string OutputFolderPath { get; set; }
        }

        enum ReturnCode
        {
            Success = 0,
            InvalidOption,
            UndefinedError = 255,
        }

        // エントリポイント
        private static int Main(string[] args)
        {
#if !DEBUG
            try
#endif
            {
                CommandLineParameters parameters;
                try
                {
                    var perser = new CommandLineParser();
                    if (!(perser.ParseArgs(args, out parameters)))
                    {
                        // ヘルプ表示、バージョン表示
                        return (int)ReturnCode.Success;
                    }
                }
                catch
                {
                    // 不正なオプション指定
                    return (int)ReturnCode.InvalidOption;
                }

                GApp.ConfigPath = parameters.SettingFilePath;

                // コンフィグの読み込み
                GApp.Config = Config.Read(GApp.ConfigPath);

                if (!string.IsNullOrEmpty(parameters.OutputFolderPath))
                {
                    // 出力フォルダを引数指定のもので上書き
                    if (System.IO.Path.IsPathRooted(parameters.OutputFolderPath))
                    {
                        GApp.Config.HtmlOutputConfig.OutputFolder = parameters.OutputFolderPath;
                    }
                    else
                    {
                        // 相対パスの場合は設定ファイルからの相対パスにする
                        string rootFolder = System.IO.Path.GetDirectoryName(GApp.ConfigPath);
                        GApp.Config.HtmlOutputConfig.OutputFolder = System.IO.Path.Combine(rootFolder, parameters.OutputFolderPath);
                    }
                }

                // スキーマの初期化
                GXsd.Initialize();

                // ドキュメントの初期化
                GDoc.Initialize();

                // サンプルの初期化
                GSmpl.Initialize();

                // HTML 出力の初期化
                GHtml.Initialize();

                // HTML 出力の生成
                GHtml.Generate();

                // サンプルの生成
                GSmpl.Generate();

                // 警告チェック
                string warning = WarningChecker.CheckWarning();
                if (warning != string.Empty)
                {
                    Debug.WriteLine(warning);
                    Console.WriteLine(warning);
                }

#if DEBUG
                // 属性範囲の書き出し
                WriteAttributeRange();
#endif
            }
#if !DEBUG
            catch (Exception exp)
            {
                // 共通例外処理
                StringBuilder errorDetail = new StringBuilder();
                while (exp != null)
                {
                    Console.WriteLine(exp.Message);
                    Console.WriteLine(exp.StackTrace);
                    errorDetail.AppendLine(exp.Message);
                    errorDetail.AppendLine(exp.StackTrace);
                    exp = exp.InnerException;
                }

                {
                    string localAppDataPath = System.Environment.GetFolderPath(Environment.SpecialFolder.LocalApplicationData);
                    if (System.IO.Directory.Exists(localAppDataPath))
                    {
                        string assemblyName = Assembly.GetEntryAssembly().GetName().Name;
                        string errorLogDir = System.IO.Path.Combine(localAppDataPath, string.Format("Nintendo\\{0}\\", assemblyName));
                        if (!System.IO.Directory.Exists(errorLogDir))
                        {
                            System.IO.Directory.CreateDirectory(errorLogDir);
                        }

                        string errorLogPath = System.IO.Path.Combine(errorLogDir, "error.log");
                        Encoding encoding = Encoding.GetEncoding("utf-8");
                        StreamWriter writer =
                          new StreamWriter(errorLogPath, true, encoding);
                        writer.Write(errorDetail.ToString());
                        writer.Close();
                    }
                }

                return (int)ReturnCode.UndefinedError;
            }
#endif

            return (int)ReturnCode.Success;
        }

        private static void WriteAttributeRange()
        {
            XsdElement[] elements = GXsd.Elements.ToArray<XsdElement>();

            StringBuilder builder = new StringBuilder();
            builder.AppendLine("要素,属性,型,最小値,最大値");

            foreach (XsdElement element in elements)
            {
                XsdComplexType complexType = element.Type as XsdComplexType;
                if (complexType == null) { continue; }
                foreach (XsdAttribute attribute in complexType.Attributes)
                {
                    XsdSimpleType type = attribute.Type;
                    if (type.IsEnumeration) { continue; }
                    if (type.PrimitiveType.Name == "xsd:boolean") { continue; }
                    if (type.PrimitiveType.Name == "xsd:string") { continue; }
                    builder.AppendFormat("{0},{1},{2},{3},{4},{5},{6}",
                        element.Name, attribute.Name, type.PrimitiveType.Name.Substring(4),
                        type.MinInclusive, type.MaxInclusive,
                        type.MinExclusive, type.MaxExclusive);
                    builder.AppendLine();
                }
            }
            File.WriteAllText("AttributeRange.csv", builder.ToString(), Encoding.UTF8);
        }
    }
}
