﻿using System.Xml;
using System;
using System.Text;
using System.Diagnostics;
using System.IO;
using System.Collections.Generic;

namespace IFFRefGen.Src
{
    // HTML element
    public class HtmlElement
    {
        // テンプレートファイルの名前
        private readonly string TemplateFileName = "TmplElement.html";
        // 保存パスのフォーマット文字列
        private readonly string SavePathFormat = "html/reference/elements/{0}.html";

        // ヘッダのタイトル XPath
        private readonly string HeadTitleXPath = "/html/head/title";
        // 本文のタイトル XPath
        private readonly string BodyTitleXPath = "/html/body/h1[@id=\"title\"]";
        // タイトルのフォーマット
        private readonly string TitleFormat = "<{0}>";

        // 概要の XPath
        private readonly string SummaryXPath = "//p[@id='summary']";
        // 詳細説明の XPath
        private readonly string DetailXPath = "//p[@id='detail']";
        // DCC 出力無効説明文
        private readonly string DccExportDisableInfo =
            "この要素は DCC ツールから出力しません。";

        // 親要素の XPath
        private readonly string ParentXPath = "//ul[@id='parent']";
        private readonly string ElementIsRoot = "ルート要素です。";

        // サンプルファイル名の XPath
        private readonly string SampleInfoXPath = "//p[@id='sample_info']";
        // サンプルファイル名のフォーマット文字列
        private readonly string SampleInfoFormat = "../../sample/{0}.html";
        // プリサンプルの XPath
        private readonly string PreSampleXPath = "//pre[@class='comment']";
        // サンプルの XPath
        private readonly string SampleXPath = "//pre[@class='code']";

        // コンストラクタ
        public HtmlElement(XsdElement element)
        {
            this.Element = element;
            if (GDoc.Elements.ContainsKey(this.Element.Name))
            {
                this.Document = GDoc.Elements[this.Element.Name];
                this.Document.IsUsed = true;
            }
            this.Template = new XmlDocument();

            this.Attribute = new HtmlElementAttribute(this.Element, this.Document, this.Template);
            this.Content = new HtmlElementContent(this.Element, this.Document, this.Template);
        }

        // 初期化
        public void Initialize()
        {
            // テンプレートの読み込み
            string loadPath = GApp.Config.HtmlOutputConfig.TemplateElementFolder + this.TemplateFileName;
            this.Template.Load(loadPath);
        }

        // 生成
        public void Generate()
        {
            // タイトルの生成
            GenerateTitle();

            // 説明の生成
            GenerateNote();

            // 親要素の生成
            GenerateParentElement();

            // 属性の生成
            this.Attribute.Generate();

            // 内容の生成
            this.Content.Generate();

            // サンプルの生成
            GenerateSample();

            // 保存
            string savePath = System.IO.Path.Combine(GApp.Config.HtmlOutputConfig.OutputFolder, string.Format(this.SavePathFormat, this.Element.Name));

            GHtml.Save(this.Template, savePath);
        }

        // タイトルの生成
        private void GenerateTitle()
        {
            string title = string.Format(TitleFormat, this.Element.Name);
            XmlNode headTitle = this.Template.SelectSingleNode(this.HeadTitleXPath);
            headTitle.InnerText = title;

            XmlNode bodyTitle = this.Template.SelectSingleNode(this.BodyTitleXPath);
            bodyTitle.InnerText = title;
        }

        // 説明の生成
        private void GenerateNote()
        {
            if (this.Document == null) { return; }

            // 概要
            if (this.Document.Summary != null)
            {
                XmlNode summary = this.Template.SelectSingleNode(this.SummaryXPath);
                summary.InnerXml = GHtml.LinkElement(this.Document.Summary);
            }

            // 詳細説明
            if (this.Document.Detail != null)
            {
                XmlNode detail = this.Template.SelectSingleNode(this.DetailXPath);
                detail.InnerXml = GHtml.LinkElement(this.Document.Detail);
            }

            if (this.Document.DccExportDisable)
            {
                XmlNode detail = this.Template.SelectSingleNode(this.DetailXPath);
                if (detail.InnerXml != string.Empty) { detail.InnerXml += "<br />"; }
                detail.InnerXml += DccExportDisableInfo;
            }
        }

        // 親要素の生成
        private void GenerateParentElement()
        {
            XmlNode parents = this.Template.SelectSingleNode(this.ParentXPath);
            List<XsdElement> referenceElements = GXsd.FindReferenceElements(this.Element);
            if (referenceElements.Count == 0)
            {
                XmlElement li = this.Template.CreateElement("li");
                li.InnerText = this.ElementIsRoot;
                parents.AppendChild(li);
            }
            else
            {
                foreach (XsdElement parent in referenceElements)
                {
                    XmlElement li = this.Template.CreateElement("li");
                    parents.AppendChild(li);
                    string innerXml = "&lt;" + parent.Name + "&gt;";
                    if (GDoc.Elements.ContainsKey(parent.Name))
                    {
                        string summary = GDoc.Elements[parent.Name].Summary;
                        if (summary != null) { innerXml += " : " + summary; }
                    }
                    li.InnerXml = GHtml.LinkElement(innerXml);
                }
            }
        }

        // サンプルの生成
        private void GenerateSample()
        {
            if (this.Document == null) { return; }

            DocSample docSample = this.Document.Sample;
            if ((docSample == null) || (docSample.FileName == null)) { return; }

            SmplData sampleData = GSmpl.FindSample(docSample.FileName);
            if (sampleData == null)
            {
                throw new Exception(
                    "サンプルファイルが見つかりません。" + docSample.FileName);
            }

            XmlNode sampleInfoNode =
                this.Template.SelectSingleNode(this.SampleInfoXPath);
            XmlElement a = this.Template.CreateElement("a");
            a.InnerText = sampleData.FileName;
            XmlAttribute href = this.Template.CreateAttribute("href");
            href.InnerText = string.Format(this.SampleInfoFormat, sampleData.FileName);
            a.Attributes.Append(href);
            sampleInfoNode.AppendChild(a);

            XmlNode preSampleNode = this.Template.SelectSingleNode(this.PreSampleXPath);
            XmlNode sampleNode = this.Template.SelectSingleNode(this.SampleXPath);
            string[] sampleStr = sampleData.GetElementSample(
                this.Element.Name, docSample.LineNumber);
            preSampleNode.InnerText = sampleStr[0];
            sampleNode.InnerText = sampleStr[1];
        }

        // 要素
        public XsdElement Element { get; private set; }
        // 要素ドキュメント
        public DocElement Document { get; private set; }
        // テンプレート
        public XmlDocument Template { get; private set; }

        // 要素の属性説明
        public HtmlElementAttribute Attribute { get; private set; }
        // 要素の内容説明
        public HtmlElementContent Content { get; private set; }
    }
}
