﻿/*--------------------------------------------------------------------------------*
  Copyright (C)2014-2015 Nintendo. All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

using CMDLINE = Common.UtilCmdline;
using LOG = Common.Logger;

namespace HtmlToVsHelp
{
    public class CmdlineOption
    {
        public const string UsageString =
@"Usage: HtmlToVsHelp [options] doxygen output directory base path
    --in-dir-name dirname    set input directory name (default is Build_VsHelp)
    --out-dir-name dirname   set output directory name (default is Out)
    --title-name titleName   set title name (default is 'NintendoSDK API Reference')
    --document-name docName  set document name (default is 'index')
    --check-parent-page      check if content parent page is found
    --ja-jp                  make Japanese Visual Studio Help file
    --en-us                  make English Visual Studio Help file
    --prepare-cabinet        make help file for cabinet
    --debug                  enable debug level output
    -h                       show this help
";

        public enum LanguageModeType
        {
            LANGUAGE_JA_JP,
            LANGUAGE_EN_US
        }

        // プロパティ
        private static string __doxygenPath = null;
        private static string __inputDirectoryName = "Build_VsHelp";
        private static string __outputDirectoryName = "Out";
        private static string __titleName = null;
        private static string __documentName = null;
        private static bool __checkParentPage = false;
        private static LanguageModeType __countryMode = LanguageModeType.LANGUAGE_JA_JP;
        private static bool __flagPrepareCabinet = false;
        private static bool __flagPrintDebug = false;
        public static string DoxygenPath
        {
            get { return __doxygenPath; }
        }
        public static string InputDirectoryName
        {
            get { return __inputDirectoryName; }
        }
        public static string OutputDirectoryName
        {
            get { return __outputDirectoryName; }
        }
        public static string TitleName
        {
            get { return __titleName; }
        }
        public static string DocumentName
        {
            get { return __documentName; }
        }
        public static bool CheckParentPage
        {
            get { return __checkParentPage; }
        }
        public static LanguageModeType CountryMode
        {
            get { return __countryMode; }
        }
        public static bool FlagPrepareCabinet
        {
            get { return __flagPrepareCabinet; }
        }
        public static bool FlagPrintDebug
        {
            get { return __flagPrintDebug; }
        }

        private enum ArgumentParseMode
        {
            PARSE_INPUT_DIRECTORY,
            PARSE_OUTPUT_DIRECTORY,
            PARSE_TITLE_NAME,
            PARSE_DOCUMENT_NAME,
            PARSE_NONE
        }

        // メソッド

        public static bool Parse(string[] args)
        {
            ArgumentParseMode mode = ArgumentParseMode.PARSE_NONE;

            foreach (string arg in args)
            {
                if (mode == ArgumentParseMode.PARSE_INPUT_DIRECTORY)
                {
                    if (!CMDLINE.CheckInputDirArgument(arg))
                    {
                        return false;
                    }
                    __inputDirectoryName = arg;
                    mode = ArgumentParseMode.PARSE_NONE;
                }
                else if (mode == ArgumentParseMode.PARSE_OUTPUT_DIRECTORY)
                {
                    if (!CMDLINE.CheckOutputDirArgument(arg))
                    {
                        return false;
                    }
                    __outputDirectoryName = arg;
                    mode = ArgumentParseMode.PARSE_NONE;
                }
                else if (mode == ArgumentParseMode.PARSE_TITLE_NAME)
                {
                    __titleName = arg.Trim('"', '\'');
                    mode = ArgumentParseMode.PARSE_NONE;
                }
                else if (mode == ArgumentParseMode.PARSE_DOCUMENT_NAME)
                {
                    __documentName = arg;
                    mode = ArgumentParseMode.PARSE_NONE;
                }
                else if (arg == "--in-dir-name")
                {
                    mode = ArgumentParseMode.PARSE_INPUT_DIRECTORY;
                }
                else if (arg == "--out-dir-name")
                {
                    mode = ArgumentParseMode.PARSE_OUTPUT_DIRECTORY;
                }
                else if (arg == "--title-name")
                {
                    mode = ArgumentParseMode.PARSE_TITLE_NAME;
                }
                else if (arg == "--document-name")
                {
                    mode = ArgumentParseMode.PARSE_DOCUMENT_NAME;
                }
                else if (arg == "--check-parent-page")
                {
                    __checkParentPage = true;
                }
                else if (arg == "--ja-jp")
                {
                    __countryMode = LanguageModeType.LANGUAGE_JA_JP;
                }
                else if (arg == "--en-us")
                {
                    __countryMode = LanguageModeType.LANGUAGE_EN_US;
                }
                else if (arg == "--prepare-cabinet")
                {
                    __flagPrepareCabinet = true;
                }
                else if (arg == "--degug")
                {
                    __flagPrintDebug = true;
                }
                else if (arg == "-h")
                {
                    Usage();
                    return false;
                }
                else
                {
                    __doxygenPath = Common.UtilFs.AdjustDirectoryPath(arg);
                }
            }
            return true;
        }

        public static void Usage()
        {
            LOG.Log(LOG.Level.LOG_ERROR, CmdlineOption.UsageString);
        }
    }
}
