﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Reflection;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;
using System.Threading.Tasks;

namespace CommitContentChecker
{
    // TORIAEZU: 現段階で保持したいパラメータは無いため static に
    internal static class FilePathChecker
    {
        // 3de1c307b0cf76bdfc05a0d1e8acf7290963a9db における最大パス長を追認
        internal static readonly int PathLengthMax = 196;

        // stage されたファイルパスをチェックし、以下に示す問題があれば false を返します
        // - ファイルパスが長すぎる
        // - ファイルシステム上のパスと異なる
        internal static bool CheckAll(string filePath)
        {
            return CheckLength(filePath) && CheckCasing(filePath);
        }

        internal static bool CheckLength(string filePath)
        {
            // パス長チェック
            if (filePath.Length > PathLengthMax)
            {
                Console.Error.WriteLine(
                    filePath + ": Error: Path too long. The path contains {0} chars. (Current limit: {1} chars.)",
                    filePath.Length,
                    PathLengthMax);
                return false;
            }

            return true;
        }

        internal static bool CheckCasing(string filePath)
        {
            // ファイルシステム上のファイルパスをログ出力するために最後までチェックを行う
            bool result = true;
            string actualFilePath = string.Empty;

            var dinfo = new DirectoryInfo(Path.GetDirectoryName(Path.GetFullPath(filePath)));
            var targetPath = filePath;
            while (dinfo != null && !string.IsNullOrEmpty(targetPath))
            {
                try
                {
                    var targetEntryName = Path.GetFileName(targetPath);
                    var fsinfo = dinfo.GetFileSystemInfos(targetEntryName);
                    if (fsinfo.Any())
                    {
                        var actualEntryName = fsinfo.Single().Name;
                        result &= string.Equals(actualEntryName, targetEntryName);
                        actualFilePath = Path.Combine(actualEntryName, actualFilePath);
                    }
                    else
                    {
                        // ファイルが見つからない場合は無視して親ディレクトリに進む
                    }
                }
                catch (DirectoryNotFoundException)
                {
                    // ディレクトリが見つからない場合は無視して親ディレクトリに進む
                }

                dinfo = dinfo.Parent;
                targetPath = Path.GetDirectoryName(targetPath);
            }

            if (!result)
            {
                Console.Error.WriteLine(filePath + ": Error: Staged file path does not match the actual file path on file system");
                Console.Error.WriteLine("Actual file path: " + actualFilePath);
            }
            return result;
        }
    }
}
