﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace CodingCheckerUtil
{
    public class FormatResult
    {
        [Flags]
        public enum FormatResultFlags
        {
            None = 0,
            [Message("Line break code is replaced by CRLF")]
            WrongLineBreak = 1 << 0,
            [Message("Tab was replaced by space")]
            ExistsTab = 1 << 1,
            [Message("Space at end of line was removed")]
            ExistsFootSpace = 1 << 2,
            [Message("Line break was Inserted at end of file")]
            NoExistsFootLineBreak = 1 << 3,
        }

        private class MessageAttribute : Attribute
        {
            public string Message { get; }

            public MessageAttribute(string message)
            {
                this.Message = message;
            }
        }

        public FormatResultFlags Result { set; get; }
        public bool IsFormatted => Result != FormatResultFlags.None;
        public string[] Messages
        {
            get
            {
                string[] resultStrings = Result.ToString().Split(',').Select(x => x.Trim()).ToArray();
                var messageAttributes = resultStrings.SelectMany(x => Result.GetType().GetField(x).GetCustomAttributes(typeof(MessageAttribute), false).Cast<MessageAttribute>());
                return messageAttributes.Select(x => x.Message).ToArray();
            }
        }

        public FormatResult()
        {
            this.Result = FormatResultFlags.None;
        }
    }

    static public class ContentFormatter
    {
        /// <summary>
        /// 与えられた文字列に対してコーディング規約に基づいた自動整形を行います。
        /// 自動整形が行われた場合に true を返します。
        /// </summary>
        /// <param name="content"></param>
        /// <returns></returns>
        static public FormatResult Format(ref string content, bool outputWarning = true, string warningHeader = "")
        {
            var result = new FormatResult();

            // 改行コード間違いを検出
            result.Result |= FormatterUtil.CheckLineBreak(content) ? FormatResult.FormatResultFlags.WrongLineBreak : FormatResult.FormatResultFlags.None;

            // 改行コードを一旦LineFeedに変換
            FormatterUtil.ConvertLineBreakToLf(ref content);

            result.Result |= FormatterUtil.FormatTabToSpace(ref content) ? FormatResult.FormatResultFlags.ExistsTab : FormatResult.FormatResultFlags.None;
            result.Result |= FormatterUtil.FormatFootSpace(ref content) ? FormatResult.FormatResultFlags.ExistsFootSpace : FormatResult.FormatResultFlags.None;
            result.Result |= FormatterUtil.FormatFootLineBreak(ref content) ? FormatResult.FormatResultFlags.NoExistsFootLineBreak : FormatResult.FormatResultFlags.None;

            // 改行をLFからCRLFに変換
            FormatterUtil.ConvertLfToCrlf(ref content);

            // 警告の出力
            if (outputWarning)
            {
                foreach (var m in result.Messages)
                {
                    Console.Out.WriteLine(warningHeader + "Warning: " + m);
                }
            }

            return result;
        }
    }
}
