﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using CodingCheckerUtil;
using Nintendo.ToolFoundation.IO;
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace CodeFormatter
{
    internal class Program
    {
        internal static void Main(string[] args)
        {
            // コマンドライン引数を解析
            var parseResult = ParseCommandLine(args);
            if (parseResult.Params == null)
            {
                Environment.Exit(parseResult.ExitCode);
                return;
            }

            var @params = parseResult.Params;

            string codeContent = string.Empty;
            UTF8Encoding encoding;
            // BOM チェック
            byte[] codeBytes;
            try
            {
                codeBytes = File.ReadAllBytes(@params.Path);
            }
            catch (Exception e) when (e is IOException || e is UnauthorizedAccessException)
            {
                Console.Error.WriteLine(@params.Path + ": Error: Failed to read file");
                Console.Error.WriteLine(e.Message);
                Environment.Exit(1);
                return;
            }
            encoding = new UTF8Encoding(FormatterUtil.CheckBOM(codeBytes, false));
            // BOM 付きの場合は BOM のバイト列は無視してデコード
            codeContent = encoding.GetString(codeBytes, encoding.GetPreamble().Length, codeBytes.Length - encoding.GetPreamble().Length);

            // 自動整形
            ContentFormatter.Format(ref codeContent, false);

            // 整形結果を出力
            var outputPath = @params.OutputPath;
            if (!string.IsNullOrEmpty(outputPath))
            {
                try
                {
                    using (var sw = new StreamWriter(outputPath, false, encoding))
                    {
                        sw.Write(codeContent);
                    }
                }
                catch (Exception e) when (e is IOException || e is UnauthorizedAccessException)
                {
                    Console.Error.WriteLine(outputPath + ": Error: Failed to write file");
                    Console.Error.WriteLine(e.Message);
                    Environment.Exit(1);
                    return;
                }
            }
            else
            {
                // 出力先についてオプション指定が無ければ標準出力へ
                Console.Out.Write(codeContent);
            }

            Environment.Exit(0);
        }

        private static ParseCommandLineResult ParseCommandLine(string[] args)
        {
            var s = new CommandLineParserSettings()
            {
                ApplicationDescription = "Format the specified file according to the Siglo coding conventions.",
            };

            try
            {
                CodeFormatterParams @params;
                if (!new CommandLineParser(s).ParseArgs(args, out @params))
                {
                    return new ParseCommandLineResult(0);
                }
                return new ParseCommandLineResult(@params);
            }
            catch (ArgumentException)
            {
                return new ParseCommandLineResult(1);
            }
        }

        private class ParseCommandLineResult
        {
            public CodeFormatterParams Params { get; }
            public int ExitCode { get; }

            public ParseCommandLineResult(CodeFormatterParams @params)
            {
                this.Params = @params;
            }

            public ParseCommandLineResult(int exitCode)
            {
                this.Params = null;
                this.ExitCode = exitCode;
            }
        }
    }
}
