﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Globalization;
using System.IO;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;
using System.Threading;
using System.Threading.Tasks;
using System.Xml.XPath;
using System.Xml.Linq;
using Nintendo.Foundation.IO;

namespace ApiLinkRewriter
{
    class Program
    {
        /**
         * <summary> Main メソッド </summary>
         */
        static void Main(string[] args)
        {
            try
            {
                CommandLineOptions parsed;
                var parser = new CommandLineParser();

                if (false == parser.ParseArgs<CommandLineOptions>(args, out parsed))
                {
                    System.Environment.Exit(1);
                }
                System.Environment.Exit(Run(parsed));
            }
            catch (Exception exception)
            {
                Console.Error.WriteLine("[ERROR] {0}", exception.Message);
                foreach (TraceListener lisners in Trace.Listeners)
                {
                    lisners.WriteLine(string.Format("StackTrace: {0}", exception.StackTrace));
                }
                System.Environment.Exit(1);
            }
        }

        /**
         * <summary> リンクタグ情報を読み出し html ファイルにリンクを付加 </summary>
         * <param name = "parsed">コマンドラインオプションで指定した引数</param>
         */
        static int Run(CommandLineOptions parsed)
        {
            // 表示レベル
            var logLevel = OutputLog.LogLevel.Default;
            if (parsed.Detail && parsed.Quiet)
            {
                OutputLog.WriteError("Cannot --detail and --quiet options are specified at a time.");
                throw new ArgumentException("Cannot --detail and --quiet options are specified at a time.");
            }
            if (parsed.Detail) { logLevel = OutputLog.LogLevel.Detail; }
            if (parsed.Quiet) { logLevel = OutputLog.LogLevel.Quiet; }

            // フォルダかファイル指定
            if (parsed.HtmlDir != null && parsed.InputFiles != null)
            {
                OutputLog.WriteError(string.Format("--html-dir: {0}", parsed.HtmlDir));
                OutputLog.WriteError(string.Format("--input-file: {0}", parsed.InputFiles));
                OutputLog.WriteError("Both of --html-dir and --input-file are specified.");
                throw new ArgumentException("Both of --html-dir and --input-file are specified.");
            }
            if (parsed.HtmlDir == null && parsed.InputFiles == null)
            {
                OutputLog.WriteError("No files or directory is specified.");
                throw new ArgumentException("No files or directory is specified.");
            }
            if (parsed.HtmlDir != null)
            {
                if (!Directory.Exists(parsed.HtmlDir))
                {
                    OutputLog.WriteError("Not exist htmldir.");
                    throw new DirectoryNotFoundException(string.Format("Not exist html directory. {0}", parsed.HtmlDir));
                }
            }

            //タグ情報を読み出し
            var apiInfo = TagInfomation.ReadTagInfo(parsed.TagInfoXmlFiles, logLevel, parsed.PreApiUrl);

            string[] filePaths;
            if (parsed.InputFiles != null)
            {
                // .html でない場合は戻る
                Parallel.ForEach(parsed.InputFiles, (fp) =>
                {
                    if (Path.GetExtension(fp) != ".html")
                    {
                        OutputLog.WriteError(string.Format(":::The input file is not HTML file. {0}.", fp));
                        throw new ArgumentException("The input file is not HTML file.");
                    }
                });
                filePaths = parsed.InputFiles;
            }
            else
            {
                // 指定フォルダの html ファイルに対して処理
                filePaths = Directory.GetFiles(parsed.HtmlDir).Where(s => s.EndsWith(".html", StringComparison.OrdinalIgnoreCase)).ToArray();
            }

            // html ファイル毎に独立した処理なので並列化
            Parallel.ForEach(filePaths, (fp) =>
                 {
                     if (!(File.Exists(fp)))
                     {
                         OutputLog.WriteError(string.Format("Not exist tag infomation file {0}.", fp));
                         throw new FileNotFoundException("Not exist tag infomation file.", fp);
                     }
                     else
                     {
                         ApiLinkArranger.ArrangeHtmlHead(apiInfo, fp, parsed.SaveAs, logLevel);
                         OutputLog.WriteDetail("Done.", logLevel);
                     }
                 });
            return 0;
        }
    }
}
