﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace VsSolutionLibrary
{
    /// <summary>
    /// プロジェクトの種類を表す列挙体
    /// </summary>
    public enum VsProjectType
    {
        VisualBasic,
        VisualC,
        CSharp,
        FSharp,
        VisualJSharp,
        Web,
        SolutionFolder
    }

    /// <summary>
    /// Visual Studio Project ファイルを表現する構造体
    /// </summary>
    public class VsSolutionProjectProperty
    {
        private readonly VsProjectType m_ProjectType;
        public VsProjectType ProjectType
        {
            get { return m_ProjectType; }
        }

        private readonly string m_Name;
        public string Name
        {
            get { return m_Name; }
        }

        private readonly string m_Path;
        public string Path
        {
            get { return m_Path; }
        }

        private readonly IEnumerable<Guid> m_Dependencies;
        public IEnumerable<Guid> Dependencies
        {
            get { return m_Dependencies; }
        }

        public VsSolutionProjectProperty(VsProjectType projectType, string name, string path, IEnumerable<Guid> dependencies = null)
        {
            if (name == null || path == null)
            {
                throw new ArgumentNullException();
            }

            m_ProjectType = projectType;
            m_Name = name;
            m_Path = path;

            if (dependencies == null)
            {
                m_Dependencies = new List<Guid>();
            }
            else
            {
                m_Dependencies = new List<Guid>(dependencies);
            }
        }

        public override bool Equals(object obj)
        {
            if (obj == null || !(obj is VsSolutionProjectProperty))
            {
                return false;
            }

            return ((VsSolutionProjectProperty)obj).ProjectType.Equals(this.ProjectType) &&
                ((VsSolutionProjectProperty)obj).Name.Equals(this.Name) &&
                ((VsSolutionProjectProperty)obj).Path.Equals(this.Path) &&
                ((VsSolutionProjectProperty)obj).Dependencies.SequenceEqual(this.Dependencies);
        }

        public override int GetHashCode()
        {
            return this.ProjectType.GetHashCode() ^ this.Name.GetHashCode() ^ this.Path.GetHashCode() ^ this.Dependencies.GetHashCode();
        }

        public static bool operator ==(VsSolutionProjectProperty lhs, VsSolutionProjectProperty rhs)
        {
            if (ReferenceEquals(lhs, null))
            {
                return ReferenceEquals(rhs, null);
            }

            return lhs.Equals(rhs);
        }

        public static bool operator !=(VsSolutionProjectProperty lhs, VsSolutionProjectProperty rhs)
        {
            if (ReferenceEquals(lhs, null))
            {
                return !ReferenceEquals(rhs, null);
            }

            return !lhs.Equals(rhs);
        }
    }

    /// <summary>
    /// ソリューションファイルのバージョンを表現する列挙型
    /// </summary>
    public enum VsSolutionVersion
    {
        VisualStudio2012,
        VisualStudio2013,
        VisualStudio2015,
        VisualStudio2017
    }

    /// <summary>
    /// ソリューションの構成
    /// </summary>
    public class VsConfigurationPair
    {
        // ビルドコンフィギュレーション
        private readonly string m_Configuration;
        public string Configuration
        {
            get { return m_Configuration; }
        }

        private readonly string m_Platform;
        public string Platform
        {
            get { return m_Platform; }
        }

        public VsConfigurationPair(string configuration, string platform)
        {
            if (configuration == null || platform == null)
            {
                throw new ArgumentNullException();
            }

            m_Configuration = configuration;
            m_Platform = platform;
        }

        /// <summary>
        /// 'Configuration|Platform' 形式の文字列から、ConfigurationPair を作成します。
        /// </summary>
        /// <param name="conigurationPair">'Configuration|Platform' 形式の文字列</param>
        public VsConfigurationPair(string conigurationPair)
        {
            if (conigurationPair == null)
            {
                throw new ArgumentNullException();
            }

            string[] strings = conigurationPair.Split('|');

            if (strings.Length == 2)
            {
                m_Configuration = strings[0];
                m_Platform = strings[1];
            }
            else
            {
                // フォーマット不一致の場合は例外を返す
                throw new ArgumentException();
            }
        }

        public override string ToString()
        {
            return Configuration + '|' + Platform;
        }

        public override bool Equals(object obj)
        {
            if (obj == null || !(obj is VsConfigurationPair))
            {
                return false;
            }

            return ((VsConfigurationPair)obj).Configuration.Equals(this.Configuration) &&
                ((VsConfigurationPair)obj).Platform.Equals(this.Platform);
        }

        public override int GetHashCode()
        {
            return this.Configuration.GetHashCode() ^ this.Platform.GetHashCode();
        }

        public static bool operator ==(VsConfigurationPair lhs, VsConfigurationPair rhs)
        {
            if (ReferenceEquals(lhs, null))
            {
                return ReferenceEquals(rhs, null);
            }

            return lhs.Equals(rhs);
        }

        public static bool operator !=(VsConfigurationPair lhs, VsConfigurationPair rhs)
        {
            if (ReferenceEquals(lhs, null))
            {
                return !ReferenceEquals(rhs, null);
            }

            return !lhs.Equals(rhs);
        }
    }

    /// <summary>
    /// プロジェクトの構成
    /// </summary>
    public class VsProjectConfiguration
    {
        private readonly VsConfigurationPair m_ConfigurationPair;
        public VsConfigurationPair ConfigurationPair
        {
            get { return m_ConfigurationPair; }
        }

        private readonly bool m_EnableBuild;
        public bool EnableBuild
        {
            get { return m_EnableBuild; }
        }

        public VsProjectConfiguration(VsConfigurationPair configurationPair, bool enableBuild)
        {
            if (configurationPair == null)
            {
                throw new ArgumentNullException();
            }

            m_ConfigurationPair = configurationPair;
            m_EnableBuild = enableBuild;
        }

        public override bool Equals(object obj)
        {
            if (obj == null || !(obj is VsProjectConfiguration))
            {
                return false;
            }

            return ((VsProjectConfiguration)obj).ConfigurationPair.Equals(this.ConfigurationPair) &&
                ((VsProjectConfiguration)obj).EnableBuild.Equals(this.EnableBuild);
        }

        public override int GetHashCode()
        {
            return this.ConfigurationPair.GetHashCode() ^ this.EnableBuild.GetHashCode();
        }

        public static bool operator ==(VsProjectConfiguration lhs, VsProjectConfiguration rhs)
        {
            if (ReferenceEquals(lhs, null))
            {
                return ReferenceEquals(rhs, null);
            }

            return lhs.Equals(rhs);
        }

        public static bool operator !=(VsProjectConfiguration lhs, VsProjectConfiguration rhs)
        {
            if (ReferenceEquals(lhs, null))
            {
                return !ReferenceEquals(rhs, null);
            }

            return !lhs.Equals(rhs);
        }
    }

    /// <summary>
    /// ソリューションフォルダ間ないしソリューションフォルダ・プロジェクトのネスト関係
    /// </summary>
    public class VsNestRelation
    {
        private readonly Guid m_Parent;
        public Guid Parent
        {
            get { return m_Parent; }
        }

        private readonly Guid m_Child;
        public Guid Child
        {
            get { return m_Child; }
        }

        public VsNestRelation(Guid parent, Guid child)
        {
            m_Parent = parent;
            m_Child = child;
        }

        public override string ToString()
        {
            return "{" + Parent.ToString() + "}, {" + Child.ToString() + "}";
        }

        public override bool Equals(object obj)
        {
            if (obj == null || !(obj is VsNestRelation))
            {
                return false;
            }

            return ((VsNestRelation)obj).Parent.Equals(this.Parent) &&
                ((VsNestRelation)obj).Child.Equals(this.Child);
        }

        public override int GetHashCode()
        {
            return Parent.GetHashCode() ^ Child.GetHashCode();
        }

        public static bool operator ==(VsNestRelation lhs, VsNestRelation rhs)
        {
            if (ReferenceEquals(lhs, null))
            {
                return ReferenceEquals(rhs, null);
            }

            return lhs.Equals(rhs);
        }

        public static bool operator !=(VsNestRelation lhs, VsNestRelation rhs)
        {
            if (ReferenceEquals(lhs, null))
            {
                return !ReferenceEquals(rhs, null);
            }

            return !lhs.Equals(rhs);
        }
    }

    /// <summary>
    /// Visual Studio Solution ファイルを表現するクラス
    /// </summary>
    public class VsSolution
    {
        /// <summary>
        /// ソリューションに属するプロジェクトの GUID の一覧
        /// </summary>
        public IEnumerable<Guid> Projects
        {
            get
            {
                return m_Projects.Keys;
            }
        }

        // Projects の本体
        private Dictionary<Guid, VsSolutionProjectProperty> m_Projects;

        /// <summary>
        /// ソリューションファイルのバージョン
        /// </summary>
        public VsSolutionVersion Version { get; set; }

        /// <summary>
        /// ソリューション構成の一覧
        /// </summary>
        public IEnumerable<VsConfigurationPair> Configurations
        {
            get
            {
                return new List<VsConfigurationPair>(m_Configurations.Values);
            }
        }

        // Configurations の本体
        // Key には VsConfigurationPair.ToString() を使う
        private SortedList<string, VsConfigurationPair> m_Configurations;

        /// <summary>
        /// ソリューションプロパティの一覧
        /// </summary>
        public IDictionary<string, string> SolutionProperties { get; set; }

        /// <summary>
        /// ソリューションフォルダ間ないしソリューションフォルダ・プロジェクトのネスト関係の一覧
        /// </summary>
        public IEnumerable<VsNestRelation> NestRelations
        {
            get
            {
                return new List<VsNestRelation>(m_NestRelations);
            }
        }

        // NestRelations のバッキングストア
        private List<VsNestRelation> m_NestRelations;

        /// <summary>
        /// 空の VsSolution オブジェクトを構築する
        /// </summary>
        public VsSolution()
        {
            Initialize();
        }

        /// <summary>
        /// .sln ファイルの文字列をパースして、対応する VsSolution オブジェクトを構築する
        /// </summary>
        /// <param name="content">.sln ファイルの内容に相当する文字列</param>
        public VsSolution(string text)
        {
            if (text == null)
            {
                throw new ArgumentNullException();
            }

            Initialize();
            Parser.Parse(text, this);
        }

        /// <summary>
        /// オブジェクトの内容に対応する、 .sln ファイルの文字列を返す
        /// </summary>
        /// <returns>.sln ファイルの内容に相当する文字列</returns>
        public override string ToString()
        {
            return Dumper.Dump(this);
        }

        /// <summary>
        /// ソリューションに VsProject を追加する
        /// </summary>
        public void AddProject(Guid guid, VsSolutionProjectProperty property)
        {
            if (property == null)
            {
                throw new ArgumentNullException();
            }

            if (m_Projects.Keys.Contains(guid))
            {
                throw new ArgumentException();
            }

            // Project 追加
            m_Projects.Add(guid, property);

            // Project 構成も追加
            foreach (var solutionConfiguration in Configurations)
            {
                m_ProjectConfiguration.Add(new ProjectConfigurationKey(solutionConfiguration, guid), new VsProjectConfiguration(solutionConfiguration, false));
            }
        }

        /// <summary>
        /// ソリューションから VsProject を削除する
        /// </summary>
        public void RemoveProject(Guid guid)
        {
            // Project 削除
            m_Projects.Remove(guid);

            // Project 構成も削除
            foreach (var solutionConfiguration in Configurations)
            {
                m_ProjectConfiguration.Remove(new ProjectConfigurationKey(solutionConfiguration, guid));
            }
        }

        /// <summary>
        /// 指定した GUID の VsProject のプロパティを取得する
        /// </summary>
        public VsSolutionProjectProperty GetProjectProperty(Guid guid)
        {
            if (!m_Projects.Keys.Contains(guid))
            {
                throw new ArgumentException();
            }

            return m_Projects[guid];
        }

        /// <summary>
        /// 指定した GUID の VsProject のプロパティを変更する
        /// </summary>
        public void SetProjectProperty(Guid guid, VsSolutionProjectProperty property)
        {
            if (!m_Projects.Keys.Contains(guid))
            {
                throw new ArgumentException();
            }

            m_Projects[guid] = property;
        }

        /// <summary>
        /// ソリューション構成を追加する
        /// </summary>
        /// <param name="configuration">追加するソリューション構成</param>
        public void AddSolutionConfiguration(VsConfigurationPair configuration)
        {
            if (configuration == null)
            {
                throw new ArgumentNullException();
            }

            if (Configurations.Contains(configuration))
            {
                throw new ArgumentException();
            }

            // ソリューション構成追加
            m_Configurations.Add(configuration.ToString(), configuration);

            // プロジェクト構成も追加
            foreach (var project in Projects)
            {
                m_ProjectConfiguration.Add(new ProjectConfigurationKey(configuration, project), new VsProjectConfiguration(configuration, false));
            }
        }

        /// <summary>
        /// ソリューション構成を削除する
        /// </summary>
        /// <param name="configuration">削除するソリューション構成</param>
        public void RemoveSolutionConfiguration(VsConfigurationPair configuration)
        {
            if (configuration == null)
            {
                return;
            }

            // ソリューション構成削除
            m_Configurations.Remove(configuration.ToString());

            // プロジェクト構成も削除
            foreach (var project in Projects)
            {
                m_ProjectConfiguration.Remove(new ProjectConfigurationKey(configuration, project));
            }
        }

        /// <summary>
        /// あるソリューション構成・プロジェクトのペアに対応するプロジェクト構成を取得する
        /// </summary>
        /// <param name="solutionConfiguration">ソリューション構成 (key)</param>
        /// <param name="project">プロジェクトの GUID (key)</param>
        /// <returns>プロジェクト構成 (value)</returns>
        public VsProjectConfiguration GetProjectConfiguration(VsConfigurationPair solutionConfiguration, Guid project)
        {
            if (solutionConfiguration == null)
            {
                throw new ArgumentNullException();
            }

            ProjectConfigurationKey key = new ProjectConfigurationKey(solutionConfiguration, project);
            if (!m_ProjectConfiguration.Keys.Contains(key))
            {
                throw new ArgumentException();
            }

            return m_ProjectConfiguration[key];
        }

        /// <summary>
        /// あるソリューション構成・プロジェクトのペアに対応するプロジェクト構成を設定する
        /// </summary>
        /// <param name="configuration">ソリューション構成 (key)</param>
        /// <param name="project">プロジェクトの GUID (key)</param>
        /// <param name="projectConfiguration">プロジェクト構成 (value)</param>
        public void SetProjectConfiguration(VsConfigurationPair solutionConfiguration, Guid project, VsProjectConfiguration projectConfiguration)
        {
            if (solutionConfiguration == null || projectConfiguration == null)
            {
                throw new ArgumentNullException();
            }

            ProjectConfigurationKey key = new ProjectConfigurationKey(solutionConfiguration, project);
            if (!m_ProjectConfiguration.Keys.Contains(key))
            {
                throw new ArgumentException();
            }

            m_ProjectConfiguration[key] = projectConfiguration;
        }

        /// <summary>
        /// ソリューションフォルダ間ないしソリューションフォルダ・プロジェクトのネスト関係を追加する
        /// </summary>
        /// <param name="relation">追加するネスト関係</param>
        public void AddNestRelation(VsNestRelation relation)
        {
            if (relation == null)
            {
                throw new ArgumentNullException();
            }

            if (m_NestRelations.Contains(relation))
            {
                throw new ArgumentException("追加しようとしたネスト関係は既に存在しています。");
            }

            m_NestRelations.Add(relation);
        }

        /// <summary>
        /// ソリューションフォルダ間ないしソリューションフォルダ・プロジェクトのネスト関係を削除する
        /// </summary>
        /// <param name="relation">削除するネスト関係</param>
        public void RemoveNestRelation(VsNestRelation relation)
        {
            if (relation == null)
            {
                throw new ArgumentNullException();
            }

            if (!m_NestRelations.Contains(relation))
            {
                throw new ArgumentException("削除しようとしたネスト関係が存在しません。");
            }

            m_NestRelations.Remove(relation);
        }

        // プロジェクト構成のデータ本体
        private Dictionary<ProjectConfigurationKey, VsProjectConfiguration> m_ProjectConfiguration;

        private class ProjectConfigurationKey
        {
            public readonly VsConfigurationPair m_SolutionConfiguration;
            public readonly Guid m_ProjectGuid;

            public ProjectConfigurationKey(VsConfigurationPair solutionConfiguration, Guid projectGuid)
            {
                m_SolutionConfiguration = solutionConfiguration;
                m_ProjectGuid = projectGuid;
            }

            public override bool Equals(object obj)
            {
                if (obj == null || !(obj is ProjectConfigurationKey))
                {
                    return false;
                }

                return ((ProjectConfigurationKey)obj).m_SolutionConfiguration.Equals(this.m_SolutionConfiguration) &&
                    ((ProjectConfigurationKey)obj).m_ProjectGuid.Equals(this.m_ProjectGuid);
            }

            public override int GetHashCode()
            {
                return this.m_SolutionConfiguration.GetHashCode() ^ this.m_ProjectGuid.GetHashCode();
            }

            public static bool operator ==(ProjectConfigurationKey lhs, ProjectConfigurationKey rhs)
            {
                if (ReferenceEquals(lhs, null))
                {
                    return ReferenceEquals(rhs, null);
                }

                return lhs.Equals(rhs);
            }

            public static bool operator !=(ProjectConfigurationKey lhs, ProjectConfigurationKey rhs)
            {
                if (ReferenceEquals(lhs, null))
                {
                    return !ReferenceEquals(rhs, null);
                }

                return !lhs.Equals(rhs);
            }
        }

        // 初期化が必要なメンバを初期化する
        private void Initialize()
        {
            this.m_Projects = new Dictionary<Guid, VsSolutionProjectProperty>();
            this.m_Configurations = new SortedList<string, VsConfigurationPair>();
            this.SolutionProperties = new Dictionary<string, string>();
            this.Version = VsSolutionVersion.VisualStudio2012;
            this.m_ProjectConfiguration = new Dictionary<ProjectConfigurationKey, VsProjectConfiguration>();
            this.m_NestRelations = new List<VsNestRelation>();
        }
    }
}
