﻿<#
    Copyright (C)Nintendo All rights reserved.

    These coded instructions, statements, and computer programs contain proprietary
    information of Nintendo and/or its licensed developers and are protected by
    national and international copyright laws. They may not be disclosed to third
    parties or copied or duplicated in any form, in whole or in part, without the
    prior written consent of Nintendo.

    The content herein is highly confidential and should be handled accordingly.
#>
<#
    .SYNOPSIS
        Test if the current environment is suitable for siglo development

    .PARAMETER TestLevel
        Every test whose test level is smaller than or equal to this value will be executed.
#>

<#
    More tests are really appreciated. Thanks!

    HOW TO ADD TESTS
    
        1. Add a test function for each item to test.
           
           The function should throw an error message if it detects some error; it should not otherwise.
           
           Also add a .SYNOPSIS comment to the function.

        2. Add invocation of that function:
           
               Invoke-Test <test level> <function name>
               
    TEST LEVELS

        * 0 : Tests executed from siglo.cmd.
        * 1 : Other tests.

    NOTE
    
        * Each test is all-or-nothing. Small tests are preferred.
    
#>

param
(
    [int]$TestLevel = [int]::MaxValue
)

#
# Script globals
#

$scriptPath          = $MyInvocation.MyCommand.Path
$scriptDirectoryPath = [System.IO.Path]::GetDirectoryName($scriptPath)


#
# Imports
#

Import-Module "${scriptDirectoryPath}\Modules\Error"
Import-Module "${scriptDirectoryPath}\Modules\Path"


#
# Framework functions
#

function New-TestStatus
{
    param(
        [string]$name,
        [string]$description,
        [string]$error = ""
    )
    
    $isSuccessful = ($error -ceq "")
    
    return New-Object -TypeName PSObject |
        Add-Member -MemberType NoteProperty -Name Successful -Value $isSuccessful -PassThru |
        Add-Member -MemberType NoteProperty -Name Name -Value $name -PassThru |
        Add-Member -MemberType NoteProperty -Name Description -Value $description -PassThru |
        Add-Member -MemberType NoteProperty -Name Error -Value $error -PassThru
}

function Invoke-Test([int]$thisTestLevel, $command)
{
    <#
        .SYNOPSIS
        Invoke the specified test command and generate a TestStatus
    #>
    if ($thisTestLevel -gt $TestLevel)
    {
        return
    }

    $help = (Get-Help $command)
    $name = $help.Name
    $synopsis = $help.Synopsis

    try
    {
        & $command
        return New-testStatus $name $synopsis
    }
    catch
    {
        return New-testStatus $name $synopsis $_.Exception.Message
    }
}


#
# Test functions
#

function Test-SdkRootPathShortEnough
{
    <#
        .SYNOPSIS
            Test if the path to siglo root contains < 48 chars. (Otherwise build may fail due to paths longer than MAXPATH.)
    #>

    $sdkRoot = $(Get-NintendoSdkRootPath)
    if (!($sdkRoot.length -lt 48))
    {
        throw "The path to siglo root is too long."
    }
}

function Test-NactExeExpanded
{
    <#
        .SYNOPSIS
            Test if nact.exe is exstorage-expanded (i.e. setupSiglo.cmd was executed).
    #>

    $sdkRoot = $(Get-NintendoSdkRootPath)
    $info = Get-item ([System.IO.Path]::Combine($sdkRoot, "Integrate\CommandLineTools\nact.exe"))
    
    if ($info.Length -lt 100)
    {
        throw "nact.exe is not exstorage-expanded."
    } 
}

function Test-NoDoubleQuotesInPath
{
    <#
        .SYNOPSIS
            Test if the PATH environment variable contains no double quote character ('"').
    #>

    if ((Get-Content env:PATH) -match '"')
    {
        throw "The PATH environment variable contains double quote character(s). Some programs cannot handle such value. (It is not necessary to quote paths with spaces in the PATH environment variable.)"
    }
}

function Test-GitPath
{
    <#
        .SYNOPSIS
            Test if SIGLO_GIT_PATH points to the git installation directory. (The environment must have Git for Windows installed with the installer.)
    #>

    $gitDirPath = $env:SIGLO_GIT_PATH
    $gitExePath = "$gitDirPath\cmd\git.exe"
    
    if (!(Test-Path $gitExePath))
    {
        throw "$gitExePath (%SIGLO_GIT_PATH%cmd\git.exe) does not exist."
    }
}

function Test-VulnerableGitVersion
{
    param([int[]]$version)

    function versionLt
    {
        param($lhs, $rhs)

        for ($i = 0; $i -lt 4; $i++)
        {
            if ($lhs[$i] -lt $rhs[$i])
            {
                return $true
            }
            elseif ($lhs[$i] -gt $rhs[$i])
            {
                return $false
            }
        }

        return $false
    }

    if (versionLt $version (2,0,0,0))
    {
        return $true
    }


    $major = $version[0]
    $minor = $version[1]
    $micro = $version[2]
    if ($major -eq 2)
    {
        switch ($version[1])
        {
            12
            {
                # earlier than 2.12.2.windows.3
                return versionLt $version (2,12,2,3)
            }
            13
            {
                # 2.13.2 and 2.13.3 are vulnerable
                if ($micro -eq 2 -or $micro -eq 3)
                {
                    return $true
                }

                # earlier than 2.13.1.windows.3
                return versionLt $version (2,13,1,3)
            }
            14
            {
                # earlier than 2.14.1.windows.1
                return versionLt $version (2,14,1,1)
            }
            { $_ -ge 15 }
            {
                return $false
            }
            default
            {
                return $true
            }
        }
    }
    
    return $false
}

function Test-GitVersion
{
    <#
        .SYNOPSIS
            Test if the installed version of Git is Git for Windows and does not have serious issues.
    #>

    $gitDirPath = $env:SIGLO_GIT_PATH
    $gitExePath = "$gitDirPath\cmd\git.exe"
    $versionString = & $gitExePath --version

    if (!($versionString -match "^git version (\d+).(\d+).(\d+).windows.(\d+)$"))
    {
        throw "$gitExePath : Version string '$versionString' cannot be recognized. Update it to Git for Windows."
    }
    
    $versionArray = $matches[1..4]
    if (Test-VulnerableGitVersion $versionArray)
    {
        throw "$gitExePath : Git '$versionString' is vulnerable. Update it to 2.12.2.windows.3, 2.13.1.windows.3, 2.14.1.windows.1 or later."
    }
}

function Test-VSInstalled([string]$version)
{
    try
    {
        $installDir = Get-SoftwareRegistryValue "Microsoft\VisualStudio\SxS\VS7\" $version
        if (!(Test-Path $installDir))
        {
            throw "Not installed. ($installDir not found)"
        }
        
        $clexepath = [System.IO.Path]::Combine($installDir, "VC\bin\cl.exe")
        if (!(Test-Path $clexepath))
        {
            throw "VC++ not installed. ($clexepath not found)"
        }
    }
    catch [System.IO.FileNotFoundException]
    {
        throw "Not installed. ($_)"
    }    
}

function Test-VS2015Installed
{
    <#
        .SYNOPSIS
            Test if VS2015 and VC++2015 are installed. (Required as of July/2016)
    #>

    Test-VSInstalled "14.0"
}

function Test-MSBuild2015Installed
{
    <#
        .SYNOPSIS
            Test if Microsoft Build Tools 2015 (or VS2015) is installed. (https://www.microsoft.com/download/details.aspx?id=48159)
    #>

    try
    {
        $installDir = Get-SoftwareRegistryValue "Microsoft\MSBuild\ToolsVersions\14.0" "MSBuildToolsPath"
        $exepath = [System.IO.Path]::Combine($installDir, "MSBuild.exe")
        if (!(Test-Path $exepath))
        {
            throw "Not installed. ($exepath not found.)"
        }
    }
    catch [System.IO.FileNotFoundException]
    {
        throw "Not installed. ($_)"
    }
}

function Get-LocateVsInstallationPath
{
    $checkerPath = Resolve-Path (Join-Path (Get-NintendoSdkRootPath) "Integrate\CommandLineTools\LocateVsInstallation.exe")
    $checkerItem = Get-Item $checkerPath

    if ($checkerItem.Length -lt 100)
    {
        throw "Installation checker tool (LocateVsInstallation.exe) is not exstorage-expanded."
    }

    return $checkerPath
}

function Test-VS2017BasicBuildComponentInstalled
{
    <#
        .SYNOPSIS
            Test if basic build tools of Visual Studio 2017 are installed. (https://www.visualstudio.com/downloads/#build-tools-for-visual-studio-2017)
    #>

    $message = & (Get-LocateVsInstallationPath) --basic 2>&1

    if ($LastExitCode)
    {
        throw "Required component not installed. Please see http://spdlybra.nintendo.co.jp/confluence/pages/viewpage.action?pageId=47813135 : $message"
    }
}

function Test-VS2017FullDevelopmentComponentInstalled
{
    <#
        .SYNOPSIS
            Test if components of Visual Studio 2017 required for VC++ library, tool and VSIX development are installed.
    #>

    $message = & (Get-LocateVsInstallationPath) --full 2>&1

    if ($LastExitCode)
    {
        throw "Required component not installed. Please see http://spdlybra.nintendo.co.jp/confluence/pages/viewpage.action?pageId=47813135 : $message"
    }
}

function Test-HibunNotInstalled
{
    <#
        .SYNOPSIS
            Test if Hibun is not installed. The Siglo tree does not work with Hibun installed.
    #>

    if (Test-Path "$env:SYSTEMROOT\sfbksys")
    {
        throw "Hibun is installed."
    }
}

function Test-UsingNactInCurrentWT
{
    <#
        .SYNOPSIS
            Test if nact.exe in PATH (if any) is the one in the current working tree.
    #>

    $nact = Get-Command nact.exe -ErrorAction SilentlyContinue | Select-Object -first 1 
    if ($nact -ne $null)
    {
        $sdkRoot = $(Get-NintendoSdkRootPath)
        $expectedNactPath = [System.IO.Path]::GetFullPath([System.IO.Path]::Combine($sdkRoot, "Integrate\CommandLineTools\nact.exe"))
        $nactPath = [System.IO.Path]::GetFullPath($nact.Definition)
        
        if ($nactPath -ne $expectedNactPath)
        {
            throw "nact.exe in another working tree is in PATH. ($nactPath)"
        }
    }
}

function Test-PreCommitHookInstalled
{
    <#
        .SYNOPSIS
            Test if the Siglo pre-commit hook is installed.
    #>

    $sdkRoot = $(Get-NintendoSdkRootPath)
    # TODO: Obtain the .git path with `git rev-parse --git-common-dir` if git 2.x is available.
    if (!(Test-Path "$sdkRoot\.git\hooks\pre-commit"))
    {
        throw "The Siglo pre-commit hook is not installed. (Run siglo.cmd.)"
    }
}

function Test-SubmodulesAreClean
{
    <#
        .SYNOPSIS
            Test if submodules are initialized and clean.
    #>

    $gitDirPath = $env:SIGLO_GIT_PATH
    $gitExePath = "$gitDirPath\cmd\git.exe"
    
    # This command takes time!
    $dirtySubmodules = & $gitExePath submodule status | Where-Object { !$_.StartsWith(' ') }
    if ($dirtySubmodules -ne $null)
    {
        throw "The following submodules are either uninitialized or dirty. (Ignore this error if these modifications are intended.) :`n $dirtySubmodules"
    }
}

function Test-PowerShell5.1IsInstalled
{
    <#
        .SYNOPSIS
            Test if PowerShell 5.1 is installed.
    #>

    $currentVersion = $PSVersionTable.PSVersion
    $expectedVersion = New-Object Version (5, 1)
    
    if ($currentVersion -lt $expectedVersion)
    {
        throw "PowerShell 5.1 is not installed. (Current version : ${currentVersion})"
    }
}

#
# Test invocations
#
Invoke-Test 0 Test-SdkRootPathShortEnough
Invoke-Test 0 Test-NactExeExpanded
Invoke-Test 0 Test-NoDoubleQuotesInPath
Invoke-Test 0 Test-GitPath
Invoke-Test 0 Test-GitVersion
Invoke-Test 0 Test-VS2015Installed
Invoke-Test 0 Test-MSBuild2015Installed
Invoke-Test 0 Test-VS2017BasicBuildComponentInstalled
Invoke-Test 1 Test-VS2017FullDevelopmentComponentInstalled
Invoke-Test 0 Test-HibunNotInstalled
Invoke-Test 1 Test-UsingNactInCurrentWT
Invoke-Test 1 Test-PreCommitHookInstalled # Level 1 because siglo.cmd installs our pre-commit hook.
Invoke-Test 1 Test-SubmodulesAreClean
Invoke-Test 0 Test-PowerShell5.1IsInstalled

exit 0

