<#
    Copyright (C)Nintendo All rights reserved.

    These coded instructions, statements, and computer programs contain proprietary
    information of Nintendo and/or its licensed developers and are protected by
    national and international copyright laws. They may not be disclosed to third
    parties or copied or duplicated in any form, in whole or in part, without the
    prior written consent of Nintendo.

    The content herein is highly confidential and should be handled accordingly.
#>

<#
    .SYNOPSIS
        Runs Bluetooth tests with sleep/wake.  These are stress tests that should run for a long time.

    .DESCRIPTION
        - Runs Bluetooth tests with sleep/wake
           - Runs testBluetooth_Pairing.nsp so the device can start pairing
           - Runs a series of ControlExtCon.exe commands on the US Writer device to automate button presses on the controller
           - Verifies the button presses were accepted on the device
        
#>

param
(
    [string]
    # SDK root directory
    $SigloRoot =
    ($env:NINTENDO_SDK_ROOT | ? { $_ } | % { $_.Trim() }),

    [string]
    # Device's IP address
    $TargetAddress =
    ($env:TARGET_ADDRESS | ? { $_ } | % { $_.Trim() }),
    
    [int]
    # Length of time to run in seconds
    $TimeToRunSeconds =
    ($env:TEST_NXAGING_TIME_SECONDS | ? { $_ } | % { $_.Trim() }),

    [string]
    # eg. Tests\Outputs\NX-NXFP2-a64\Tests\testBluetooth_Pairing\Develop\testBluetooth_Pairing.nsp
    $TestListApp =
    ($env:TESTRUNNER_EXE_LIST),
    
    [string]
    # Port number of US Writer
    #   eg. com11
    $PortOfUsWriter =
    ($env:PORT_US_WRITER | ? { $_ } | % { $_.Trim() })
)


# Telnet port
$TelnetPort = 10023
# Number of bytes to read at a time
$BytesToRead = 1024
# Max time in seconds to wait for telnet log
$MaxWaitTimeSec = 10

# Controller pair command success string in Telnet logs
$PairSuccess = "[bluetooth] BSA_SEC_AUTH_CMPL_EVT received"
# Controller page command success string in Telnet logs
$PageSuccess = "[bluetooth] BSA_HH_OPEN_EVT"
# Controller wake command success string in Telnet logs
$WakeSuccess = "[bluetooth] Got FullAwake Notification"



# reads the telnet port 
function ReadTelnetPort ($LocalTargetAddress,$LocalTelnetPort)
{
    # Telnet into device port
    $socket = New-Object System.Net.Sockets.TCPClient($LocalTargetAddress,$LocalTelnetPort)
    if ($socket)
    {  
        $stream = $socket.GetStream()
        $buffer = New-Object System.Byte[] $BytesToRead
        $encoding = New-Object System.Text.AsciiEncoding
        
        # allow data to buffer
        Start-sleep -s $MaxWaitTimeSec
        
        $readResult = ""
        while ($stream.DataAvailable)
        {
            $read = $stream.Read($buffer, 0, $BytesToRead)
            $readResult += ($encoding.GetString($buffer, 0, $read))
        }

        $stream.Close()
        return $readResult
    }
}


try
{
    # make sure Target Manager is connected
    & Tools\CommandLineTools\ControlTarget.exe connect

    # enable logging on the device
    Write-Host "Enable logging on the device SD card"
    & Tools\CommandLineTools\RunOnTarget.exe ./Programs/Eris/Outputs/NX-NXFP2-a64/TargetTools/DevMenuCommandSystem/Develop/DevMenuCommandSystem.nsp -- debug enable-sd-card-logging
    & Tools\CommandLineTools\ControlTarget.exe reset --target $TargetAddress
    & Tools\CommandLineTools\ControlTarget.exe connect

    # time parameters
    $timeToRunTestsSeconds = $TimeToRunSeconds
    $stopWatch = [Diagnostics.Stopwatch]::StartNew()
    $elapsedTime = 0
    $elapsedTimeSeconds = 0

    # run tests until time has elapsed
    while ($elapsedTimeSeconds -lt $timeToRunTestsSeconds)
    {
        # put device into sleep, and then wake
        & Tools\CommandLineTools\ControlTarget.exe press-power-button
        Start-sleep -s 3
        & Tools\CommandLineTools\ControlTarget.exe press-power-button
        Start-sleep -s 3
        
        # delete all controller pairings
        & Tools\CommandLineTools\RunOnTarget.exe ./Programs/Eris/Outputs/NX-NXFP2-a64/TargetTools/DevMenuCommandSystem/Develop/DevMenuCommandSystem.nsp -- controller delete-all-pairing

        # start the pairing app on the device
        & Tools\CommandLineTools\RunOnTarget.exe --no-wait $TestListApp

        # press pair button on the controller, then verify in telnet log that the controller paired
        Write-Host "Sending controller pair command"
        & Integrate\Scripts\NX\GamePad\ControlExtCon.exe --port $PortOfUsWriter reset-pair
        $readStuffPair = ReadTelnetPort $TargetAddress $TelnetPort
        if (-Not ($readStuffPair.Contains($PairSuccess)))
        {
            Write-Host "##teamcity[testFailed name='Pair test']"
        }

        # press page button on the controller, then verify in telnet log is the expected result
        Write-Host "Sending controller page command"
        & Integrate\Scripts\NX\GamePad\ControlExtCon.exe --port $PortOfUsWriter reset-page
        $readStuffPage = ReadTelnetPort $TargetAddress $TelnetPort
        if (-Not ($readStuffPage.Contains($PageSuccess)))
        {
            Write-Host "##teamcity[testFailed name='Page test']"
        }

        # power off the device
        & Tools\CommandLineTools\ControlTarget.exe press-power-button
        Start-sleep -s 3

        # press home button on the controller, then verify in telnet log that the device turned on
        Write-Host "Sending controller Home button command"
        & Integrate\Scripts\NX\GamePad\ControlExtCon.exe --port $PortOfUsWriter reset-poweron
        $readStuffPowerOn = ReadTelnetPort $TargetAddress $TelnetPort
        if (-Not ($readStuffPowerOn.Contains($WakeSuccess)))
        {
            Write-Host "##teamcity[testFailed name='Wake test']"
        }

        # check time elapsed
        $elapsedTime = $stopWatch.Elapsed
        $elapsedTimeSeconds = $elapsedTime.TotalSeconds
    }
}

finally
{
    # disable logging on the device
    Write-Host "Disable logging on the device SD card"
    & Tools\CommandLineTools\RunOnTarget.exe ./Programs/Eris/Outputs/NX-NXFP2-a64/TargetTools/DevMenuCommandSystem/Develop/DevMenuCommandSystem.nsp -- debug disable-sd-card-logging
    & Tools\CommandLineTools\ControlTarget.exe reset --target $TargetAddress
    & Tools\CommandLineTools\ControlTarget.exe connect
}