﻿<#
    Copyright (C)Nintendo All rights reserved.

    These coded instructions, statements, and computer programs contain proprietary
    information of Nintendo and/or its licensed developers and are protected by
    national and international copyright laws. They may not be disclosed to third
    parties or copied or duplicated in any form, in whole or in part, without the
    prior written consent of Nintendo.

    The content herein is highly confidential and should be handled accordingly.
#>

<#
    .SYNOPSIS
        PowerUsb Module

    .DESCRIPTION
        This file defines functions for PowerUSB operations
#>

<#
    .SYNOPSIS
        Adds PwrUSBDll.dll to the current session
#>
function Import-PowerUsbDll
{
    param
    (
        [Parameter(Mandatory=$true)]
        [string]
        # The path to PwrUSBDll.dll
        $Path
    )

    if ([System.IntPtr]::Size -ne 4)
    {
        throw "PwrUSBDll.dll only works on 32-bit mode"
    }

    Add-Type -Language CSharp -TypeDefinition @"
using System.Runtime.InteropServices;
using System.Text;
public class PowerUSB
{
    [DllImport(@"$Path", CallingConvention = CallingConvention.Cdecl)]
    public static extern int InitPowerUSB(
        out int mode, StringBuilder firmwareVersion);
    [DllImport(@"$Path", CallingConvention = CallingConvention.Cdecl)]
    public static extern int ClosePowerUSB();
    [DllImport(@"$Path", CallingConvention = CallingConvention.Cdecl)]
    public static extern int SetPortPowerUSB(
        int port1_power, int port2_power, int port3_power);
    [DllImport(@"$Path", CallingConvention = CallingConvention.Cdecl)]
    public static extern int ReadPortStatePowerUSB(
        out int state1, out int state2, out int state3);
}
"@
}
Export-ModuleMember -Function Import-PowerUsbDll

<#
    .SYNOPSIS
        Initializes the PowerUSB API
#>
function Initialize-PowerUsb
{
    $model = 0
    $firmwareVersion = New-Object "System.Text.StringBuilder"
    $powerUsbCount = [PowerUSB]::InitPowerUSB([ref]$model, $firmwareVersion)
    if ($powerUsbCount -le 0) { throw "Power USB not found" }
}
Export-ModuleMember -Function Initialize-PowerUsb

<#
    .SYNOPSIS
        Close the PowerUSB API
#>
function Close-PowerUsb
{
    [void][PowerUSB]::ClosePowerUSB()
}
Export-ModuleMember -Function Close-PowerUsb

<#
    .SYNOPSIS
        Sets the power on/off state of the three outlets
#>
function Set-PowerUsbPortState
{
    param
    (
        [Parameter(Mandatory=$true)]
        [int]
        [ValidateRange(1, 3)]
        # Port number
        $PortNumber,

        [Parameter(Mandatory=$true)]
        [bool]
        # Port state
        $PortState
    )

    $x, $y, $z = 0, 0, 0
    if ([PowerUSB]::ReadPortStatePowerUSB([ref]$x, [ref]$y, [ref]$z) -lt 0)
    {
        throw "failed to read PowerUSB states"
    }

    $value = if ($PortState) { 1 } else { 0 }
    if ($PortNumber -eq 1) { $x = $value }
    if ($PortNumber -eq 2) { $y = $value }
    if ($PortNumber -eq 3) { $z = $value }

    if ([PowerUSB]::SetPortPowerUSB($x, $y, $z) -lt 0)
    {
        throw "failed to set PowerUSB states"
    }
}
Export-ModuleMember -Function Set-PowerUsbPortState
