﻿<#
    Copyright (C)Nintendo All rights reserved.

    These coded instructions, statements, and computer programs contain proprietary
    information of Nintendo and/or its licensed developers and are protected by
    national and international copyright laws. They may not be disclosed to third
    parties or copied or duplicated in any form, in whole or in part, without the
    prior written consent of Nintendo.

    The content herein is highly confidential and should be handled accordingly.
#>

<#
    .SYNOPSIS
        Path Module

    .DESCRIPTION
        This file defines functions for getting path
#>


$script:nintendoSdkRootPath = $null

function Get-TestedFilePath([string]$name, [string]$path)
{
    if ( Test-Path "${path}" )
    {
        return $path
    }
    $message = "${name} not found at '${path}'"
    throw New-Object "System.IO.FileNotFoundException" $message
}

function Get-TestedDirectoryPath([string]$name, [string]$path)
{
    if ( Test-Path "${path}" )
    {
        return $path
    }
    $message = "${name} directory not found"
    throw New-Object "System.IO.DirectoryNotFoundException" $message
}
Export-ModuleMember -Function Get-TestedDirectoryPath

<#
    .SYNOPSIS
        Obtains a registry value under SOFTWARE keys.

    .PARAMETER relativePathToKey
        Relative path to the key from SOFTWARE keys.

    .PARAMETER valueName
        Name of the value to obtain.

    .Description
        This function searches for a value with the specified name in each registry key below:

            "HKLM:SOFTWARE\$relativePathToKey"
            "HKCU:SOFTWARE\$relativePathToKey"
            "HKLM:SOFTWARE\Wow6432Node\$relativePathToKey"
            "HKCU:SOFTWARE\Wow6432Node\$relativePathToKey"

        This function throws System.IO.FileNotFoundException if no value is found.

    .OUTPUTS
        Registry value

#>
function Get-SoftwareRegistryValue
{
    param(
        [Parameter(Mandatory=$true)]
        [string]
        $relativePathToKey,

        [Parameter(Mandatory=$true)]
        [string]
        $valueName
     )

    $roots = "HKLM:SOFTWARE", "HKCU:SOFTWARE", "HKLM:SOFTWARE\Wow6432Node", "HKCU:SOFTWARE\Wow6432Node"
    foreach ($root in $roots)
    {
        $keyPath = "$root\$relativePathToKey"
        if (Test-Path $keyPath)
        {
            $value = (Get-ItemProperty $keyPath).$valueName
            if ($value -ne $null)
            {
                return $value
            }
        }
    }

    throw New-Object "System.IO.FileNotFoundException" "Registry value $relativePathToKey\$valueName not found in $roots"
}
Export-ModuleMember -Function Get-SoftwareRegistryValue

<#
    .SYNOPSIS
        Returns the Git path

    .OUTPUTS
        String instance that contains path information.
#>
function Get-GitPath
{
    try
    {
        $intallLocation = Get-SoftwareRegistryValue "Microsoft\Windows\CurrentVersion\Uninstall\Git_is1" InstallLocation
    }
    catch [System.IO.FileNotFoundException]
    {
        throw New-Object "System.IO.FileNotFoundException" ("git not installed", $_.Exception)
    }
    return Get-TestedFilePath "git" ([System.IO.Path]::Combine($intallLocation, "cmd\git.exe"))
}
Export-ModuleMember -Function Get-GitPath

<#
    .SYNOPSIS
        Returns the 64-bit MSBuild path

    .OUTPUTS
        String instance that contains path information.
#>
function Get-MSBuildX64Path
{
    $path = "${env:WINDIR}\Microsoft.NET\Framework64\v4.0.30319\MSBuild.exe"
    return Get-TestedFilePath "MSBuild" $path
}
Export-ModuleMember -Function Get-MSBuildX64Path

<#
    .SYNOPSIS
        Returns the 32-bit MSBuild path

    .OUTPUTS
        String instance that contains path information.
#>
function Get-MSBuildX86Path
{
    trap [System.Management.Automation.ActionPreferenceStopException]
    {
        $ex      = $_.Exception
        $message = "MSBuildTool not installed"
        throw New-Object "System.IO.FileNotFoundException" ($message, $ex)
    }
    $key  = "HKLM:\SOFTWARE\Wow6432Node\Microsoft\MSBuild\ToolsVersions\14.0"
    $prop = Get-ItemProperty $key -ErrorAction Stop
    $path = $prop.MSBuildToolsPath + "\MSBuild.exe"
    return Get-TestedFilePath "MSBuild" $path
}
Export-ModuleMember -Function Get-MSBuildX86Path

<#
    .SYNOPSIS
        Returns the root directory of MSBuild 15.
        ex) "C:\Program Files (x86)\Microsoft Visual Studio\2017\Professional\MSBuild\15.0"

    .OUTPUTS
        String instance that contains path information.
#>
function Get-MSBuild15RootPath
{
    trap [System.Management.Automation.ActionPreferenceStopException]
    {
        $ex      = $_.Exception
        $message = "MSBuild 15 not found."
        throw New-Object "System.IO.FileNotFoundException" ($message, $ex)
    }

    $locatorPath = Resolve-Path (Join-Path (Get-NintendoSdkRootPath) "Integrate\CommandLineTools\LocateVsInstallation.exe")
    $vsPath = & $locatorPath 2> $null
    if ($LastExitCode)
    {
        throw New-Object "System.IO.FileNotFoundException" "MSBuild 15 not found."
    }

    $msbuildRoot = Join-Path $vsPath "MSBuild\15.0"

    return Get-TestedFilePath "MSBuild 15" $msbuildRoot
}
Export-ModuleMember -Function Get-MSBuild15RootPath

<#
    .SYNOPSIS
        Returns the csi.exe path.
        ex) "C:\Program Files (x86)\Microsoft Visual Studio\2017\Professional\MSBuild\15.0\Bin\Roslyn\csi.exe"

    .OUTPUTS
        String instance that contains path information.
#>
function Get-CsiPath
{
    $path = Join-Path (Get-MSBuild15RootPath) "Bin\Roslyn\csi.exe"
    return Get-TestedFilePath "csi.exe" $path
}
Export-ModuleMember -Function Get-CsiPath

<#
    .SYNOPSIS
        Returns the TTPMacro path

    .OUTPUTS
        String instance that contains path information.
#>
function Get-TTPMacroPath
{
    trap [System.Management.Automation.ActionPreferenceStopException]
    {
        $ex      = $_.Exception
        $message = "Tera Term Pro not installed"
        throw New-Object "System.IO.FileNotFoundException" ($message, $ex)
    }
    $key  = "HKLM:\SOFTWARE\Wow6432Node\Microsoft\Windows\CurrentVersion"
    $key += "\Uninstall\Tera Term_is1"
    $prop = Get-ItemProperty $key -ErrorAction Stop
    $path = $prop.InstallLocation + "ttpmacro.exe"
    return Get-TestedFilePath "ttpmacro" $path
}
Export-ModuleMember -Function Get-TTPMacroPath

<#
    .SYNOPSIS
        Returns the NintendoSDK root path

    .OUTPUTS
        String instance that contains path information.
#>
function Get-NintendoSdkRootPath
{
    if ( $null -ne $script:nintendoSdkRootPath )
    {
        return $script:nintendoSdkRootPath
    }

    $thisModulePath = $script:MyInvocation.MyCommand.Path
    $dirName    = [System.IO.Path]::GetDirectoryName($thisModulePath)
    while ( $dirName )
    {
        if ( Test-Path "${dirName}\NintendoSdkRootMark" )
        {
            $script:nintendoSdkRootPath = $dirName
            return $script:nintendoSdkRootPath
        }
        $dirName = [System.IO.Path]::GetDirectoryName($dirName)
    }

    $message = "NintendoSdkRootMark not found"
    throw New-Object "System.IO.FileNotFoundException" $message
}
Export-ModuleMember -Function Get-NintendoSdkRootPath

<#
    .SYNOPSIS
        Returns the NintendoSDK Integrate root path

    .OUTPUTS
        String instance that contains path information.
#>
function Get-NintendoSdkIntegrateRootPath
{
    $path = "$(Get-NintendoSdkRootPath)\Integrate"
    return Get-TestedDirectoryPath "Integrate" $path
}
Export-ModuleMember -Function Get-NintendoSdkIntegrateRootPath

<#
    .SYNOPSIS
        Returns the NintendoSDK command line tools root path

    .OUTPUTS
        String instance that contains path information.
#>
function Get-NintendoSdkCommandLineToolsRootPath
{
    $path = "$(Get-NintendoSdkIntegrateRootPath)\CommandLineTools"
    return Get-TestedDirectoryPath "CommandLineTools" $path
}
Export-ModuleMember -Function Get-NintendoSdkCommandLineToolsRootPath

<#
    .SYNOPSIS
        Returns the NintendoSDK artifacts root path

    .OUTPUTS
        String instance that contains path information.
#>
function Get-NintendoSdkArtifactsRootPath
{
    $path = "$(Get-NintendoSdkRootPath)\Integrate\Artifacts"
    [void][System.IO.Directory]::CreateDirectory($path)
    return $path
}
Export-ModuleMember -Function Get-NintendoSdkArtifactsRootPath

<#
    .SYNOPSIS
        Returns the NintendoSDK Integrate output root path

    .OUTPUTS
        String instance that contains path information.
#>
function Get-NintendoSdkIntegrateOutputRootPath
{
    return "$(Get-NintendoSdkIntegrateRootPath)\Outputs"
}
Export-ModuleMember -Function Get-NintendoSdkIntegrateOutputRootPath

<#
    .SYNOPSIS
        Returns the NintendoSDK documents root path

    .OUTPUTS
        String instance that contains path information.
 #>
function Get-NintendoSdkDocumentsRootPath()
{
    $path = "$(Get-NintendoSdkRootPath)\Documents"
    return Get-TestedDirectoryPath "Documents" $path
}
Export-ModuleMember -Function Get-NintendoSdkDocumentsRootPath

<#
    .SYNOPSIS
        Returns the Tera Term Pro resources root path

    .OUTPUTS
        String instance that contains path information.
 #>
function Get-TeraTermProResourcesRootPath
{
    $path = "$(Get-NintendoSdkIntegrateRootPath)\Resources"
    [void](Get-TestedDirectoryPath "Resources" $path)
    $path = "${path}\Scripts"
    [void](Get-TestedDirectoryPath "Scripts" $path)
    $path = "${path}\TeraTermPro"
    return Get-TestedDirectoryPath "TeraTermPro" $path
}
Export-ModuleMember -Function Get-TeraTermProConfigurationsRootPath

<#
    .SYNOPSIS
        Returns the Tera Term Pro configurations root path

    .OUTPUTS
        String instance that contains path information.
 #>
function Get-TeraTermProConfigurationsRootPath
{
    $path = "$(Get-TeraTermProResourcesRootPath)\Configurations"
    return Get-TestedDirectoryPath "Configurations" $path
}
Export-ModuleMember -Function Get-TeraTermProConfigurationsRootPath

<#
    .SYNOPSIS
        Returns the Tera Term Pro macros root path

    .OUTPUTS
        String instance that contains path information.
 #>
function Get-TeraTermProMacrosRootPath
{
    $path = "$(Get-TeraTermProResourcesRootPath)\Macros"
    return Get-TestedDirectoryPath "Macros" $path
}
Export-ModuleMember -Function Get-TeraTermProMacrosRootPath

<#
    .SYNOPSIS
        Returns the nact solution file path

    .OUTPUTS
        String instance that contains path information.
#>
function Get-NactSolutionFilePath
{
    $path = "$(Get-NintendoSdkRootPath)\Externals\nact\nact2.sln"
    return Get-TestedFilePath "nact solution file" $path
}
Export-ModuleMember -Function Get-NactSolutionFilePath

<#
    .SYNOPSIS
        Returns the nact path

    .OUTPUTS
        String instance that contains path information.
#>
function Get-NactPath
{
    $path = "$(Get-NintendoSdkCommandLineToolsRootPath)\nact.exe"
    return Get-TestedFilePath "nact" $path
}
Export-ModuleMember -Function Get-NactPath

<#
    .SYNOPSIS
        Returns the TestRunner path

    .OUTPUTS
        String instance that contains path information.
#>
function Get-TestRunnerPath
{
    $path = "$(Get-NintendoSdkCommandLineToolsRootPath)\TestRunner.exe"
    return Get-TestedFilePath "TestRunner" $path
}
Export-ModuleMember -Function Get-TestRunnerPath

<#
    .SYNOPSIS
        Returns the MakeApiReferenceLink path

    .OUTPUTS
        String instance that contains path information.
#>
function Get-MakeApiReferenceLinkPath
{
    $outRoot = "$(Get-NintendoSdkIntegrateOutputRootPath)"
    $prjRel  = "Tools\MakeApiReferenceLink\MakeApiReferenceLink"
    $binPath = "${outRoot}\AnyCPU\${prjRel}\Release\MakeApiReferenceLink.exe"
    return Get-TestedFilePath "MakeApiReferenceLink" $binPath
}
Export-ModuleMember -Function Get-MakeApiReferenceLinkPath

<#
    .SYNOPSIS
        Returns the YamlConverter path

    .OUTPUTS
        String instance that contains path information.
#>
function Get-YamlConverterPath
{
    $path = "$(Get-NintendoSdkCommandLineToolsRootPath)\YamlConverter.exe"
    return Get-TestedFilePath "YamlConverter" $path
}
Export-ModuleMember -Function Get-YamlConverterPath

<#
    .SYNOPSIS
        Returns the FailyWrites path

    .OUTPUTS
        String instance that contains path information.
#>
function Get-FairlyWritesCuiPath()
{
    $path = "$(Get-NintendoSdkRootPath)\Externals\documenttools\FairlyWrites\FairlyWritesCUI.exe"
    return Get-TestedFilePath "FailyWrites" $path
}
Export-ModuleMember -Function Get-FairlyWritesCuiPath

<#
    .SYNOPSIS
        Returns the FairlyLinguist path

    .OUTPUTS
        String instance that contains path information.
#>
function Get-FairlyLinguistCuiPath()
{
    $path = "$(Get-NintendoSdkRootPath)\Externals\documenttools\FairlyLinguistCui\FairlyLinguistCUI.exe"
    return Get-TestedFilePath "FairlyLinguist" $path
}
Export-ModuleMember -Function Get-FairlyLinguistCuiPath

<#
    .SYNOPSIS
        Returns the CreateSearchData path

    .OUTPUTS
        String instance that contains path information.
#>
function Get-CreateSearchDataPath()
{
    $path = "$(Get-NintendoSdkRootPath)\Externals\documenttools\FairlyLinguistCui\CreateSearchData\CreateSearchData.exe"
    return Get-TestedFilePath "CreateSearchData" $path
}
Export-ModuleMember -Function Get-CreateSearchDataPath

<#
    .SYNOPSIS
        Returns the 7-Zip path

    .OUTPUTS
        String instance that contains path information.
#>
function Get-7ZipPath()
{
    $path = "$(Get-NintendoSdkRootPath)\Externals\Binaries\7-Zip\7z.exe"
    return Get-TestedFilePath "7-Zip" $path
}
Export-ModuleMember -Function Get-7ZipPath

<#
    .SYNOPSIS
        Returns the NuGet path

    .OUTPUTS
        String instance that contains path information.
#>
function Get-NuGetPath()
{
    $path = "$(Get-NintendoSdkRootPath)\Externals\NuGet.Client\4.1.0\NuGet.exe"
    return Get-TestedFilePath "NuGet" $path
}
Export-ModuleMember -Function Get-NuGetPath

<#
    .SYNOPSIS
        Returns the origin path of the Siglo NuGet feed, that is, the place where packages should be uploaded.

    .OUTPUTS
        String instance that contains path information.
#>
function Get-SigloNuGetFeedOrigin()
{
    $path = "$(Get-NintendoSdkRootPath)\Externals\NuGetFeed"
    return Get-TestedFilePath "SigloNuGetFeed" $path
}
Export-ModuleMember -Function Get-SigloNuGetFeedOrigin

<#
    .SYNOPSIS
        Returns MSBuild TargetFrameworkMonikerAssemblyAttribute files path

    .OUTPUTS
        String array instance that contains path information.
#>
function Get-TargetFrameworkAttributeFiles()
{
    $cppDirectory = Get-TestedDirectoryPath `
        "VcTargetFrameworkAttributeFiles" `
        "$(Get-NintendoSdkIntegrateRootPath)\Build\Vc\TargetFrameworkAttributeFiles"
    $csDirectory  = Get-TestedDirectoryPath `
        "VcsharpTargetFrameworkAttributeFiles" `
        "$(Get-NintendoSdkIntegrateRootPath)\Build\Vcsharp\TargetFrameworkAttributeFiles"

    return (Get-Item "$cppDirectory\*.cpp") + (Get-Item "$csDirectory\*.cs")
}
Export-ModuleMember -Function Get-TargetFrameworkAttributeFiles

<#
    .SYNOPSIS
        Returns Oasis Installer file (.msi) path

    .OUTPUTS
        String array instance that contains path information.
#>
function Get-OasisInstallerPath
{
    $name = "OasisInstaller"
    $directoryPath = "$(Get-NintendoSdkRootPath)\Externals\Oasis\Installer"

    foreach ( $file in Get-ChildItem($directoryPath) )
    {
        if ($file.Name.StartsWith("TargetManager."))
        {
            $filename = $file.Name
            return Get-TestedFilePath $name "${directoryPath}\${filename}"
            #return "${directoryPath}\${filename}"
        }
    }

    $message = "${name} not found"
    throw New-Object "System.IO.FileNotFoundException" $message
}
Export-ModuleMember -Function Get-OasisInstallerPath

<#
    .SYNOPSIS
        Returns the Siglo shared directory path

    .OUTPUTS
        String instance that contains path information.
#>
function Get-SigloSharedDirectoryPath
{
    $path = "\\devdatasrv2.ncl.nintendo.co.jp\share-siglo$"
    return Get-TestedDirectoryPath "share-siglo$" $path
}
Export-ModuleMember -Function Get-SigloSharedDirectoryPath

<#
    .SYNOPSIS
        Returns the ForAll root path

    .OUTPUTS
        String instance that contains path information.
#>
function Get-ForAllRootPath
{
    $path = "$(Get-SigloSharedDirectoryPath)\ForAll"
    return Get-TestedDirectoryPath "ForAll" $path
}
Export-ModuleMember -Function Get-ForAllRootPath

<#
    .SYNOPSIS
        Returns the NclDevelopers root path

    .OUTPUTS
        String instance that contains path information.
#>
function Get-NclDevelopersRootPath
{
    $path = "$(Get-SigloSharedDirectoryPath)\NclDevelopers"
    return Get-TestedDirectoryPath "NclDevelopers" $path
}
Export-ModuleMember -Function Get-NclDevelopersRootPath

<#
    .SYNOPSIS
        Returns the CiAgents root path

    .OUTPUTS
        String instance that contains path information.
#>
function Get-CiAgentsRootPath
{
    $path = "$(Get-ForAllRootPath)\CiAgents"
    return Get-TestedDirectoryPath "CiAgents" $path
}
Export-ModuleMember -Function Get-CiAgentsRootPath