﻿<#
    Copyright (C)Nintendo All rights reserved.

    These coded instructions, statements, and computer programs contain proprietary
    information of Nintendo and/or its licensed developers and are protected by
    national and international copyright laws. They may not be disclosed to third
    parties or copied or duplicated in any form, in whole or in part, without the
    prior written consent of Nintendo.

    The content herein is highly confidential and should be handled accordingly.
#>

<#
    .SYNOPSIS
        HostBridge Module

    .DESCRIPTION
        This file defines functions for HostBridge operations
#>

$moduleFilePath = $MyInvocation.MyCommand.Path
$modulePath     = [System.IO.Path]::GetDirectoryName($moduleFilePath)
$moduleRootPath = [System.IO.Path]::GetDirectoryName($modulePath)

Import-Module "${moduleRootPath}\Path"
Import-Module "${moduleRootPath}\Utility"

$GPIOControllerCUI = "$(Get-NintendoSdkRootPath)/Externals/HostBridge/tools/HostBridgeController/Resources/GPIOControllerCUI/GPIOControllerCUI.exe"
$ControlTargetPath = "$(Get-NintendoSdkRootPath)/Tools/CommandLineTools/ControlTarget.exe"
$ControlTargetPrivatePath = "$(Get-NintendoSdkRootPath)/Tools/CommandLineTools/ControlTargetPrivate.exe"
$HostBridgeTelnetCommandPath = "$(Get-NintendoSdkRootPath)/Externals/HostBridge/tools/HostBridgeTelnetCommand/HostBridgeTelnetCommand.exe"
$HostBridgeLogDirectory = "$(Get-NintendoSdkRootPath)/Integrate/Outputs/HostBridge"
$HostBridgeLogPath = "$HostBridgeLogDirectory/HostBridge_log.txt"
$HostBridgeLogAppendPath = "$HostBridgeLogDirectory/HostBridge_log_append.txt"
$UpdateHostBridgePath = "$(Get-NintendoSdkRootPath)/Tools/CommandLineTools/UpdateHostBridge.exe"

<#
    .SYNOPSIS
        Get target addresses
#>
function Get-TargetAddresses
{
    param
    (
        [string]
        # Regular expression for IP address
        $AddressPattern = '.*',

        [int]
        # The number of required targets
        $NecessaryNumber = 1
    )

    $retryCount = 10

    $entries = @()

    for ($i = 0; $i -lt $retryCount; ++$i)
    {
        if (0 -ne $i)
        {
            [Threading.Thread]::Sleep(100)

            $entries = @()
        }

        Invoke-Expression "& `"$ControlTargetPath`" detect-target" | % {
            if ($_ -cmatch "\b\d+\.\d+\.\d+\.\d+\b")
            {
                $entry = $matches[0]

                if ($entry -cmatch $AddressPattern)
                {
                    $entries += $entry
                }
            }
        }

        if (0 -ne $LastExitCode)
        {
            throw "Target enumeration failed. ExitCode = $LastExitCode"
        }

        if ($NecessaryNumber -le $entries.Count)
        {
            return ,$entries
        }
    }

    $message = "Failed to find $NecessaryNumber target{0}. "

    $message = if (1 -eq $NecessaryNumber)
    {
        $message -f ""
    }
    else
    {
        $message -f "s"
    }

    $message += switch ($entries.Count)
    {
        0       { "No target found." }
        1       { "1 target found: " + $entries[0] }
        default { "${_} targets found: " + ($entries -join ";") }
    }

    throw $message
}

Export-ModuleMember -Function Get-TargetAddresses

<#
    .SYNOPSIS
        Get a sigle target address
#>
function Get-SigleTargetAddress
{
    param
    (
        [string]
        # The host IP address pattern as RegularExpression
        $AddressPattern = '.*'
    )
    
    # If the AddressPattern parameter was a single IP address and not a regular expression,
    # add an anchor to it so that we scan for the exact address.
    if([bool]($AddressPattern -as [ipaddress]))
    {
        return $AddressPattern
    }
    else
    {
        $AddressPattern = "^$AddressPattern`$"
    }

    $retry = 1
    while ($retry -le 10) {
        $EnumerateOutput = Invoke-Expression "& `"$ControlTargetPath`" detect-target"
        if(![String]::IsNullOrEmpty($EnumerateOutput))
        {
            $TargetCount = (
                Write-Output $EnumerateOutput |
                Where-Object { $_ -match ".*\s.*" } | 
                ForEach-Object { $_  -split('\s+') } | 
                Where-Object { $_ -match "[0-9]+\.[0-9]+\.[0-9]+\.[0-9]+" } |
                Where-Object { $_ -match $AddressPattern } |
                Tee-Object -Variable TargetAddress |
                Measure-Object
            ).Count
            if($TargetCount -ne 0)
            {
                break
            }
        }
        [System.Threading.Thread]::Sleep(100)
        $retry++
    }

    if($LastExitCode -ne 0)
    {
        throw "Target enumeration failed. ExitCode = $LastExitCode"
    }

    if($TargetCount -ne 1)
    {
        Write-Output $EnumerateOutput
        throw "Failed to find single target. AddressPattern = $AddressPattern. NumTarget = $TargetCount. Discovered Targets = `{ $TargetAddress `}"
    }

    return $TargetAddress
}
Export-ModuleMember -Function Get-SigleTargetAddress

<#
    .SYNOPSIS
        Check existing the HostBridge.
#>
function Test-ExistHostBridge
{
    param
    (
        [Parameter(Mandatory=$true)]
        [string]
        $HostBridgeAddress = '.*'
    )

    $EnumerateOutput = Invoke-Expression "& `"$ControlTargetPath`" detect-target"

    if([String]::IsNullOrEmpty($EnumerateOutput))
    {
        return $FALSE;
    }

    $TargetCount = (
            Write-Output $EnumerateOutput |
            Where-Object { $_ -match ".*\s.*" } | 
            ForEach-Object { $_  -split('\s+') } | 
            Where-Object { $_ -match "[0-9]+\.[0-9]+\.[0-9]+\.[0-9]+" } |
            Where-Object { $_ -match $HostBridgeAddress } |
            Measure-Object
        ).Count

    if($TargetCount -eq 1)
    {
        return $TRUE;
    }
    else
    {
        return $FALSE;
    }
}
Export-ModuleMember -Function Test-ExistHostBridge

<#
    .SYNOPSIS
        Wait until find the HostBridge
#>
function Wait-HostBridge
{
    param
    (
        [Parameter(Mandatory=$true)]
        [String]
        $HostBridgeAddress = '.*',

        [Int]
        $RetryCount = 20
    )

    for($i=1; $i -le $RetryCount; $i++)
    {
        $Exist = Test-ExistHostBridge -HostBridgeAddress $HostBridgeAddress

        if($Exist)
        {
            return;
        }
        else
        {
            Write-Output "Wait HostBridge. ($i/$RetryCount)";
        }

        Start-Sleep -s 1
    }

    Write-Host "Failed to find the target. HostBridgeAddress = $HostBridgeAddress"
    Write-Host "The target may be out of the subnet so keep processing ..."
}
Export-ModuleMember -Function Wait-HostBridge

<#
    .SYNOPSIS
        Reboot the target HostBridge
#>
function Restart-HostBridge
{
    param
    (
        [string]
        # The HostBridge IP address
        $HostBridgeAddress
    )

    Invoke-Expression "& `"$GPIOControllerCUI`" --reboot_hb --ip $HostBridgeAddress"

    if($LastExitCode -ne 0)
    {
        throw "HostBridge restart failed. ExitCode = $LastExitCode"
    }
}
Export-ModuleMember -Function Restart-HostBridge

<#
    .SYNOPSIS
        Restart the target by HostBridge
#>
function Restart-TargetByHostBridge
{
    param
    (
        [string]
        # The HostBridge IP address
        $HostBridgeAddress
    )

    Invoke-Expression "& `"$GPIOControllerCUI`" --reset --ip $HostBridgeAddress"

    if($LastExitCode -ne 0)
    {
        throw "Target restart failed. ExitCode = $LastExitCode"
    }
}
Export-ModuleMember -Function Restart-TargetByHostBridge

<#
    .SYNOPSIS
        Start the target by HostBridge
#>
function Start-TargetByHostBridge
{
    param
    (
        [string]
        # The HostBridge IP address
        $HostBridgeAddress
    )

    Invoke-Expression "& `"$GPIOControllerCUI`" --poweron --ip $HostBridgeAddress"

    if($LastExitCode -ne 0)
    {
        throw "Target stop failed. ExitCode = $LastExitCode"
    }
}
Export-ModuleMember -Function Start-TargetByHostBridge

<#
    .SYNOPSIS
        Stop the target by HostBridge
#>
function Stop-TargetByHostBridge
{
    param
    (
        [string]
        # The HostBridge IP address
        $HostBridgeAddress
    )

    Invoke-Expression "& `"$GPIOControllerCUI`" --poweroff --ip $HostBridgeAddress"

    if($LastExitCode -ne 0)
    {
        throw "Target stop failed. ExitCode = $LastExitCode"
    }
}
Export-ModuleMember -Function Stop-TargetByHostBridge

<#
    .SYNOPSIS
        Get the target HostBridge firmware version
#>
function Get-HostBridgeFirmwareVersion
{
    param
    (
        [Parameter(Mandatory=$true)]
        [string]
        # The HostBridge IP address
        $HostBridgeAddress
    )

    $Version = Invoke-Expression "& `"$GPIOControllerCUI`" --get_hb_build_version --ip $HostBridgeAddress"

    if($LastExitCode -ne 0)
    {
        throw "Get HostBridge firmware version failed. ExitCode = $LastExitCode"
    }
    else
    {
        return $Version
    }
}
Export-ModuleMember -Function Get-HostBridgeFirmwareVersion

<#
    .SYNOPSIS
        Get serial number from a HostBridge
#>
function Get-SerialNumber
{
    param
    (
        [Parameter(Mandatory=$true)]
        [string]
        # The HostBridge IP address
        $HostBridgeAddress
    )

    $retry = 1
    while ($retry -le 5) {
        $EnumerateOutput = Invoke-Expression "& `"$ControlTargetPath`" detect-target --csv"
        if(![String]::IsNullOrEmpty($EnumerateOutput))
        {
            $SerialNumber = Write-Output $EnumerateOutput |
                Where-Object { $_ -match "`"$HostBridgeAddress`"" } | 
                ForEach-Object { $_.Split(',')[3].Trim("`"") }
            if (![String]::IsNullOrEmpty($SerialNumber))
            {
                if($SerialNumber -eq "Unknown")
                {
                    return $null
                }
                return $SerialNumber
            }
        }

        [System.Threading.Thread]::Sleep(100)
        Write-Host "Failed to get serial number. Retry... ($retry/5)"
        $retry++
    }

    if ($LastExitCode -ne 0)
    {
        throw "Get serial number failed. ExitCode = $LastExitCode"
    }
}
Export-ModuleMember -Function Get-SerialNumber

function Update-HostBridgeByConsole
{
    param
    (
        [Parameter(Mandatory=$true)]
        [string]$HostBridgeAddress

    )

    $TargetSerial = Get-SerialNumber -HostBridgeAddress $HostBridgeAddress

    # ターゲット側FWとホストブリッジFWの組み合わせにによって初回のアップデートが失敗することがあるのでリトライする
    $retry = 1
    while ($retry -le 2) {
        Invoke-Expression "& `"$UpdateHostBridgePath`" update --target $HostBridgeAddress --image $TargetImagePath --verbose"
        if($LastExitCode -eq 0)
        {
            Start-Sleep -s 20
            $HostBridgeAddress = Get-TargetAddressFromSerial -Target $TargetSerial
            Wait-HostBridge $HostBridgeAddress
            return
        }
        $retry++
    }
}

function Update-HostBridgeRecoveryFirmware
{
    param
    (
        [Parameter(Mandatory=$true)]
        [string]$HostBridgeAddress

    )

    if($UpdateRecoveryFirmwareRequired)
    {
        Update-HostBridgeRecoveryFirmwareCore -HostBridgeAddress $HostBridgeAddress
    }
}

function Update-HostBridgeRecoveryFirmwareCore
{
    param
    (
        [Parameter(Mandatory=$true)]
        [string]$HostBridgeAddress

    )

    $output = Invoke-Expression "& `"$UpdateHostBridgePath`" required --target $HostBridgeAddress --verbose"
    if (![String]::IsNullOrEmpty($output))
    {
        $required = $output.ToString().Split("=")[1]
        if([String]::Equals($required, "True"))
        {
            $RecoveryImagePath =  "$(Get-NintendoSdkRootPath)/Externals/HostBridge/images/recovery.nhf"
            Write-Host "Updating HostBridge recovery firmware."
            Invoke-Expression "& `"$UpdateHostBridgePath`" update --target $HostBridgeAddress --recovery --image $RecoveryImagePath --verbose"
        }
    }
}

<#
    .SYNOPSIS
        Update the target HostBridge
#>
function Update-HostBridge
{
    param
    (
        [Parameter(Mandatory=$true)]
        [string]
        # The HostBridge IP address
        $HostBridgeAddress,

        [string]
        $FirmwareImagePath,

        [ref]$Updated
    )

    $Updated.Value = $true

    $UpdateRecoveryFirmwareRequired = $false
    $HwVersion = Get-HostBridgeHardwareVersion -HostBridgeAddress $HostBridgeAddress
    # SDEV1.6-PreMP2を更新対象とする
    if($HwVersion -lt 7)
    {
        $UpdateRecoveryFirmwareRequired = $true
    }

    if([String]::IsNullOrEmpty($FirmwareImagePath))
    {
        $TargetImagePath =  "$(Get-NintendoSdkRootPath)/Externals/HostBridge/images/package.nhf"
        $Version = Get-HostBridgeFirmwareVersion -HostBridgeAddress $HostBridgeAddress

        switch -casesensitive ( $Version )
        {
            "3.11 Sun, 15 Apr 2018 21:36:53 -0700" # Official build 1642
            {
                Write-Output "The firmware of the specified hostbridge is already latest."
                Update-HostBridgeRecoveryFirmware -HostBridgeAddress $HostBridgeAddress
                $Updated.Value = $false
                exit 0
            }
            default
            {
                Write-Output "Version `"$Version`" is not expected."
                Write-Output "The hostbridge firmware will be updated to the latest."
            }
        }
    }
    else
    {
        $TargetImagePath = $FirmwareImagePath
    }

    # Test existing files
    Test-EnsureExistPath $TargetImagePath

    Update-HostBridgeByConsole -HostBridgeAddress $HostBridgeAddress
    if($LastExitCode -ne 0)
    {
        Update-HostBridgeByConsole -HostBridgeAddress $HostBridgeAddress

        if($LastExitCode -ne 0)
        {
            throw "HostBridge update failed. ExitCode = $LastExitCode"
        }
        Update-HostBridgeRecoveryFirmware -HostBridgeAddress $HostBridgeAddress
    }
    else
    {
        Update-HostBridgeRecoveryFirmware -HostBridgeAddress $HostBridgeAddress
    }
}
Export-ModuleMember -Function Update-HostBridge

<#
    .SYNOPSIS
        Get the target HostBridge hardware version
#>
function Get-HostBridgeHardwareVersion
{
    param
    (
        [Parameter(Mandatory=$true)]
        [string]
        # The HostBridge IP address
        $HostBridgeAddress
    )

    $Version = Invoke-Expression "& `"$ControlTargetPrivatePath`" get-hostbridge-version --target-ip $HostBridgeAddress"

    if($LastExitCode -ne 0)
    {
        throw "Get HostBridge hardware version failed. ExitCode = $LastExitCode"
    }
    else
    {
        return Write-Output $Version |
               Where-Object { $_ -match "###HardwareVersion=\d+" } |
               ForEach-Object { $_.Split("=")[1].Trim() }
    }
}
Export-ModuleMember -Function Get-HostBridgeHardwareVersion

<#
    .SYNOPSIS
        Start logging HostBridge syslog message
#>
function Start-SaveHostBridgeLog
{
    param
    (
        [Parameter(Mandatory=$true)]
        [string]
        # The HostBridge IP address
        $HostBridgeAddress,

        [switch]
        $rebooted = $false
    )

    if([string]::IsNullOrEmpty($env:ARTIFACT_PATHS))
    {
        return
    }
    
    $SerialNumber = Get-SerialNumber -HostBridgeAddress $HostBridgeAddress
    $HostBridgeLogPathWithSerialNumber = $HostBridgeLogPath -replace"_log.", ("_" + $SerialNumber + "_log.")
    $HostBridgeLogAppendPathWithSerialNumber = $HostBridgeLogAppendPath -replace "_log.", ("_" + $SerialNumber + "_log.")

    New-Item $HostBridgeLogDirectory -ItemType Directory -ErrorAction SilentlyContinue
    Stop-SaveHostBridgeLog -HostBridgeAddress $HostBridgeAddress
    $date = Get-Date -UFormat "%Y-%m-%d %H:%M:%S"
    Invoke-Expression "& `"$HostBridgeTelnetCommandPath`" --ip $HostBridgeAddress -c `"date '$date'`""
    
    if(Test-Path $HostBridgeLogPathWithSerialNumber)
    {
        Start-Process $HostBridgeTelnetCommandPath -ArgumentList "--ip $HostBridgeAddress -c `"tail -F /var/log/messages -n +0`" -p -f" -RedirectStandardOutput $HostBridgeLogAppendPathWithSerialNumber -NoNewWindow

    }
    else
    {
        Start-Process $HostBridgeTelnetCommandPath -ArgumentList "--ip $HostBridgeAddress -c `"tail -F /var/log/messages -n +0`" -p -f" -RedirectStandardOutput $HostBridgeLogPathWithSerialNumber -NoNewWindow
    }
}
Export-ModuleMember -Function Start-SaveHostBridgeLog


<#
    .SYNOPSIS
        Stop logging HostBridge syslog message
#>
function Stop-SaveHostBridgeLog
{
    param
    (
        [Parameter(Mandatory=$true)]
        [string]
        # The HostBridge IP address
        $HostBridgeAddress
    )

    if([string]::IsNullOrEmpty($env:ARTIFACT_PATHS))
    {
        return
    }

    $telnetCommands = Get-WmiObject Win32_Process -Filter "name = 'HostBridgeTelnetCommand.exe'"
    foreach($i in $telnetCommands)
    {
        if($null -ne $i -and $null -ne $i.CommandLine -and $i.CommandLine.Contains($HostBridgeAddress))
        {
            Stop-Process -Id $i.ProcessId -ErrorAction SilentlyContinue
            break
        }
    }

    $SerialNumber = Get-SerialNumber -HostBridgeAddress $HostBridgeAddress
    $HostBridgeLogPathWithSerialNumber = $HostBridgeLogPath -replace"_log.", ("_" + $SerialNumber + "_log.")
    $HostBridgeLogAppendPathWithSerialNumber = $HostBridgeLogAppendPath -replace "_log.", ("_" + $SerialNumber + "_log.")

    if(Test-Path $HostBridgeLogAppendPathWithSerialNumber)
    {
        Get-Content $HostBridgeLogAppendPathWithSerialNumber | Out-File -Append -FilePath $HostBridgeLogPathWithSerialNumber -Encoding ascii
        Remove-Item $HostBridgeLogAppendPathWithSerialNumber
    }
}
Export-ModuleMember -Function Stop-SaveHostBridgeLog

function Get-TargetAddressFromSerial
{
    param
    (
        [string]
        # The target serial number
        $Target
    )
    
    $retry = 1
    while ($retry -le 10) {
        $EnumerateOutput = Invoke-Expression "& `"$ControlTargetPath`" detect-target"
        if(![String]::IsNullOrEmpty($EnumerateOutput))
        {
            $TargetCount = (
                Write-Output $EnumerateOutput |
                Where-Object { $_ -match $Target } | 
                ForEach-Object { $_  -split('\s+') } | 
                Where-Object { $_ -match "[0-9]+\.[0-9]+\.[0-9]+\.[0-9]+" } |
                Tee-Object -Variable TargetAddress |
                Measure-Object
            ).Count
            if($TargetCount -ne 0)
            {
                break
            }
        }
        [System.Threading.Thread]::Sleep(100)
        $retry++
    }

    if($LastExitCode -ne 0)
    {
        throw "Target enumeration failed. ExitCode = $LastExitCode"
    }

    if($TargetCount -ne 1)
    {
        Write-Output $EnumerateOutput
        throw "Failed to find $Target."
    }

    Write-Host "$Target : $TargetAddress"

    return $TargetAddress
}
Export-ModuleMember -Function Get-TargetAddressFromSerial
