﻿<#
    Copyright (C)Nintendo All rights reserved.

    These coded instructions, statements, and computer programs contain proprietary
    information of Nintendo and/or its licensed developers and are protected by
    national and international copyright laws. They may not be disclosed to third
    parties or copied or duplicated in any form, in whole or in part, without the
    prior written consent of Nintendo.

    The content herein is highly confidential and should be handled accordingly.
#>

<#
    .SYNOPSIS
        BootConfig Module

    .DESCRIPTION
        This file defines functions for BootConfig operations
#>

$moduleFilePath = $MyInvocation.MyCommand.Path
$modulePath     = [System.IO.Path]::GetDirectoryName($moduleFilePath)
$moduleRootPath = [System.IO.Path]::GetDirectoryName($modulePath)

Import-Module "${moduleRootPath}\Path"
Import-Module "${moduleRootPath}\HostBridge"

$BootConfigRoot = "$(Get-NintendoSdkRootPath)/Externals/NxBootConfigs"

$ControlTargetPath = "$(Get-NintendoSdkRootPath)/Tools/CommandLineTools/ControlTarget.exe"
$WriteBootConfigScript = "$(Get-NintendoSdkRootPath)/Integrate/Scripts/NX/Recovery/Write-BootConfig.ps1"
$WriteSerialNumberProgram = "$(Get-NintendoSdkRootPath)/Tools/CommandLineTools/WriteSerialNumberToHostBridge.exe"

<#
    .SYNOPSIS
        Write the proper boot-config to the target
#>
function Write-ProperBootConfig
{
    param
    (
        [Parameter(Mandatory=$true)]
        [string]
        # The host IP address
        $HostBridgeAddress,

        [switch]
        $All
    )

    # Serial number を取得
    $SerialNumber = Get-SerialNumber $HostBridgeAddress

    if ([String]::IsNullOrEmpty($SerialNumber))
    {
        #
        # !!! ここは TargetManager が接続できる状態でないと動かない
        # !!! 例えば Dummy BootConfig + Unsigned Kernel が書かれているとダメ
        #
        # 現時点では CI での Signed kernel テスト後、元に戻す際用
        #
        Write-Host "Write serial number to the target."

        # Serial number が HostBridge に書き込まれていない場合は書き込む
        Invoke-Expression "& `"$WriteSerialNumberProgram`" --target $HostBridgeAddress --write-name"

        # Serial number を再取得
        $SerialNumber = Get-SerialNumber $HostBridgeAddress
        Write-Host "Done. Serial number=$SerialNumber."
    }

    # BootConfig のパスを設定
    $bcFilePath = "$BootConfigRoot\bc-" + $SerialNumber + ".bin"

    if (!(Test-Path $bcFilePath))
    {
        throw "BootConfig file not found. Serial=${SerialNumber}. Path=${bcFilePath}"
    }
    Invoke-Expression "& `"$WriteBootConfigScript`" -AddressPattern $HostBridgeAddress -All:`$$All -BootConfig `"$bcFilePath`""
}
Export-ModuleMember -Function Write-ProperBootConfig


<#
    .SYNOPSIS
        Write the dummy boot-config to the target
#>
function Write-DummyBootConfig
{
    param
    (
        [Parameter(Mandatory=$true)]
        [string]
        # The host IP address
        $HostBridgeAddress,

        [switch]
        $All
    )

    $bcFilePath = "$BootConfigRoot\bc-dummy.bin"

    if (!(Test-Path $bcFilePath))
    {
        throw "Dummy BootConfig file not found. Path=${bcFilePath}"
    }
    Invoke-Expression "& `"$WriteBootConfigScript`" -AddressPattern $HostBridgeAddress -All:`$$All -BootConfig `"$bcFilePath`""
}
Export-ModuleMember -Function Write-DummyBootConfig
