﻿<#
    Copyright (C)Nintendo. All rights reserved.

    These coded instructions, statements, and computer programs contain proprietary
    information of Nintendo and/or its licensed developers and are protected by
    national and international copyright laws. They may not be disclosed to third
    parties or copied or duplicated in any form, in whole or in part, without the
    prior written consent of Nintendo.
 
    The content herein is highly confidential and should be handled accordingly.
#>

<#
    .SYNOPSIS
    ローカライズパッケージを作成します。

    .DESCRIPTION
    ローカライズパッケージを作成します。

    .INPUTS
    なし。

    .OUTPUTS
    なし。
#>
[CmdletBinding()]
Param(
    # 翻訳用リポジトリのルートを指定します。
    [parameter(Mandatory = $true)]
    [string] $TranslationRoot,

    # パス変換リストファイルを指定します。
    [parameter(Mandatory = $true)]
    [string[]] $PathTranslateRule,

    [string[]] $Langs = ("ja", "en"),

    # 除外するパッケージのパターンを指定します。
    [string] $IgnorePackagesPattern = '^Documents'
)

$sigloRoot = (Join-Path $MyInvocation.MyCommand.Path ../../../.. | Resolve-Path).Path

Import-Module $sigloRoot/Integrate/Scripts/Modules/Localize
Import-Module $sigloRoot/Integrate/Outputs/AnyCPU/Tools/YamlCmdlet/YamlCmdlet/Release/Yaml.psd1

$refPackageRoot = Join-Path $sigloRoot Integrate/Outputs/Packages
# $outPackageRoot = Join-Path $sigloRoot Integrate/Outputs/LocPkg
$outPackageRoot = Join-Path $sigloRoot Integrate/Outputs/LP
# $intPackageRoot = Join-Path $sigloRoot Integrate/Intermediates/LocPkg
$intPackageRoot = Join-Path $sigloRoot Integrate/Outputs/IP

$zipNames = @(
    Get-ZipName $refPackageRoot |
    Where-Object { $_ -notmatch $IgnorePackagesPattern }
)

#
# zip の解凍
#
$intPackageRoot | Join-ManyPath $Langs |
ForEach-Object {
    $outPath = $_
    $zipNames |
    ForEach-Object {
        Expand-Zip (Join-Path $refPackageRoot ($_ + ".zip")) (Join-Path $outPath $_)
    }
}


#
# 英語用サテライトアセンブリのコピー
#
$sigloToolsRoot = Join-Path $sigloRoot Tools
$sateLang = $Langs -eq "en"

if ($sateLang) {
    $intPackageRoot | Join-ManyPath ($sateLang) $zipNames NintendoSDK/Tools |
    Select-ExistPath |
    ForEach-Object {
        Get-ChildItem $_ -Include *.exe,*.dll -Recurse |
        Get-RelativePath $_ |
        Get-ResourceDllName |
        ForEach-Object { Join-Path $sigloToolsRoot $_ } |
        Select-ExistPath |
        Get-RelativePath $sigloToolsRoot |
        Copy-TreeFile $sigloToolsRoot $_
    }
}


#
# ローカライズされたファイルで存在するファイルを置換
#
$pathTranslateList = Import-TranslateList $PathTranslateRule

$intPackageRoot | Join-ManyPath $Langs |
ForEach-Object {
    $lang = Split-Path $_ -Leaf
    $_ | Join-ManyPath $zipNames |
    ForEach-Object {
        Set-ExistFile $TranslationRoot/$lang/ApiReference $_/NintendoSDK -TranslateList $pathTranslateList
    }
}


#
# スマートデバイス向けコードのトリミング処理
#
$trimConfig = Import-Yaml $sigloRoot/Integrate/Scripts/Trim-SmartDeviceCode/config.yml

$trimTargets =
    $sigloRoot | Join-ManyPath $trimConfig.TargetDirectory |
    Get-ChildItem -Recurse -Include $trimConfig.TargetExtension |
    Get-RelativePath $sigloRoot/Programs/Iris

$intPackageRoot | Join-ManyPath $Langs $zipNames NintendoSDK $trimTargets |
Select-ExistPath |
& $sigloRoot/Integrate/Scripts/Trim-SmartDeviceCode/Trim-SmartDeviceCodeCommon.ps1


#
# Zarfの作成
#

# .zarf.yml のパッケージ名と.zarf.yml のパスとの辞書を作成
$zarfDefDic = @{}
Get-ChildItem $sigloRoot/Integrate/Packages/ZarfDefinitions -Recurse -Include *.zarf.yml |
ForEach-Object {
    $config = Import-Yaml $_.FullName
    if ($?) { # Yaml が読み込めたときのみ
        $zarfDefDic[$config.Zarf.Name] = $_.FullName
    }
}

$Langs |
ForEach-Object {
    $lang = $_
    $_ | Join-ManyPath $zipNames |
    ForEach-Object {
        $zipDir = Join-Path $intPackageRoot $_
        $zipPath = (Join-Path $outPackageRoot $_) + '.zip'

        # zip の作成
        New-Zip $zipPath $zipDir

        # Zarf の作成
        $zarfName = Split-Path $_ -Leaf
        if ($lang -eq "en") {
            $zarfName = $zarfName + "-en"
        }
        $zarfDir = Join-Path $outPackageRoot $_ | Split-Path -Parent
        New-Zarf $zipPath $zarfDefDic[$zarfName] $zarfDir $zipDir
    }
}
