﻿<#
    Copyright (C)Nintendo All rights reserved.

    These coded instructions, statements, and computer programs contain proprietary
    information of Nintendo and/or its licensed developers and are protected by
    national and international copyright laws. They may not be disclosed to third
    parties or copied or duplicated in any form, in whole or in part, without the
    prior written consent of Nintendo.

    The content herein is highly confidential and should be handled accordingly.
#>

<#
    .SYNOPSIS
        Invoke DockIn and DockOut Test on Target

    .DESCRIPTION
        Run DockIn and DockOut Test.
#>

param
(
    [string]
    # The path to test_detail.xml
    $TestDetailXmlPath,

    [int]
    # Number of repetitions.(-1 means repeating forever.)
    $RepeatCount = 3,

    [int]
    # Time of execution interval at repeat.(millisecond)
    $TimeOfRepeatInterval = 1000,

    [int]
    # Number of seconds before it times out.(second)
    $Timeout = 30,

    [int]
    # Time to first dock in/out.(second)
    $TimeOfDockInOutStart = 5,

    [int]
    $TimeOfSleepBeforeTest = 10,

    [string]
    # Test path
    $Path,

    [string]
    # Target Name
    $TargetName,

    [string]
    # Target Address
    $TargetAddress,

    [string]
    # Target Address Pattern
    $AddressPattern = $env:TARGET_ADDRESS_PATTERN,

    [string]
    # The platform name
    $Platform = "NX64",

    [string]
    # The build type
    $BuildType = "Develop"
)

$scriptPath          = $MyInvocation.MyCommand.Path
$scriptDirectoryPath = [System.IO.Path]::GetDirectoryName($scriptPath)

Import-Module "${scriptDirectoryPath}\Modules\Error"
Import-Module "${scriptDirectoryPath}\Modules\HostBridge"
Import-Module "${scriptDirectoryPath}\Modules\Path"
Import-Module "${scriptDirectoryPath}\Modules\Utility"
Import-Module "${scriptDirectoryPath}\Modules\SystemTestUtility"

trap [Exception]
{
    Write-ErrorRecord $_
    exit 1
}

if ([string]::IsNullOrEmpty($AddressPattern))
{
    # If AddressPattern is not specified and env:TARGET_ADDRESS_PATTERN is not
    # difined
    $AddressPattern = "^169\.*"
}

$VsPlatform = switch ($Platform)
{
    "NX32"      { "NX-NXFP2-a32" }
    "NXFP2-a32" { "NX-NXFP2-a32" }
    "NX64"      { "NX-NXFP2-a64" }
    "NXFP2-a64" { "NX-NXFP2-a64" }
    default     { $_ }
}

if ([string]::IsNullOrEmpty($Path))
{
    $Path = Get-NintendoSdkRootPath
    $Path = [IO.Path]::Combine($Path, "Integrate\Outputs")
    $Path = [IO.Path]::Combine($Path, "Scripts\Demo1Internal")
    $Path = [IO.Path]::Combine($Path, "Demo1Internal.$VsPlatform.$BuildType.nsp")
}

if ([string]::IsNullOrEmpty($TargetName))
{
    $TargetName = Get-SigleTargetAddress -AddressPattern $AddressPattern
    $TargetAddress = $TargetName
}

if (-not $TestDetailXmlPath)
{
    Write-ErrorMessage "TestDetailXmlPath option not specified"
    exit 1
}

# TORIAEZU :
#  dll が読めないので、暫定対応として、Outputs 以下の ControlTarget を実行
$ControlTarget = Get-NintendoSdkRootPath
$ControlTarget = [IO.Path]::Combine($ControlTarget, "Programs\Chris\Outputs")
$ControlTarget = [IO.Path]::Combine($ControlTarget, "x86")
$ControlTarget = [IO.Path]::Combine($ControlTarget, "Tools\RunnerTools")
$ControlTarget = [IO.Path]::Combine($ControlTarget, "ControlTargetPrivate")
$ControlTarget = [IO.Path]::Combine($ControlTarget, "Release")
$ControlTarget = [IO.Path]::Combine($ControlTarget, "ControlTargetPrivate.exe")

$RunOnTarget = Get-NintendoSdkRootPath
$RunOnTarget = [IO.Path]::Combine($RunOnTarget, "Tools")
$RunOnTarget = [IO.Path]::Combine($RunOnTarget, "CommandLineTools")
$RunOnTarget = [IO.Path]::Combine($RunOnTarget, "RunOnTarget.exe")

$ReadHbLog = Get-NintendoSdkRootPath
$ReadHbLog = [IO.Path]::Combine($ReadHbLog, "Tools")
$ReadHbLog = [IO.Path]::Combine($ReadHbLog, "CommandLineTools")
$ReadHbLog = [IO.Path]::Combine($ReadHbLog, "ReadHostBridgeLog.exe")

$logDir = $scriptDirectoryPath
$logDir = $logDir.Replace(
    (Get-NintendoSdkIntegrateRootPath),
    (Get-NintendoSdkIntegrateOutputRootPath))
$logDir = [IO.Path]::Combine(
    $logDir, [IO.Path]::GetFileNameWithoutExtension($scriptPath))
$logDir = [IO.Path]::Combine($logDir, (Get-Date -Format "yyyyMMddHHmmss"))
[void](New-Item $logDir -ItemType Directory -ErrorAction "Stop")

$logFile = [IO.Path]::GetFileNameWithoutExtension($Path)
$logFile = [IO.Path]::Combine($logDir, ($logFile + ".log"))

Write-Host "Start ${Path}"
Write-Host "logFile: ${logFile}"

# FromHost 起動など、spsm が psc のモジュール登録を待たない場合は、
# すぐにテストを走らせるとモジュールが死ぬため待つ
Start-Sleep -Seconds $TimeOfSleepBeforeTest

# 全てのテストでのドックイン/ドックアウトにかかる時間格納用の配列
$script:timeArrayAll = @()

# テスト対象プログラムの実行
function Invoke-TestProgram()
{
    # テストを起動 (実行直後に RunOnTarget は終了)
    $command  = ("&", "`"$RunOnTarget`"", "`"$Path`"")
    $command += ("--target", $TargetName)
    $command += ("--verbose", "--no-wait", "--failure-timeout", "$Timeout")
    $command += ("--suppress-auto-kill")
    $command = $command -join " "
    Invoke-Expression $command | Set-Content -Passthru $logFile | Write-Host
    $TestResult = $LastExitCode
    if (0 -ne $TestResult)
    {
        Invoke-Expression "& `"$ControlTarget`" reset --target $TargetName"

        throw ("Failed to run " + [IO.Path]::GetFileName($Path))
    }

    return $TestResult
}

# ドックイン/アウトの繰り返しテストの実行
function Invoke-DockTest($type,$repeatCount,$repeatTime)
{
    # 単一のテストケースでのドックイン/アウトにかかる時間格納用の配列
    $timeArray = @()

    # ドックイン/アウトされたことを判定する為のログ
    $DockInOutLog = "nn::oe::MessagePerformanceModeChanged"
    # 初期化が成功したことを判定する為のログ
    $Demo1InitSuccessLog = "Demo1 Initialize Success"
    # テスト開始時の操作を選択
    if( $type -eq "DockInTest" )
    {
        # 最初の状態
        $BeforeDockState = "disable-cradle"
        # 切り替え後の状態
        $AfterDockState = "enable-cradle"
    }
    elseif( $type -eq "DockOutTest" )
    {
        # 最初の状態
        $BeforeDockState = "enable-cradle"
        # 切り替え後の状態
        $AfterDockState = "disable-cradle"
    }
    else
    {
        throw ("Invalid argument.")
    }

    # プログラム実行前のドック状態を設定
    $command  = ("&", "`"$ControlTarget`"", $BeforeDockState)
    $command += ("--target", $TargetName)
    $command = $command -join " "
    Invoke-Expression $command | Set-Content -Passthru $logFile | Write-Host
    if (0 -ne $LastExitCode)
    {
        Invoke-Expression "& `"$ControlTarget`" reset --target $TargetName"

        throw ("Failed to run " + [IO.Path]::GetFileName($ControlTarget))
    }

    # テストプログラム実行 (してすぐ RunOnTarget は終了)
    $TestResult = Invoke-TestProgram
    if (0 -ne $TestResult)
    {
        Invoke-Expression "& `"$ControlTarget`" reset --target $TargetName"

        throw ("Failed to run " + [IO.Path]::GetFileName($Path))
    }

    # HostBridge Log を監視してDemo1の初期化時のエラーを監視する
    # 初期化成功ログが出力されている場合は成功
    # アサート関連ログが出力されている場合にエラー
    $command  = ("&", "`"$ReadHbLog`"")
    $command += ("--address", $TargetAddress)
    $command += ("--verbose", "--failure-timeout", "$Timeout")
    $command += ("--pattern-success-exit", "`"$Demo1InitSuccessLog`"")
    $command += ("--pattern-failure-exit", " Assert")
    $command += ("--pattern-failure-exit", "`"Assertion Failure:`"")
    $command += ("--pattern-failure-exit", "`"Break\(\) called`"")
    $command = $command -join " "
    Invoke-Expression $command | Set-Content -Passthru $logFile | Write-Host
    $TestResult = $LastExitCode

    # ドックイン/アウト操作を開始する時間(デフォルト5秒)待つ
    Start-Sleep -Seconds $TimeOfDockInOutStart

    $count = $repeatCount
    while($count -ne 0)
    {
        # バックグラウンドジョブでドックイン/アウト処理を実行
        $list = ($ControlTarget, $TargetName, $AfterDockState)
        $job = Start-Job -ArgumentList $list -ScriptBlock {
            param($path, $name, $cmd)
            $subCommand  = ("&", "`"$path`"", $cmd)
            $subCommand += ("--target", $name)
            $subCommand = $subCommand -join " "
            Invoke-Expression $subCommand
            $LastExitCode
        }

        # ドックイン/アウトにかかる時間の計測開始
        $localWatch = New-Object "System.Diagnostics.StopWatch"
        $localWatch.Start()

        # HostBridge Log を監視してドックイン/アウトの完了を待つ
        #  下記のいずれかが発生した場合にエラー
        #  - タイムアウト時間(デフォルト30秒)内にドックイン/アウト完了のログが出力されない
        #  - アサート関連ログが出力されている
        $command  = ("&", "`"$ReadHbLog`"")
        $command += ("--address", $TargetAddress)
        $command += ("--verbose", "--failure-timeout", "$Timeout")
        $command += ("--pattern-success-exit", "`"$DockInOutLog`"")
        $command += ("--pattern-failure-exit", "`"Assertion Failure:`"")
        $command += ("--pattern-failure-exit", "`"Break\(\) called`"")
        $command = $command -join " "
        Invoke-Expression $command | Set-Content -Passthru $logFile | Write-Host
        $TestResult = $LastExitCode

        # 時間計測終了
        $localWatch.Stop()

        if (0 -ne $TestResult)
        {
            # テストが失敗したら終了
            break
        }

        # バックグラウンドジョブの完了待ち
        Wait-Job -job $job > $null

        # バックグラウンドジョブ結果取得
        $BgJobResult = Receive-Job -job $job
        if (0 -ne $BgJobResult)
        {
            Invoke-Expression "& `"$ControlTarget`" reset --target $TargetName"

            throw ("Failed to run " + [IO.Path]::GetFileName($ControlTarget))
        }

        # 明示的にバックグラウンドジョブを停止／削除する
        Stop-Job $job > $null
        Remove-Job $job > $null

        # ドックイン/アウト検出時間を配列にpush
        $timeArray += $localWatch.Elapsed.Milliseconds

        # 次のドックイン/アウト操作をトグル
        if( $AfterDockState -eq "enable-cradle" )
        {
            $AfterDockState = "disable-cradle"
        }
        elseif( $AfterDockState -eq "disable-cradle" )
        {
            $AfterDockState = "enable-cradle"
        }
        else
        {
            throw ("Invalid Value.")
        }

        # 次のドックイン/アウト実施時間まで待つ
        Start-Sleep -m $repeatTime

        # 繰り返し回数をデクリメント
        if($count -gt 0)
        {
            $count -= 1
        }
    }

    # ターゲットをリセットする
    $command  = ("&", "`"$ControlTarget`"", "reset")
    $command += ("--target", $TargetName)
    $command = $command -join " "
    Invoke-Expression $command | Set-Content -Passthru $logFile | Write-Host
    if (0 -ne $LastExitCode)
    {
        throw ("Failed to run " + [IO.Path]::GetFileName($ControlTarget))
    }
    Start-Sleep -Seconds $TimeOfSleepBeforeTest

    $script:timeArrayAll += $timeArray
    # 平均時間をテスト実行時間とする
    $max,$min,$avg,$unbiasedVariance = Get-Statistic $timeArray
    if (0 -ne $TestResult)
    {
        return $false ,$avg
    }
    return $true ,$avg
}

# テストケース
$tests = New-Object "System.Collections.Specialized.OrderedDictionary"

# ドックアウト状態→ドッグイン切り替えでDemo1が正常に動作するか
$tests["DockInTest"] =
{
    return Invoke-DockTest "DockInTest" $RepeatCount $TimeOfRepeatInterval
}

# ドックイン状態→ドッグアウト切り替えでDemo1が正常に動作するか
$tests["DockOutTest"] =
{
    return Invoke-DockTest "DockOutTest" $RepeatCount $TimeOfRepeatInterval
}

# テストスイート名
$classname = "DockInOutSuite"

# テスト実行
$xml = Start-TestSuite $classname $tests

# テスト結果出力
$xml.Save($TestDetailXmlPath)

# fail発生回数取得
$failures = $xml.testsuites.failures

# ドックイン/ドックアウト実行回数が２回以上の場合、ドックイン/ドックアウトにかかる時間の最大値・最小値・平均値・不偏分散をログ出力
if ($script:timeArrayAll.Length -ge 2)
{
    $max,$min,$avg,$unbiasedVariance = Get-Statistic $script:timeArrayAll

    # 結果の出力
    Write-Host "DockIn/Out measurement time,"
    Write-Host "DockIn/Out total num = " $script:timeArrayAll.Length
    Write-Host "max                  = " $max ms
    Write-Host "min                  = " $min ms
    Write-Host "average              = " $avg ms
    Write-Host "unbiased variance    = " $unbiasedVariance
}

Write-Host "Done."

exit $failures
