﻿<#
    Copyright (C)Nintendo All rights reserved.

    These coded instructions, statements, and computer programs contain proprietary
    information of Nintendo and/or its licensed developers and are protected by
    national and international copyright laws. They may not be disclosed to third
    parties or copied or duplicated in any form, in whole or in part, without the
    prior written consent of Nintendo.

    The content herein is highly confidential and should be handled accordingly.
#>

<#
    .SYNOPSIS
        Check uniqueness of definited files in pakages

    .SYNTAX
        Check-DuplicatedFile [-Exclude <String[]>]

    .PARAMETER
        [-Exclude <String[]>] Check packages excluding the regex strings
#>
Param([String[]] $exclude=@())


# パッケージ定義 yml のパスから、使用しているバンドルを配列で返す
function Get-BundleArrayInPackageYml
{
    param(
        [parameter(mandatory=$true)]
        [String] $pakageYmlPath
    )
    # パッケージ定義かどうかチェック
    if($pakageYmlPath -notmatch ".*package\.yml$")
    {
        Write-Host "Error: The argment of Get-BundleArrayInPackageYml() is worng. ( $pakageYmlPath )"
        exit 1
    }

    # xml に変換してバンドル部分を取り出す
    $yml = Get-Content $pakageYmlPath -Raw -Encoding Default
    $xml = ((ConvertFrom-Yaml $yml).DocumentElement.ChildNodes)
    return $xml[0].Bundles._ | ?{ $_ -ne $null }
}

# バンドル定義 yml のパスから、使用しているファイルを配列で返す
function Get-FileArrayInBundleYml {
    param(
        [parameter(mandatory=$true)]
        [String] $bundleYmlPath
    )
    # バンドル定義かどうかチェック
    if($bundleYmlPath -notmatch ".*bundle\.yml$")
    {
        Write-Host "Error: The argment of Get-FileArrayInBundleYml() is worng. ( $bundleYmlPath )"
        exit 1
    }

    # xml に変換してファイル部分を取り出す
    $bundleRawYml = Get-Content $bundleYmlPath -Raw
    $bundleXml    = ((ConvertFrom-Yaml $bundleRawYml).DocumentElement.ChildNodes)
    return $bundleXml[0].Files._ | ?{ $_ -ne $null }
}

# 複数のバンドルに同じファイルが入っていないかを確認(バンドル単位),引数は sort -Unique 済みの配列
function Confirm-FileUniquenessInBundles {
    param(
        [parameter(mandatory=$true)]
        $bundleArray
    )
    $isUnique = $true

    $fileHash = @{}

    # 全バンドル内のファイルを追加しつつ、1つのパッケージで使われているファイルとその他のパッケージで使われているファイルに重複が無いかを検査
    $bundleArray | %{
        $bundleYmlPath = "$sigloroot/$_"
        $files = Get-FileArrayInBundleYml $bundleYmlPath

        # バンドル内に同一ファイルが複数記述されていた場合に警告
        $duplicatedFilesInSameBundle = $files | ?{ ($files -eq $_).count -ge 2 } | sort -Unique
        $duplicatedFilesInSameBundle | %{ Write-Warning "$_ is multiple described in $bundleYmlPath"}

        $files = $files | sort -Unique

        # ここまでで作成したハッシュテーブルとの重複をチェック
        $duplicatedFiles = $files | ?{ $fileHash.ContainsKey($_) }
        $uniqueFiles = $files | ?{ -not $fileHash.ContainsKey($_) }

        if($duplicatedFiles.Length -ne 0)
        {
            $isUnique = $false
            # 重複したファイルはエラー表示
            $duplicatedFiles | %{
                $errorMessage = "Error: Duplicated with $_ `nin $bundleYmlPath `nand $($fileHash[$_]).`n"
                Write-Host $errorMessage.Replace($sigloroot, "")
            }
        }

        # ファイル名とバンドルのパスをハッシュテーブルに追加
        $uniqueFiles | %{ $fileHash.Add($_, "$bundleYmlPath") }
    }
    return $isUnique
}

# subjectPackageYmlArray 内のファイルを comparisonPackageYmlArray 内で使用していないかチェック
function Confirm-FileUniquenessInTwoPackages
{
    param(
        [parameter(mandatory=$true)]
        $subjectPackageYmlArray,
        [parameter(mandatory=$true)]
        $comparisonPackageYmlArray
    )

    $isUnique = $true

    # 使用される全ファイル名とそれを使用しているバンドル名(パッケージ名)を保持するハッシュテーブル
    $subjectHash = Get-FileHashFromPackages $subjectPackageYmlArray
    $comparisonHash = Get-FileHashFromPackages $comparisonPackageYmlArray

    # 重複したファイルはエラー表示
    $subjectHash.Keys | ?{ $comparisonHash.ContainsKey($_) } | %{
        Write-Host "Error: Duplicated with $_ `nin $($subjectHash[$_]) `nand $($comparisonHash[$_])).`n"
        $isUnique = $false
    }
    return $isUnique
}

# 引数のパッケージ定義で使用するファイル名とそのバンドル名(パッケージ名)を保持するハッシュテーブルを取得
function Get-FileHashFromPackages
{
    param(
        [parameter(mandatory=$true)]
        $packageArray
    )
    # パッケージ定義で使用するファイル名とそのバンドル名(パッケージ名)を保持するハッシュテーブル
    $filesHash = @{}

    $packageArray | %{
        $ymlPath = $_

        # パッケージ定義内の全バンドル内のファイルをハッシュテーブルに追加していく
        Get-BundleArrayInPackageYml $ymlPath | %{
            $bundleYmlPath = "$sigloroot/$_"
            Get-FileArrayInBundleYml $bundleYmlPath | %{

                # 同じキーを含んでいるか
                if($filesHash.ContainsKey($_))
                {
                    # キーから得られたリストに追加
                    $filesHash[$_].Add("$bundleYmlPath($ymlPath)")
                }
                else
                {
                    # キーに対して、値をリストで追加する(1ファイルは複数のパッケージ内のバンドルから参照されるため)
                    $bundleYmlPathList = New-Object 'System.Collections.Generic.List[String]'
                    $bundleYmlPathList.Add("$bundleYmlPath($ymlPath)")
                    $filesHash.Add($_, $bundleYmlPathList)
                }
            }
        }
    }
    return $filesHash
}

# 文字列に特定の文字が含まれていないことを確認
function Test-RegexMatch
{
    param(
        [parameter(mandatory=$true)] [String] $TargetString,
        [parameter(mandatory=$true)] $RegexStrings
    )
    $RegexStrings | %{ if( $TargetString -cmatch $_) { return $true }}
    return $false
}

#### main ####

# パスを定義
$scriptPath          = $MyInvocation.MyCommand.Path
$scriptDirectoryPath = [System.IO.Path]::GetDirectoryName($scriptPath)
Import-Module "${scriptDirectoryPath}/Modules/Yaml"
Import-Module "${scriptDirectoryPath}/Modules/Error"
$integrateDirectoryPath = [System.IO.Path]::GetDirectoryName($scriptDirectoryPath)
$sigloroot = [System.IO.Path]::GetDirectoryName($integrateDirectoryPath)

$packageDefinitionsPath = "${integrateDirectoryPath}/Packages/PackageDefinitions"

$nxPrivatePackagePath = "${packageDefinitionsPath}/NX/Private"
$nxPublicPackagePath = "${packageDefinitionsPath}/NX/Public"
$genericPrivatePackagePath = "${packageDefinitionsPath}/Generic/Private"
$genericPublicPackagePath = "${packageDefinitionsPath}/Generic/Public"

# 重複発見時の終了用
$isUnique = $true

try
{
    #############################################################
    # バンドル単位で重複チェック
    #############################################################
    Write-Host "Start checking uniqueness of files in bundles."
    # プライベートパッケージ内の全バンドルを配列に格納
    $bundleArray = Get-ChildItem $nxPrivatePackagePath, $genericPrivatePackagePath *.yml | ?{ -not (Test-RegexMatch $_ $exclude) } | %{ Get-BundleArrayInPackageYml $_.FullName }
    $bundleArray = $bundleArray | sort -Unique

    # 複数のバンドルに同じファイルが入っていないかを確認
    if(Confirm-FileUniquenessInBundles $bundleArray)
    {
        Write-Host "OK. Ensure the file uniqueness in private bundles."
    }
    else
    {
        Write-Host "Find the duplication of the file in private bundles."
        $isUnique = $false
    }

    #############################################################
    # 公開パッケージとプライベートパッケージをzip単位で重複チェック
    #############################################################
    Write-Host "Start checking uniqueness of files in private packages for public packages"

    # 各パッケージ定義を取得
    $publicPackageYmlArray = Get-ChildItem $genericPublicPackagePath,$nxPublicPackagePath *.yml | ?{ -not (Test-RegexMatch $_ $exclude) }  | %{ $_.FullName }
    $privatePagkageYmlArray = Get-ChildItem $genericPrivatePackagePath,$nxPrivatePackagePath *.yml | ?{ -not (Test-RegexMatch $_ $exclude) } | %{ $_.FullName }

    # 2つのパッケージ定義配列間で、使用しているファイルが重複していないことを確認（zip 単位）
    if(Confirm-FileUniquenessInTwoPackages $publicPackageYmlArray $privatePagkageYmlArray)
    {
        Write-Host "OK. Ensure that the files in public packages aren't used in private packages."
    }
    else
    {
        Write-Host "Find the duplication of the file in public package with one of private package."
        $isUnique = $false
    }

    if(-not $isUnique)
    {
        exit 1
    }
}
catch [Exception] {
    Write-ErrorRecord $_
    exit 1
}
