﻿// 文字コード:UTF-8
/// @file
#pragma once

#include <cstdarg>
#include <lib/logger/ILoggerHandler.hpp>
#include <lib/logger/LogLevel.hpp>
#include <string>

//------------------------------------------------------------------------------
namespace lib {
namespace logger {

/// @addtogroup LIB-Logger
//@{
/// ログを扱うクラス。
class Logger
    : public ILoggerHandler
{
public:
    /// @name 定数
    //@{
    static const LogLevel::EnumType DefaultLevel; ///< デフォルトのレベル。
    //@}

    /// @name デフォルトのハンドラ
    //@{
    static ILoggerHandler& DefaultLoggerHandler(); ///< デフォルトのログハンドラ。
    //@}

    /// @name コンストラクタとデストラクタ
    //@{
    /// コンストラクタ。
    explicit Logger(
        const LogLevel::EnumType aLevel = DefaultLevel,
        ILoggerHandler& aHandler = DefaultLoggerHandler()
        );
    //@}

    /// @name レベル
    //@{
    LogLevel::EnumType level() const; ///< レベルの取得。
    void setLevel(LogLevel::EnumType aLevel); ///< レベルの設定。
    //@}

    /// @name ログの書き込み
    /// @details 第1引数のaLevelはログのレベル。level()未満のレベルならログは記録されない。
    //@{
    void writef(LogLevel::EnumType aLevel, const char* aFormat, ...); ///< printf構文で書き込み。
    /// va_listで書き込み。va_end()は内部で呼びません。
    void write(
        LogLevel::EnumType aLevel,
        const char* aFormat,
        va_list aArgList
        );
    void write(LogLevel::EnumType aLevel, const char* aMessage); ///< const char*で書き込み。
    void write(LogLevel::EnumType aLevel, const ::std::string& aStr); ///< std::stringで書き込み。
    //@}

    /// @name ハンドラ
    //@{
    void setHandler(ILoggerHandler* aHandler); ///< ILoggerHandlerの設定。
    //@}

    /// @name ILoggerHandlerの実装
    //@{
    /// @details Level条件をクリアしたら、登録されているILoggerHandlerを呼ぶ。
    virtual void onLogWritten(
        const Logger& aSender,
        LogLevel::EnumType aLevel,
        const char* aMessage,
        TimeType aTime
        );
    //@}

private:
    void writeCore(
        LogLevel::EnumType aLevel,
        const char* aMessage,
        const TimeType aTime
        );

    ILoggerHandler* mHandler;
    LogLevel::EnumType mLevel;
};

}} // namespace
// EOF
