﻿// 文字コード:UTF-8
/// @file
#include <lib/logger/Logger.hpp>

#include <cstdio>
#include "lib/CString.hpp"
#include "lib/debug/Assert.hpp"
#include "lib/logger/LogLevelUtil.hpp"
#include "lib/logger/SimpleLoggerHandler.hpp"

//------------------------------------------------------------------------------
namespace lib {
namespace logger {

const LogLevel::EnumType Logger::DefaultLevel = LogLevel::Info;

//------------------------------------------------------------------------------
ILoggerHandler& Logger::DefaultLoggerHandler()
{
    return SimpleLoggerHandler::Instance();
}

//------------------------------------------------------------------------------
Logger::Logger(
    const LogLevel::EnumType aLevel,
    ILoggerHandler& aHandler
    )
: mHandler(&aHandler)
, mLevel()
{
    setLevel(aLevel);
}

//------------------------------------------------------------------------------
LogLevel::EnumType Logger::level() const
{
    return mLevel;
}

//------------------------------------------------------------------------------
void Logger::setLevel(const LogLevel::EnumType aLevel)
{
    SYS_ASSERT(LogLevelUtil::IsValid(aLevel));
    mLevel = aLevel;
}

//------------------------------------------------------------------------------
void Logger::writef(const LogLevel::EnumType aLevel, const char* aFormat, ...)
{
    // 準備
    va_list arg_list;
    va_start(arg_list, aFormat);

    // 呼び出し
    write(aLevel, aFormat, arg_list);

    // 後片付け
    va_end(arg_list);
} //lint !e438

//------------------------------------------------------------------------------
void Logger::write(
    const LogLevel::EnumType aLevel,
    const char* aFormat,
    va_list aArgList
    )
{
    SYS_ASSERT(LogLevelUtil::IsValid(aLevel));
    if (!LogLevelUtil::IsReceiveLog(aLevel, mLevel)) {
        // レベル不足。
        return;
    }

    // メモ：
    // 内部でnewを呼ぶとヒープを汚すため固定値バッファで実装します。
    const int BufferSize = 512; // 1行分を想定。512あれば足りるでしょう。
    char buffer[BufferSize] = {};
    SYS_ASSERT_POINTER(aFormat);
    CString::FormatVAsPossible(buffer, BufferSize, aFormat, aArgList);
    write(aLevel, buffer);
}

//------------------------------------------------------------------------------
void Logger::write(const LogLevel::EnumType aLevel, const char* aMessage)
{
    TimeType timeValue = TimeType();
    writeCore(aLevel, aMessage, timeValue);
}

//------------------------------------------------------------------------------
void Logger::write(const LogLevel::EnumType aLevel, const ::std::string& aStr)
{
    write(aLevel , aStr.c_str());
}

//------------------------------------------------------------------------------
void Logger::setHandler(ILoggerHandler* aHandler)
{
    SYS_ASSERT_POINTER(aHandler);
    mHandler = aHandler;
}

//------------------------------------------------------------------------------
void Logger::onLogWritten(
    const Logger& aSender,
    const LogLevel::EnumType aLevel,
    const char* aMessage,
    const TimeType aTime
    )
{
    LIB_UNUSED(aSender);
    writeCore(aLevel, aMessage, aTime);
}

//------------------------------------------------------------------------------
void Logger::writeCore(
    const LogLevel::EnumType aLevel,
    const char* aMessage,
    const TimeType aTime
    )
{
    SYS_ASSERT(LogLevelUtil::IsValid(aLevel));
    if (!LogLevelUtil::IsReceiveLog(aLevel, mLevel)) {
        // レベル不足。
        return;
    }

    // ハンドラに処理を委譲
    mHandler->onLogWritten(*this, aLevel, aMessage, aTime);
}

}} // namespace
// EOF
