﻿// 文字コード:UTF-8
/// @file
#pragma once

#include <lib/Type.hpp>
#include <lib/logger/Logger.hpp>
#include <lib/ShortString.hpp>
#include <lib/TypeTraitsEnableIf.hpp>
#include <lib/TypeTraitsIsConvertible.hpp>

//------------------------------------------------------------------------------

/// @addtogroup LIB-Debug
//@{
/// @name コンソール出力
//@{
/// printfの代用。LIB_LOG_DEBUGのエイリアス。
#define LIB_PRINTF(...) \
    LIB_LOG_DEBUG(__VA_ARGS__)

/// 関数の名前をprintfで出力する。
#define LIB_DEBUG_PRINT_FUNCTION_NAME() \
    LIB_PRINTF("%s\n", __FUNCTION__)

/// コードの実行位置を出力する。
#define LIB_CHECK_POINT() \
    LIB_PRINTF("LIB_CHECK_POINT(): %s, line %d\n", __FILE__, __LINE__)

/// 変数の簡単表示。
/// varの型は、整数・浮動小数・toShortString()を実装しているクラスのどれか。
#define LIB_PRINT(aVar) \
    LIB_PRINTF("%s\n", LIB_SPRINT(aVar).readPtr())

/// 型がなんであれ、強制的に16進表示する。
/// アドレスらしき整数値を表示する時等に便利。
#define LIB_PRINT_HEX(aVar) \
    LIB_PRINTF("%s\n", LIB_SPRINT_HEX(aVar).readPtr())

#define LIB_PRINT_FLOAT_DETAIL(aFloatVar) \
    ::lib::debug::Print::PrintFloatDetail(aFloatVar)
//@}

/// @name ログコマンド（これらは内部でnewを呼びません。）
///
/// LIB_LOG_DEBUG()を除いて、末尾に改行は自動で挿入されるようだ。
/// また、LIB_LOG_DEBUG()以外は、行頭に[Trace]のようなプリフィックスがつくようだ。
/// (2013/08/20 hiroyuki)
//@{
#define LIB_LOG_DEBUG(...) \
    LIB_LOG_WRITEF(::lib::logger::LogLevel::Debug, __VA_ARGS__)
#define LIB_LOG_TRACE(...) \
    LIB_LOG_WRITEF(::lib::logger::LogLevel::Trace, __VA_ARGS__)
#define LIB_LOG_INFO(...) \
    LIB_LOG_WRITEF(::lib::logger::LogLevel::Info, __VA_ARGS__)
#define LIB_LOG_WARN(...) \
    LIB_LOG_WRITEF(::lib::logger::LogLevel::Warn, __VA_ARGS__)
#define LIB_LOG_ERROR(...) \
    LIB_LOG_WRITEF(::lib::logger::LogLevel::Error, __VA_ARGS__)
//@}

/// @name ::lib::ShortString変換
//@{
/// ShortStringへの変換
#define LIB_SPRINTF(...) \
    ::lib::debug::Print::HelSPrintfImpl(__VA_ARGS__)
/// 変数を名前なしでShortString化。出力例：12345
#define LIB_SPRINT_VAR(aVar) \
    ::lib::debug::Print::HelSPrintVarImpl(aVar)
/// 変数を名前つきでShortString化。出力例：someInt: 12345
#define LIB_SPRINT(aVar) \
    ::lib::debug::Print::HelSPrintImpl(aVar, #aVar)
/// 変数を名前つきの１６進数でShortString化
#define LIB_SPRINT_HEX(aVar) \
    ::lib::debug::Print::HelSPrintfImpl("%s: 0x%x", #aVar, aVar)
//@}

/// @name コンパイル時値出力
//@{
/// コンパイルエラーにより、コンパイル時に数値を無理やり出力するマクロ。
/// @details
/// @code
///  LIB_COMPILER_PRINT_VALUE(SomeConst);
/// error: invalid ... HelCompilerPrintValue::ValueIs[12345]'
/// @endcode
#define LIB_COMPILER_PRINT_VALUE(aValue) \
struct HelCompilerPrintValue \
{ \
    struct ValueIs{}; \
    void dummy() \
    { \
        const int dummy2 = static_cast<ValueIs[aValue]>(dummy2); \
    } \
}

/// コンパイルエラーにより、コンパイル時に型のサイズを無理やり出力するマクロ。
/// @details
/// @code
/// LIB_COMPILER_PRINT_SIZE(SomeType);
/// error: invalid ... HelCompilerPrintSize::SizeIs[16]'
/// @endcode
#define LIB_COMPILER_PRINT_SIZE(aType) \
struct HelCompilerPrintSize \
{ \
    struct SizeIs{}; \
    void dummy() \
    { \
        const int dummy2 = static_cast<SizeIs[sizeof(aType)]>(dummy2);  \
    } \
}
//@}

/// @name 内部実装
//@{
/// Logコマンドの本体（内部実装）
#if defined(LIB_OPT_USE_PRINTF)
    #define LIB_LOG_WRITEF(aLevel, ...) \
        ::lib::debug::Print::Logger().writef(aLevel, __VA_ARGS__)
#else
    #define LIB_LOG_WRITEF(aLevel, ...) do {} while (false)
#endif

//@}
//@}

//------------------------------------------------------------------------------
namespace lib {
namespace debug {

struct Print
{
    /// helのLogger。
    static ::lib::logger::Logger& Logger();

    /// floatの詳細を出力する。
    static void PrintFloatDetail(float aValue);

    /// printf構文をShortStringに変換する関数。
    static ::lib::ShortString HelSPrintfImpl(const char* aFormat, ...);

    /// @name HelSPrintの内部実装たち。
    //@{
    template <typename TType>
        static inline ::lib::ShortString HelSPrintImpl(const TType& aVar,
            const char* aName
            );
    //@}

    /// @name HelSPrintVarImplたち。
    //@{
    template <typename TType>
        static inline ::lib::ShortString HelSPrintVarImpl(
            const TType& aVar
            );

    // テンプレートの特殊化で書こうとしたが、
    // そうすると実装をヘッダに書く必要があり、実装を書くとプリコンパイルできなくなるらしい
    // 特殊化していないテンプレートは実装を書いても大丈夫。不思議だ
    static ::lib::ShortString HelSPrintVarImpl(
        const ::lib::ShortString& aVar
        );

    static ::lib::ShortString HelSPrintVarImpl(char aVar);
    static ::lib::ShortString HelSPrintVarImpl(int8_t aVar);
    static ::lib::ShortString HelSPrintVarImpl(int16_t aVar);
    static ::lib::ShortString HelSPrintVarImpl(int aVar);
    static ::lib::ShortString HelSPrintVarImpl(int64_t aVar);

    static ::lib::ShortString HelSPrintVarImpl(uint8_t aVar);
    static ::lib::ShortString HelSPrintVarImpl(uint16_t aVar);
    static ::lib::ShortString HelSPrintVarImpl(uint32_t aVar);
    static ::lib::ShortString HelSPrintVarImpl(uint64_t aVar);

    static ::lib::ShortString HelSPrintVarImpl(float aVar);
    static ::lib::ShortString HelSPrintVarImpl(double aVar);

    static ::lib::ShortString HelSPrintVarImpl(bool aVar);
    static ::lib::ShortString HelSPrintVarImpl(void* aVar);
    static ::lib::ShortString HelSPrintVarImpl(const void* aVar);
    static ::lib::ShortString HelSPrintVarImpl(const char* aVar);

private:
    /// TTypeがintに変換可能な場合、今回は特にenumの場合にこちらを使いたい。
    template <typename TType>
        static inline ::lib::ShortString HelSPrintVarImplImpl(
            const TType& aVar,
            typename ::lib::TypeTraitsEnableIf<
                ::lib::TypeTraitsIsConvertible<TType, int>::Value>::Type*
                    = nullptr
            );

    /// TTypeがintに変換不可能な場合、つまり一般的な場合のHelSPrintfの実装
    template <typename TType>
        static inline ::lib::ShortString HelSPrintVarImplImpl(
            const TType& aVar,
            typename ::lib::TypeTraitsEnableIf<
                !::lib::TypeTraitsIsConvertible<TType, int>::Value>::Type*
                    = nullptr
            );
    //@}
};

//------------------------------------------------------------------------------
template <typename TType>
    ::lib::ShortString Print::HelSPrintImpl(const TType& aVar,
        const char* aName
        )
{
    const ::lib::ShortString varShortString = HelSPrintVarImpl(aVar);
    return HelSPrintfImpl("%s: %s", aName, varShortString.readPtr());
}

//------------------------------------------------------------------------------
template <typename TType>
    ::lib::ShortString Print::HelSPrintVarImpl(const TType& aVar)
{
    // HelSPrintVarImplImplに処理を委譲する。
    //
    // 以下ややこしいが、このようにわざわざ委譲している理由を述べる。
    // まずそもそもTTypeがenumの場合にはtoShortStringを呼ぼうとせずに
    // %dで出力をしたいという目的がある。(そうしないとそもそもenumの場合に
    // コンパイルエラーになる)
    //
    // その実装にTypeTraitsIsConvertibleを用いているがHelSPrintVarImplで
    // 直接TypeTraitsIsConvertibleを用いた実装をするとTTypeがfloatの場合等に
    // 下で定義されているHelSPrintVarImpl(float)の特殊化にマッチせず、
    // テンプレートの汎用の定義版にもマッチするようで、結果として
    // floatからintへの情報欠落のキャストが発生というワーニングが
    // 発生した。これを回避するため。
    //
    // 一段階設けることで、float等はHelSPrintVarImpl(float)に
    // マッチするようになる。
    // (2014/02/18 hiroyuki)
    return HelSPrintVarImplImpl(aVar);
}

//------------------------------------------------------------------------------
template <typename TType>
    ::lib::ShortString Print::HelSPrintVarImplImpl(const TType& aVar,
        typename ::lib::TypeTraitsEnableIf<
            ::lib::TypeTraitsIsConvertible<TType, int>::Value>::Type*
        )
{
    return HelSPrintfImpl("%d", aVar);
}

//------------------------------------------------------------------------------
template <typename TType>
    ::lib::ShortString Print::HelSPrintVarImplImpl(const TType& aVar,
        typename ::lib::TypeTraitsEnableIf<
            !::lib::TypeTraitsIsConvertible<TType, int>::Value>::Type*
        )
{
    return HelSPrintfImpl("%s", aVar.toShortString().readPtr());
}

}} // namespace
// EOF
