﻿// 文字コード:UTF-8
/// @file
#include "lib/Vector3Pod.hpp"

#include <limits>
#include "lib/Matrix4x3.hpp"
#include "lib/Matrix4x4.hpp"
#include "lib/Vector3.hpp"
#include "lib/Vector4.hpp"

//------------------------------------------------------------------------------
namespace lib {

//------------------------------------------------------------------------------
const Vector3Pod Vector3Pod::Zero()
{
    return Vector3Pod::Create(0.0f, 0.0f, 0.0f);
}

//------------------------------------------------------------------------------
const Vector3Pod Vector3Pod::One()
{
    return Vector3Pod::Create(1.0f, 1.0f, 1.0f);
}

//------------------------------------------------------------------------------
const Vector3Pod Vector3Pod::Min()
{
    const float val = -::std::numeric_limits<float>::infinity();
    return Vector3Pod::Create(val, val, val);
}

//------------------------------------------------------------------------------
const Vector3Pod Vector3Pod::Max()
{
    const float val = ::std::numeric_limits<float>::infinity();
    return Vector3Pod::Create(val, val, val);
}

//------------------------------------------------------------------------------
const Vector3Pod Vector3Pod::UnitX()
{
    return Vector3Pod::Create(1.0f, 0.0f, 0.0f);
}

//------------------------------------------------------------------------------
const Vector3Pod Vector3Pod::UnitY()
{
    return Vector3Pod::Create(0.0f, 1.0f, 0.0f);
}

//------------------------------------------------------------------------------
const Vector3Pod Vector3Pod::UnitZ()
{
    return Vector3Pod::Create(0.0f, 0.0f, 1.0f);
}

//------------------------------------------------------------------------------
const Vector3Pod Vector3Pod::NegUnitX()
{
    return Vector3Pod::Create(-1.0f, 0.0f, 0.0f);
}

//------------------------------------------------------------------------------
const Vector3Pod Vector3Pod::NegUnitY()
{
    return Vector3Pod::Create(0.0f, -1.0f, 0.0f);
}

//------------------------------------------------------------------------------
const Vector3Pod Vector3Pod::NegUnitZ()
{
    return Vector3Pod::Create(0.0f, 0.0f, -1.0f);
}

//------------------------------------------------------------
const Vector2Pod Vector3Pod::toXX() const
{
    return Vector2(x, x);
}

//------------------------------------------------------------
const Vector2Pod Vector3Pod::toXY() const
{
    return Vector2(x, y);
}

//------------------------------------------------------------
const Vector2Pod Vector3Pod::toXZ() const
{
    return Vector2(x, z);
}

//------------------------------------------------------------
const Vector2Pod Vector3Pod::toYX() const
{
    return Vector2(y, x);
}

//------------------------------------------------------------
const Vector2Pod Vector3Pod::toYY() const
{
    return Vector2(y, y);
}

//------------------------------------------------------------
const Vector2Pod Vector3Pod::toYZ() const
{
    return Vector2(y, z);
}

//------------------------------------------------------------
const Vector2Pod Vector3Pod::toZX() const
{
    return Vector2(z, x);
}

//------------------------------------------------------------
const Vector2Pod Vector3Pod::toZY() const
{
    return Vector2(z, y);
}

//------------------------------------------------------------
const Vector2Pod Vector3Pod::toZZ() const
{
    return Vector2(z, z);
}

//------------------------------------------------------------
const Vector3Pod Vector3Pod::toXY0() const
{
    return Vector3(x, y, 0);
}

//------------------------------------------------------------
const Vector3Pod Vector3Pod::toX0Z() const
{
    return Vector3(x, 0, z);
}

//------------------------------------------------------------
const Vector3Pod Vector3Pod::toX00() const
{
    return Vector3(x, 0, 0);
}

//------------------------------------------------------------
const Vector3Pod Vector3Pod::to0YZ() const
{
    return Vector3(0, y, z);
}

//------------------------------------------------------------
const Vector3Pod Vector3Pod::to0Y0() const
{
    return Vector3(0, y, 0);
}

//------------------------------------------------------------
const Vector3Pod Vector3Pod::to00Z() const
{
    return Vector3(0, 0, z);
}

//------------------------------------------------------------
const Vector3Pod Vector3Pod::toXXX() const
{
    return Vector3(x, x, x);
}

//------------------------------------------------------------
const Vector3Pod Vector3Pod::toYYY() const
{
    return Vector3(y, y, y);
}

//------------------------------------------------------------
const Vector3Pod Vector3Pod::toZZZ() const
{
    return Vector3(z, z, z);
}

//------------------------------------------------------------
const Vector4Pod Vector3Pod::toXYZ0() const
{
    return Vector4(x, y, z, 0);
}

//------------------------------------------------------------
const Vector4Pod Vector3Pod::toXXXX() const
{
    return Vector4(x, x, x, x);
}

//------------------------------------------------------------
const Vector4Pod Vector3Pod::toYYYY() const
{
    return Vector4(y, y, y, y);
}

//------------------------------------------------------------
const Vector4Pod Vector3Pod::toZZZZ() const
{
    return Vector4(z, z, z, z);
}

//------------------------------------------------------------------------------
Vector3Pod& Vector3Pod::operator*=(const Matrix4x3& aMtx)
{
    *this = Vector3(
        x * aMtx.at(0, 0) + y * aMtx.at(1, 0) + z * aMtx.at(2, 0) + aMtx.at(3, 0),
        x * aMtx.at(0, 1) + y * aMtx.at(1, 1) + z * aMtx.at(2, 1) + aMtx.at(3, 1),
        x * aMtx.at(0, 2) + y * aMtx.at(1, 2) + z * aMtx.at(2, 2) + aMtx.at(3, 2));
    return *this;
}

//------------------------------------------------------------------------------
Vector3Pod& Vector3Pod::operator*=(const Matrix4x4& aMtx)
{
    *this = Vector3(
        x * aMtx.at(0, 0) + y * aMtx.at(1, 0) + z * aMtx.at(2, 0) + aMtx.at(3, 0),
        x * aMtx.at(0, 1) + y * aMtx.at(1, 1) + z * aMtx.at(2, 1) + aMtx.at(3, 1),
        x * aMtx.at(0, 2) + y * aMtx.at(1, 2) + z * aMtx.at(2, 2) + aMtx.at(3, 2));
    float w = x * aMtx.at(0, 3) + y * aMtx.at(1, 3) + z * aMtx.at(2, 3) + aMtx.at(3, 3);
    *this *= 1.0f / w;
    return *this;
}

//------------------------------------------------------------------------------
const Vector3Pod Vector3Pod::operator*(const Matrix4x3& aRhs) const
{
    Vector3Pod tmp = *this;
    return tmp *= aRhs;
}

//------------------------------------------------------------------------------
const Vector3Pod Vector3Pod::operator*(const Matrix4x4& aRhs) const
{
    Vector3Pod tmp = *this;
    return tmp *= aRhs;
}

//------------------------------------------------------------------------------
const Vector3Pod Vector3Pod::unitCross(const Vector3Pod& aRhs) const
{
    Vector3Pod result = cross(aRhs);
    result.unitAssign();
    return result;
}

//------------------------------------------------------------------------------
const Vector3Pod Vector3Pod::permittedUnitCross(const Vector3Pod& aRhs) const
{
    Vector3Pod result = cross(aRhs);
    float len = result.permittedUnitAssign();
    if (len == 0.0f) {
        return Vector3Pod();
    }
    return result;
}

//------------------------------------------------------------------------------
float Vector3Pod::setLength(const float aTarget)
{
    float length = this->length();

    if (length == 0.0f) {
        *this = Zero();
    } else {
        *this *= aTarget / length;
    }

    return length;
}

//------------------------------------------------------------------------------
bool Vector3Pod::limit(const float aLimit)
{
    float length = this->length();

    if (length > aLimit) {
        *this *= aLimit / length;
        return true;
    }
    return false;
}

//------------------------------------------------------------------------------
float Vector3Pod::unitAssign()
{
    float len = length();
    if (len == 0.0f) {
        SYS_ASSERT_NOT_REACHED();
        return len; // fail safe code
    }
    operator/=(len);
    // 誤差で1.0fより大きな値になった時のための補正。
    if (1.0f <= x) {
        x = 1.0f;
        y = z = 0.0f;
    } else if (x <= -1.0f) {
        x = -1.0f;
        y = z = 0.0f;
    } else if (1.0f <= y) {
        y = 1.0f;
        z = x = 0.0f;
    } else if (y <= -1.0f) {
        y = -1.0f;
        z = x = 0.0f;
    } else if (1.0f <= z) {
        z = 1.0f;
        x = y = 0.0f;
    } else if (z <= -1.0f) {
        z = -1.0f;
        x = y = 0.0f;
    }
    return len;
}

//------------------------------------------------------------------------------
float Vector3Pod::cos(const Vector3Pod& aRhs) const
{
    float denominator = length() * aRhs.length();
    if (denominator == 0.0f) {
        SYS_ASSERT_NOT_REACHED();
        return 0.0f;
    }
    float result = dot(aRhs) / denominator;
    if (1.0f < result) {
        return 1.0f;
    }
    if (result < -1.0f) {
        return -1.0f;
    }
    return result;
}

//------------------------------------------------------------------------------
Vector3Pod Vector3Pod::getVertical(const Vector3Pod& aRhs) const
{
    SYS_ASSERT(!isZeroStrict() && !aRhs.isZeroStrict());
    Vector3Pod result = cross(aRhs);

    // ２ベクトルが平行な時は、Y軸やZ軸と外積をとって何とか見つけ出す。
    if (result.isZeroStrict()) {
        result = cross(UnitY());
        if (result.isZeroStrict()) {
            result = cross(UnitZ());
            SYS_ASSERT(!result.isZeroStrict());
        }
    }

    result.unitAssign();
    return result;
}

//------------------------------------------------------------------------------
void Vector3Pod::rotateDeg(const Vector3Pod& aAxis, const float aDegree)
{
    rotateRad(aAxis, LIB_MATH_DEG_TO_RAD(aDegree));
}

//------------------------------------------------------------------------------
void Vector3Pod::rotateDegVec(const Vector3Pod& aAxis, const float aDegree)
{
    rotateDeg(aAxis, aDegree);
}

//------------------------------------------------------------------------------
void Vector3Pod::rotateDeg(const AxisKind::EnumType aAxisKind, const float aDegree)
{
    rotateRad(aAxisKind, LIB_MATH_DEG_TO_RAD(aDegree));
}

//------------------------------------------------------------------------------
void Vector3Pod::rotateRad(const Vector3Pod& aAxis, const float aRadian)
{
    LIB_UNUSED(aAxis);
    LIB_UNUSED(aRadian);
    // 必要があれば実装すること
    {}
    SYS_ASSERT_NOT_REACHED();
}

//------------------------------------------------------------------------------
void Vector3Pod::rotateRadVec(const Vector3Pod& aAxis, const float aRadian)
{
    rotateRad(aAxis, aRadian);
}

//------------------------------------------------------------------------------
void Vector3Pod::rotateRad(const AxisKind::EnumType aAxisKind, const float aRadian)
{
    switch (aAxisKind) {
        case AxisKind::X:
            rotateRad(UnitX(), aRadian);
            break;
        case AxisKind::Y:
            rotateRad(UnitY(), aRadian);
            break;
        case AxisKind::Z:
            rotateRad(UnitZ(), aRadian);
            break;
        default:
            SYS_ASSERT_NOT_REACHED();
            break;
    }
}

//------------------------------------------------------------------------------
bool Vector3Pod::rotateBySpeedDeg(
    const Vector3Pod& aDest,
    const float aAngularSpeedDegree
    )
{
    return rotateBySpeedRad(aDest, LIB_MATH_DEG_TO_RAD(aAngularSpeedDegree));
}

//------------------------------------------------------------------------------
bool Vector3Pod::rotateBySpeedRad(
    const Vector3Pod& aDest,
    const float aAngularSpeedRadian
    )
{
    SYS_ASSERT(isUnit() && aDest.isUnit());
    SYS_ASSERT(0.0f <= aAngularSpeedRadian && aAngularSpeedRadian <= LIB_MATH_PI);

    // 目標地点にすでに到達していたら
    float agl = angle(aDest);
    if (agl <= aAngularSpeedRadian) {
        *this = aDest;
        return true;
    }

    // 回転軸を外積で求める
    Vector3Pod axis = cross(aDest);
    SYS_ASSERT(!axis.isZeroStrict());

    // 指定ベクトルで回転
    rotateRad(axis, aAngularSpeedRadian);
    unitAssign();
    return false;
}

//------------------------------------------------------------------------------
bool Vector3Pod::rotateBySpeedChangeDestDeg(
    const Vector3Pod& aDest,
    const float aAngularSpeedDegree,
    const Vector3Pod& aAxis
    )
{
    return rotateBySpeedChangeDestRad(
        aDest,
        LIB_MATH_DEG_TO_RAD(aAngularSpeedDegree),
        aAxis
        );
}

//------------------------------------------------------------------------------
bool Vector3Pod::rotateBySpeedChangeDestRad(
    const Vector3Pod& aDest,
    const float aAngularSpeedRadian,
    const Vector3Pod& aAxis
    )
{
    SYS_ASSERT(isUnit() && aDest.isUnit());
    SYS_ASSERT(0.0f <= aAngularSpeedRadian && aAngularSpeedRadian <= LIB_MATH_PI);

    // 目標地点にすでに到達していたら
    float agl = angle(aDest);
    if (agl <= aAngularSpeedRadian) {
        *this = aDest;
        return true;
    }

    // 回転軸を外積で求める
    Vector3Pod axis = cross(aDest);

    // thisベクトルとaDestが平行だと外積を使って回転軸が求められない。
    // その時はaDestとaAxisの外積を取ってそこを回転軸にする。
    if (axis.isZeroStrict()) {
        axis = aDest.unitCross(aAxis);
    }

    // 指定ベクトルで回転
    SYS_ASSERT(!axis.isZeroStrict());
    rotateRad(axis, aAngularSpeedRadian);
    unitAssign();
    return false;
}

//------------------------------------------------------------------------------
Vector3Pod Vector3Pod::slerp(const Vector3Pod& aDest, const float aRatio) const
{
    SYS_ASSERT(isUnit() && aDest.isUnit());

    float sclp, sclq;
    float cosom = this->dot(aDest);

    if ((1.0f + cosom) > LIB_MATH_EPSILON) {
        if ((1.0f - cosom) > LIB_MATH_EPSILON) {
            float omega = Math::ArcCosRad(cosom);
            float sinom = Math::SinRad(omega);
            sclp = Math::SinRad((1.0f - aRatio) * omega) / sinom;
            sclq = Math::SinRad(aRatio * omega) / sinom;
        } else {
            sclp = 1.0f - aRatio;
            sclq = aRatio;
        }
        return (*this * sclp + aDest * sclq).unit();
    } else {
        Vector3Pod d = Create(-y, x, 0);

        if (aRatio < 0.5f) {
            sclp = Math::SinRad((1.0f - 2.0f * aRatio) * (LIB_MATH_PI / 2.0f));
            sclq = Math::SinRad(2.0f * aRatio * (LIB_MATH_PI / 2.0f));

            return (*this * sclp + d * sclq).unit();
        } else {
            float ratio = aRatio - 0.5f;
            sclp = Math::SinRad((1.0f - 2.0f * ratio) * (LIB_MATH_PI / 2.0f));
            sclq = Math::SinRad(2.0f * ratio * (LIB_MATH_PI / 2.0f));

            return (d * sclp + aDest * sclq).unit();
        }
    }
}

//------------------------------------------------------------------------------
bool Vector3Pod::rotSign(const Vector3Pod& aDest, const Vector3Pod& aAxis) const
{
    Vector3Pod result = cross(aDest);
    return result.dot(aAxis) >= 0.0f;
}

//------------------------------------------------------------------------------
void Vector3Pod::projection(const Vector3Pod& aNormal)
{
    float dot = this->dot(aNormal);
    *this += aNormal * (-dot);
}

//------------------------------------------------------------------------------
void Vector3Pod::projectionKeepLength(const Vector3Pod& aNormal)
{
    float length = this->length();

    if (length > 0.0f) {
        projection(aNormal);
        setLength(length);
    }
}

//------------------------------------------------------------------------------
float Vector3Pod::slopeAngle() const
{
    SYS_ASSERT(isUnit());
    return (LIB_MATH_PI / 2.0f) - Math::ArcSinRad(y);
}

//------------------------------------------------------------------------------
void Vector3Pod::reflect(const Vector3Pod& aNormal)
{
    reflect(aNormal, 1.0f);
}

//------------------------------------------------------------------------------
void Vector3Pod::reflect(const Vector3Pod& aNormal, const float aRatio)
{
    float dot = this->dot(aNormal);

    if (::std::fabs(dot) > LIB_MATH_EPSILON) {
        *this += aNormal * ((-1 - aRatio) * dot);
    }
}

//------------------------------------------------------------------------------
Vector3Pod Vector3Pod::getReflected(const Vector3Pod& aNormal) const
{
    Vector3Pod result(*this);
    result.reflect(aNormal);
    return result;
}

//------------------------------------------------------------------------------
Vector3Pod Vector3Pod::getReflected(const Vector3Pod& aNormal, const float aRatio) const
{
    Vector3Pod result(*this);
    result.reflect(aNormal, aRatio);
    return result;
}

//------------------------------------------------------------------------------
void Vector3Pod::scale(const Vector3Pod& aScale)
{
    x *= aScale.x;
    y *= aScale.y;
    z *= aScale.z;
}

//------------------------------------------------------------------------------
Vector3Pod Vector3Pod::getScaled(const Vector3Pod& aScale) const
{
    Vector3Pod result(*this);
    result.scale(aScale);
    return result;
}

//------------------------------------------------------------------------------
const ::lib::ShortString Vector3Pod::toShortString() const
{
    return LIB_SPRINTF("(%f, %f, %f)", x, y, z);
}

} // namespace
// EOF
