﻿// 文字コード:UTF-8
/// @file
#pragma once

LIB_FORWARD_DECLARE_2(lib, class String);

//------------------------------------------------------------------------------
namespace lib {

/// 特別な事をしない限りはアロケーションが発生しないU16文字列
/// @note
/// ・「U16」への「Format」はコンパイラ対応が不安定なのであえて利用できなくしている
class U16String
{
public:
    /// 容量1, 長さ0の空文字を作成
    static U16String GetEmpty();
    // UTF8文字列から生成
    static U16String FromUtf8(const String& aStr);
    // UTF8文字列から生成、容量が足りない場合アサート
    static U16String FromUtf8(int aCap, const char* aStr);
    // UTF8文字列から生成、容量が足りない場合欠落
    static U16String FromUtf8AsPossible(int aCap, const char* aStr);
    // １文字の文字列を生成
    static U16String FromCharacter(char16_t aChar);

#if PLATFORM_IS_WINDOWS
    // SJIS文字列から生成
    static U16String FromSjis(const String& aStr);
    // SJIS文字列から生成、容量が足りない場合アサート
    static U16String FromSjis(int aCap, const char* aStr);
    // SJIS文字列から生成、容量が足りない場合欠落
    static U16String FromSjisAsPossible(int aCap, const char* aStr);
#endif

    // 環境に応じて(UTF8 or SJIS)文字列から生成
    static U16String FromLiteral(const String& aStr);
    // 環境に応じて(UTF8 or SJIS)文字列から生成 容量が足りない場合アサート
    static U16String FromLiteral(int aCap, const char* aStr);
    // 環境に応じて(UTF8 or SJIS)文字列から生成 容量が足りない場合欠落
    static U16String FromLiteralAsPossible(int aCap, const char* aStr);

    /// 整形済の文字列を生成
    static U16String FromFormat(int aCap, const char16_t* aStr, ...);
    static U16String FromFormatV(int aCap, const char16_t* aStr, ::std::va_list aVaList);
    static U16String FromFormatAsPossible(int aCap, const char16_t* aStr, ...);
    static U16String FromFormatVAsPossible(int aCap, const char16_t* aStr, ::std::va_list aVaList);

    // 容量ゼロの状態で生成
    U16String();
    // 容量を指定して生成
    explicit U16String(int aCap);
    // U16文字列から生成
    explicit U16String(const char16_t* aStr);
#if COMPILER_IS_CLANG // @memo VC環境ではchar16_tはtypedef unsigned shortと定義されているため　多重定義になってしまうのでClang限定としています。
    explicit U16String(const uint16_t* aStr) : U16String(reinterpret_cast<const char16_t*>(aStr)) {}
#endif
    explicit U16String(int aCap, const char16_t* aStr);
#if COMPILER_IS_CLANG // @memo VC環境ではchar16_tはtypedef unsigned shortと定義されているため　多重定義になってしまうのでClang限定としています。
    explicit U16String(int aCap, const uint16_t* aStr) : U16String(aCap, reinterpret_cast<const char16_t*>(aStr)) {}
#endif
    // ASCII文字列から生成
    explicit U16String(int aCap, const char* aStr);
    // 文字列から生成
    U16String(const U16String& aStr);
    // 破棄
    ~U16String();
    /// 有効か？
    bool isValid() const { return(mCap > 0); }
    // 容量
    int capacity() const { return mCap; }
    // C文字列として取得
    const char16_t* buffer() const { SYS_ASSERT(isValid()); return mBuf; }
    const uint16_t* uint16Buffer() const { SYS_ASSERT(isValid()); return reinterpret_cast<const uint16_t*>(mBuf); }
    /// 文字の取得
    char16_t& at(int i) { SYS_ASSERT(0 <= i && i < mCap); SYS_ASSERT_POINTER(mBuf); return(mBuf[i]); }
    /// 文字の取得
    char16_t at(int i) const { SYS_ASSERT(0 <= i && i < mCap); SYS_ASSERT_POINTER(mBuf); return(mBuf[i]); }
    // 空文字列化する、メモリは開放しない
    void clear();
    // メモリを開放
    void reset();
    // 容量の再設定、メモリも再確保される
    void reset(int aCap);
    // 再設定、容量が再計算されメモリも再確保される
    void reset(const char16_t* aStr);
#if COMPILER_IS_CLANG // @memo VC環境ではchar16_tはtypedef unsigned shortと定義されているため　多重定義になってしまうのでClang限定としています。
    void reset(const uint16_t* aStr) { reset(reinterpret_cast<const char16_t*>(aStr)); }
#endif
    void reset(const U16String& aStr);
    // 再設定、容量が再計算されメモリも再確保される
    void reset(int aCap, const char16_t* aStr);
    // 再設定、容量が再計算されメモリも再確保される
    void reset(int aCap, const char* aStr);
    // 設定、容量が足りない場合アサート
    void set(const U16String& aStr);
    // 設定、容量が足りない場合アサート
    U16String& operator=(const U16String& aStr);
    // 設定、容量が足りない場合欠落
    void setAsPossible(const U16String& aStr);
    // U16文字列から設定、容量が足りない場合アサート
    void set(const char16_t* aStr);
#if COMPILER_IS_CLANG // @memo VC環境ではchar16_tはtypedef unsigned shortと定義されているため　多重定義になってしまうのでClang限定としています。
    void set(const uint16_t* aStr) { set(reinterpret_cast<const char16_t*>(aStr)); }
#endif
    // U16文字列から設定、容量が足りない場合欠落
    void setAsPossible(const char16_t* aStr);
#if COMPILER_IS_CLANG // @memo VC環境ではchar16_tはtypedef unsigned shortと定義されているため　多重定義になってしまうのでClang限定としています。
    void setAsPossible(const uint16_t* aStr) { setAsPossible(reinterpret_cast<const char16_t*>(aStr)); }
#endif
    // U16文字列から設定、容量が足りない場合アサート
    U16String& operator=(const char16_t* aStr);
    // ヌル終端を持たないU16文字列から設定、容量が足りない場合アサート
    void set(const char16_t* aStr, int aLen);
#if COMPILER_IS_CLANG // @memo VC環境ではchar16_tはtypedef unsigned shortと定義されているため　多重定義になってしまうのでClang限定としています。
    void set(const uint16_t* aStr, int aLen) { set(reinterpret_cast<const char16_t*>(aStr), aLen); }
#endif
    // ヌル終端を持たないU16文字列から設定、容量が足りない場合欠落
    void setAsPossible(const char16_t* aStr, int aLen);
#if COMPILER_IS_CLANG // @memo VC環境ではchar16_tはtypedef unsigned shortと定義されているため　多重定義になってしまうのでClang限定としています。
    void setAsPossible(const uint16_t* aStr, int aLen) { setAsPossible(reinterpret_cast<const char16_t*>(aStr), aLen); }
#endif
    // ASCII文字列から設定、容量が足りない場合アサート
    void set(const char* aStr);
    // UTF8文字列から設定する
    void setFromUtf8(const String& aStr);
    // UTF8文字列から設定、容量が足りない場合アサート
    void setFromUtf8(const char* aStr);
    // UTF8文字列から設定、容量が足りない場合欠落
    void setFromUtf8AsPossible(const String& aStr);
    // UTF8文字列から設定、容量が足りない場合欠落
    void setFromUtf8AsPossible(const char* aStr);

#if PLATFORM_IS_WINDOWS
    // SJIS文字列から設定、容量が足りない場合欠落
    void setFromSjisAsPossible(const char* aStr);
#endif
    // 環境に応じて(UTF8 or SJIS)文字列から設定、容量が足りない場合欠落
    void setFromLiteralAsPossible(const char* aStr);
    // 文字列の置き換え、挿入できた長さを返す
    int replace(int index, int removeLen, const char16_t* insertStr, int insertLen = -1);
    /// 文字の置き換え。removeWord -> insertWord
    void replaceWord(char16_t removeWord, char16_t insertWord);
    // 文字列の削除、lenが文字列範囲を超えてもアサートしない（限界まで削除）
    void remove(int index, int aLen);
    // 文字列の挿入、容量が足りない場合は限界まで挿入、挿入できた長さを返す
    int insert(int index, const char16_t* aStr, int aLen = -1);
    /// 部分文字列の取得
    /// 部分文字列の取得、末尾まで
    U16String getSubstring(int aStartIdx) const { return getSubstringByLength(aStartIdx, getLength() - aStartIdx); }
    /// 部分文字列の取得、長さ指定
    U16String getSubstringByLength(int aStartIdx, int aLen) const;
    /// 部分文字列の取得、指定
    U16String getSubstringByIndex(int aStartIdx, int aEndIdx) const { return getSubstringByLength(aStartIdx, aEndIdx - aStartIdx); }
    // 文字列長の取得
    int getLength() const;
    // 空チェック(O(1) で動作する(getLength() == 0 より高速))
    bool isEmpty() const;
    // ASCII文字列か？(ASCII文字以外が1文字でも含まれていたらfalse)
    bool isASCII() const;
    /// UTF16として正しくなるように補正する(サロゲートペアの補正)
    void correctAsUtf16IfPossible();
    /// 先頭と末尾の空白文字を削除
    void strip();
    // 文字の連結
    U16String& operator+=(const U16String& aStr);
    U16String operator+(const U16String& aStr) const;
    /// 同一か？
    bool operator==(const char16_t* aStr) const;
    bool operator==(const U16String& aStr) const;
    /// 同一か？
    bool equals(const char16_t* aStr, bool aIgnoresCase = false) const;
    bool startsWith(const char16_t* aStr, bool aWillIgnoresCase) const;
    /// 指定文字を含むか？
    bool contains(char16_t aCh) const;
    /// 整形(欠落不許可)
    void format(const char16_t* aFmt, ...);
    void formatV(const char16_t* aFmt, ::std::va_list aVaList);
    void formatAsPossible(const char16_t* aFmt, ...);
    void formatVAsPossible(const char16_t* aFmt, ::std::va_list aVaList);
    // 文字情報をdump
    DEBUG_EVAL(void dump(const char* aMsg = nullptr) const);
private:
    char16_t* mBuf;
    int mCap;
};
} // namespace
// EOF
